package Transactions;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.postgresql.util.PSQLException;

import Exceptions.EmptyResultException;
import Exceptions.EmptyResultListException;
import Exceptions.IdNotFoundException;
import Exceptions.MoreRowsReturnedException;
import Exceptions.WrongInputException;
import main.DBContext;
import rdg.Assigned;
import rdg.Event;
import rdg.EventFinder;
import rdg.Sector;
import rdg.Tariff;

public class EventManager {
	
	/**
	 * Returns sales of a specified Event
	 * Doesn't include sales from permanent Tickets
	 * Is transaction with isolation level: READ COMMITTED:
	 * 		Should not return sales with tickets with uncommitted changes - avoid dirty read
	 * 		If Non-repeatable read occurs: if tickets fits - count it, else ignore it.
	 * @param eventId wanted Event's id
	 * @return sales of a Event with specified eventId
	 * @throws SQLException when sql throws its exception
	 * @throws IdNotFoundException when id is null
	 * @throws MoreRowsReturnedException one of the classes finder throws its exception
	 */
	public static Double getSalesByID(int eventId) throws SQLException, IdNotFoundException, MoreRowsReturnedException {
		
		Connection c = DBContext.getConnection();
		c.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);
		c.setAutoCommit(false);
		
		Event event = EventFinder.getInstance().findById(eventId);
		if(event == null) {
			throw new IdNotFoundException("Event with this id not found");
		}

		Double sales = null;
		
		try {
			sales = event.getSales();
		} catch (PSQLException | MoreRowsReturnedException | EmptyResultException e) {
			e.printStackTrace();
		} finally {
			c.commit();
			c.setAutoCommit(true);
		}
		
		return sales;
	}
	
	/**
	 * Insert an Event into database with the type of 'zakladna cast' or 'play-off'
	 * Inserts Assigneds into database for this Event with chosen Sectors and Tariffs
	 * Creates all available Tickets with null category
	 * Uses pre-defined Tariffs
	 * Is transaction with isolation level: READ COMMITTED:
	 * 		Safely inserts Event, Assigned, Tickets
	 * @param event new Event with values set in the menu by user
	 * @param chosenSectors list of Sectors chosen by user to be assigned with event
	 * @return generated id of inserted Event
	 * @throws SQLException when sql throws its exception
	 * @throws EmptyResultListException when one of the classes' methods throws this exception
	 */
	public static int addGame(Event event, List<Sector> chosenSectors) throws SQLException, EmptyResultListException {
		
		Connection c = DBContext.getConnection();
		c.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);
		c.setAutoCommit(false);
			
		//ziskaj spravne tariffy na hry
		int[] tariffs;
		if(event.getType().equals("zakladna cast")) tariffs = Tariff.zakladnaCast; //tariffy zc
		else tariffs = Tariff.playOff; //tariffy p-o
		
		try {
			//pridaj hru
			event.insert();
			
			//novy assigned
			Assigned a = new Assigned();
			a.setEventId(event.getId());
			
			//pre kazdy sektor:
			for(Sector s : chosenSectors) {
				
				//vytvor listky pre tento sektor
				TicketManager.createTickets(event, s);
				
				//prirad(assign) tarifu pre tento sektor
				for(int t : tariffs) {
					a.setSectorId(s.getId());
					a.setTariffId(t);
					//vytvor assigned
					a.insert();
				}
			}
			
			//z vytvorenych zarezervuj permanentky
			TicketManager.updatePermanents(event);
			
		} catch(PSQLException e ) {
			e.printStackTrace();
		} finally {
			c.commit();
			c.setAutoCommit(true);
		}
		
		return event.getId();
	}
	
	/**
	 * Insert an Event into database with the type of 'podujatie'
	 * Inserts Assigneds into database for this Event with chosen Sectors and Tariffs
	 * Uses Tariffs created by user
	 * Is transaction with isolation levet: READ COMMITTED:
	 * 		Safely inserts Event, Assigneds, Tickets, Tariffs
	 * @param event new Event with values set in the menu by user
	 * @param sectTarMap map of chosen Sectors with respective tariffs created by user
	 * @return generated id of inserted Event
	 * @throws SQLException when sql throws its exception
	 */
	public static int addEvent(Event event, Map<Sector, List<Tariff>> sectTarMap) throws SQLException {
		
		Connection c = DBContext.getConnection();
		c.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);
		c.setAutoCommit(false);
		
//		if(event.getId() == 0) {
//			System.out.println("Event id: 0");
//			return 0;
//		}
		
		try {
			//pridaj event
			event.insert();
			
			for(Sector sector : sectTarMap.keySet()) {
				
				//vytvor listky pre tento sektor
				TicketManager.createTickets(event, sector);
				
				for(Tariff tariff : sectTarMap.get(sector)) {
					//vytvor tarifu
					tariff.setId(tariff.insert());
					//vytvor assigned
					Assigned assigned = new Assigned();
					assigned.setEventId(event.getId());
					assigned.setSectorId(sector.getId());
					assigned.setTariffId(tariff.getId());
					assigned.insert();
				}
			}
			
		} catch (PSQLException e) {
			e.printStackTrace();
		} finally {
			c.commit();
			c.setAutoCommit(true);
		}

		return event.getId();
	}
	
	/**
	 * Returns list of Events' ids whose type is 'zakladna cast'
	 * @return list of ids of Events where type = 'zakladna cast'
	 * @throws SQLException when sql throws its exception
	 * @throws EmptyResultListException when no ids are returned
	 */
	public static List<Integer> getZakladnaCastGameIds() throws SQLException, EmptyResultListException {
		List<Integer> ids = new ArrayList<>();
		String sql = "SELECT id FROM events "
				+ "WHERE type = 'zakladna cast'";
		try(Statement s = DBContext.getConnection().createStatement()) {
			try(ResultSet rs = s.executeQuery(sql)){
				while(rs.next()) {
					ids.add(rs.getInt(1));
				}
			}
		}
		if(ids.isEmpty()) throw new EmptyResultListException("No events of type 'zakladna cast' returned");
		
		return ids;
	}
	
	/**
	 * Returns list of Events' ids whose type is 'play-off' 
	 * @return list of ids of Events where type = 'play-off'
	 * @throws SQLException when sql throws its exception
	 * @throws EmptyResultListException when no ids are returned for year
	 */
	public static List<Integer> getPlayOffGameIds() throws SQLException, EmptyResultListException {
		List<Integer> ids = new ArrayList<>();
		String sql = "SELECT id FROM events "
				+ "WHERE type = 'play-off'";
		try(Statement s = DBContext.getConnection().createStatement()) {
			try(ResultSet rs = s.executeQuery(sql)){
				while(rs.next()) {
					ids.add(rs.getInt(1));
				}
			}
		}
		if(ids.isEmpty()) throw new EmptyResultListException("No events of type 'play-off' returned");
		
		return ids;
	}
	
	/**
	 * Calls getAvgMonthlyVisits(Integer year) function within a Transaction
	 * Is transaction with isolation levet: READ COMMITTED:
	 * 		Only count in committed events - avoid dirty read
	 * 		If Non-repeatable read occurs:if events fits year - count it, else ignore it
	 * @param year wanted year
	 * @throws WrongInputException when year is null
	 * @throws SQLException when sql throws its exception
	 * @throws EmptyResultException when no events happen in specified year
	 */
	public static void getAvgVisitsTransaction(Integer year) throws WrongInputException, SQLException, EmptyResultException {
		if(year == null) throw new WrongInputException("Year in null");
		
		Connection c = DBContext.getConnection();
		c.setAutoCommit(false);
		c.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);
		
		try {
			getAvgMonthlyVisits(year);
		} catch(PSQLException e) {
			e.printStackTrace();
		} finally {
			c.commit();
			c.setAutoCommit(true);
		}
		
		
	}
	
	/**
	 * Prints average monthly visits to Events whose type is not 'podujatie' which take place during specified year
	 * Shape: 'No. of month'. Month has average number of visits: 'result for this month no.'
	 * @param year wanted year
	 * @throws SQLException when sql throws its exception
	 * @throws EmptyResultException when no events happen in specified year
	 */
	public static void getAvgMonthlyVisits(Integer year) throws SQLException, EmptyResultException {
		String sql = "SELECT sub.month, AVG(sub.counts) "
				+ "FROM (SELECT e.id, EXTRACT(MONTH FROM e.date) as month, count(*) as counts FROM events e "
				+ "			INNER JOIN tickets t ON (t.event_id = e.id)"
				+ "			WHERE EXTRACT(YEAR FROM e.date) = ? AND t.is_used = true AND e.type != 'podujatie'"
				+ "			GROUP BY e.id) as sub "
				+ "GROUP BY sub.month"
		;
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			ps.setInt(1, year);
			try(ResultSet rs = ps.executeQuery()) {
				System.out.println("Year " + year + ":");
				int pocet = 0;
				while(rs.next()) {
					pocet++;
					System.out.println(rs.getString(1) + ". Month has average number of visits:" + rs.getInt(2));
				}
				if(pocet == 0) throw new EmptyResultException("No events returned");
			}
		}
		
	}


}
