package Transactions;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.Date;
import java.sql.SQLException;
import java.util.List;
import org.postgresql.util.PSQLException;
import Exceptions.IdNotFoundException;
import Exceptions.InsufficientFundsException;
import Exceptions.MoreRowsReturnedException;
import Exceptions.ProcessCancelledException;
import Exceptions.WrongInputException;
import main.DBContext;
import rdg.Admin;
import rdg.Customer;
import rdg.Payment;
import rdg.Reservation;
import rdg.Ticket;
import rdg.TicketFinder;

public class PaymentsManager {
	
	/**
	 * Adds points to a Customer in a database based on how much he paid
	 * Returns amount of gained points
	 * Instance perPoints represents how much money needed to be paid to gain one point
	 * amount depends on perPoints
	 * @param to specified Customer to add points to
	 * @param amount how much money did Customer paid
	 * @return amount of given points
	 * @throws IdNotFoundException when Customer is nulls
	 * @throws WrongInputException when amount is null or less than 0
	 * @throws SQLException when sql throws its exception
	 */
	public static int givePoints(Customer to, Double amount) throws IdNotFoundException, WrongInputException, SQLException {
		//check for nulls
		if(to == null) throw new IdNotFoundException("Customer id not found");
		if(amount == null) throw new WrongInputException("Amount is null");
		if(amount < 0) throw new WrongInputException("Amount is less than 0");
		
		//get amount of gained points
		int perPoints = 5;
		int pointsGained = (int) Math.floor(amount / perPoints);
		
		//update
		to.setPoints(to.getPoints() + pointsGained);
		to.update();
		
		return pointsGained;
	}
	
	/**
	 * Used to transfer money from Customer from to Admin to for list of Reservations
	 * Transfer money for each Reservation individually using payOnlineForReservation(Customer from, Admin to, Reservation reservation)
	 * Is transaction with isolation level: REPEATABLE READ
	 * 		When Non-Repeatable read occurs: the transaction is cancelled, throws PocessCancelledException
	 * 		Serialization anomaly should not occur.
	 * @param from Customer who pays
	 * @param to Admin who receives payment
	 * @param reservations list of Reservations Customer choise to pay for
	 * @throws SQLException when sql throws its exception
	 * @throws IdNotFoundException when one of classes' ids is null
	 * @throws WrongInputException when reservations is null or empty list
	 * @throws InsufficientFundsException when Customer from doesn't have enough credit on his account
	 * @throws MoreRowsReturnedException when on of classes' methods throws this exception
	 * @throws ProcessCancelledException when transaction non-repeatable read occurs
	 */
	public static void payOnlineForReservations(Customer from, Admin to, List<Reservation> reservations) throws SQLException, IdNotFoundException, WrongInputException, InsufficientFundsException, MoreRowsReturnedException, ProcessCancelledException {
		
		//checks
		if(from == null) throw new IdNotFoundException("Customer id not found");
		if(to == null) throw new IdNotFoundException("Admin id not found");
		if(reservations == null) throw new WrongInputException("Reservations are null");
		if(reservations.isEmpty()) throw new WrongInputException("No reservations to pay");
		
		Connection c = DBContext.getConnection();
		c.setTransactionIsolation(Connection.TRANSACTION_REPEATABLE_READ);
		c.setAutoCommit(false);
		
		try {
			for(Reservation reservation : reservations) {
				payOnlineForReservation(from, to, reservation);
			}
			
		} catch(PSQLException e) {
			throw new ProcessCancelledException("Transaction exception thrown");
		} finally {
			c.commit();
			c.setAutoCommit(true);
		}
		
	}

	/**
	 * Used to transfer money from Customer from to Admin to for Reservation
	 * Inserts new Payments into database
	 * @param from Customers who pays
	 * @param to Admin who received payment
	 * @param reservation Reservation to pay for
	 * @throws IdNotFoundException when from or to ids is null
	 * @throws WrongInputException when amount is null or less than 0
	 * @throws InsufficientFundsException when Customer doesn't have enough credit on his account
	 * @throws SQLException when sql throws its exception
	 * @throws MoreRowsReturnedException when one of classes' methods throws this exception
	 */
	public static void payOnlineForReservation(Customer from, Admin to, Reservation reservation) throws IdNotFoundException, WrongInputException, InsufficientFundsException, SQLException, MoreRowsReturnedException {
		
		//get ticket for this reservation
		Ticket ticket = TicketFinder.getInstance().findById(reservation.getTicketId());

		//get amount from ticket's price
		Double amount = ticket.getPrice();
		
		//check for nulls
		if(from == null) throw new IdNotFoundException("Customer id not found");
		if(to == null) throw new IdNotFoundException("Box office id not found");
		if(amount == null || amount < 0) throw new WrongInputException("Amount is null");
		//check if sufficient amount in customer's wallet
		if(from.getWallet().doubleValue() < amount) throw new InsufficientFundsException("Not enough funds on customer's wallet.");

		//money transfer
		from.setWallet(from.getWallet().subtract(BigDecimal.valueOf(amount)));
		to.setWallet(to.getWallet().add(BigDecimal.valueOf(amount)));

		//save payemt
		Payment payment = new Payment();
		payment.setAmount(BigDecimal.valueOf(amount));
		payment.setPaymentDate(new Date(System.currentTimeMillis())); //today
		payment.setCustomerId(from.getId());
		payment.setAdminId(to.getId());
		payment.setTicketId(ticket.getId());

		//ticket change
		ticket.setSold(true);
		//reservation change
		reservation.setPaid(true);

		//update
		payment.insert();
		from.update();
		to.update();
		ticket.update();
		reservation.update();
			

	}
	
	/**
	 * Used to add money to Admin to for paid list of Tickets
	 * Uses payBOTicket(Admin to, Ticket ticket) to pay for each Ticket individually
	 * Is transaction with isolation level: REPEATABLE READ
	 * 		When Non-Repeatable read occurs: the transaction is cancelled, throws PocessCancelledException
	 * 		Serialization anomaly should not occur.
	 * @param to Admin who received payments
	 * @param tickets list of Tickets to pay for
	 * @throws IdNotFoundException when admin's id is null
	 * @throws WrongInputException when tickets is null or empty list
	 * @throws SQLException when sql throws its exception
	 * @throws MoreRowsReturnedException when one of classes' methods throw this exception
	 * @throws ProcessCancelledException when transaction non-repeatable read occurs
	 */
	public static void payBOForTickets(Admin to, List<Ticket> tickets) throws IdNotFoundException, WrongInputException, SQLException, MoreRowsReturnedException, ProcessCancelledException {
		//check
		if(to == null) throw new IdNotFoundException("Admin id not found");
		if(tickets == null) throw new WrongInputException("Tickets are null");
		if(tickets.isEmpty()) throw new WrongInputException("No tickets to pay for");
		
		Connection c = DBContext.getConnection();
		c.setAutoCommit(false);
		c.setTransactionIsolation(Connection.TRANSACTION_REPEATABLE_READ);
		
		try {
			
			for(Ticket ticket : tickets) {
				//check for permanents - can't buy them in Box Office
				if(ticket.getCategory().equals("permanent")) throw new WrongInputException("Cannot buy permanents in BO");
				payBOForTicket(to, ticket);
			}
			
		} catch (PSQLException e) {
			throw new ProcessCancelledException("Transaction expection thrown");
		} finally {
			c.commit();
			c.setAutoCommit(true);
		}
	}
	
	/**
	 * Used to add Payment for a Ticket paid in the Box Office
	 * Inserts new Payments
	 * @param to Admin who received payments
	 * @param ticket Ticket paid for
	 * @throws WrongInputException when Ticket is not free
	 * @throws IdNotFoundException when to's or ticket's id is null
	 * @throws SQLException when sql throws its exception
	 * @throws MoreRowsReturnedException when one of classes' methods throws this exception
	 */
	public static void payBOForTicket(Admin to, Ticket ticket) throws WrongInputException, IdNotFoundException, SQLException, MoreRowsReturnedException {
		//checks
		if(ticket == null) throw new IdNotFoundException("Ticket id not found");
		if(to == null) throw new IdNotFoundException("Box office id not found");
		Ticket ticketCheck = TicketFinder.getInstance().findById(ticket.getId());
		if(!ticketCheck.isFree()) throw new WrongInputException("Ticket is no longer free");
		
		//change ticket first
		ticket.setFree(false);
		ticket.setSold(true);
		Double amount = ticket.getPrice();
		
		//add money to admin
		to.setWallet(to.getWallet().add(BigDecimal.valueOf(amount)));
		
		//save payment
		Payment payment = new Payment();
		payment.setAmount(BigDecimal.valueOf(amount));
		payment.setPaymentDate(new Date(System.currentTimeMillis())); //today
		payment.setAdminId(to.getId());
		payment.setTicketId(ticket.getId());

		
		//update
		to.update();
		payment.insert();
		ticket.update();
	}

	
}
