package Transactions;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import Exceptions.EmptyResultListException;
import Exceptions.MoreRowsReturnedException;
import main.DBContext;
import rdg.Event;
import rdg.Place;
import rdg.PlaceFinder;
import rdg.Sector;

public class PlaceManager {
	
	/**
	 * Returns list of row numbers of specified Sector
	 * @param sectorId wanted Sector's Id
	 * @return list of row numbers in specified Sector
	 * @throws SQLException when sql throws its exception
	 * @throws EmptyResultListException when no row numbers are returned 
	 */
	public static List<Integer> getRows(int sectorId) throws SQLException, EmptyResultListException {
		List<Integer> rows = new ArrayList<Integer>();
		
		String sql = "SELECT DISTINCT(row_number) FROM PLACES "
				+ "WHERE sector_id = ? AND lounge_name IS NULL";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			ps.setInt(1, sectorId);
			try(ResultSet rs = ps.executeQuery()) {
				while(rs.next()) {
					rows.add(rs.getInt(1));
				}
			}
		}
		if(rows.isEmpty()) throw new EmptyResultListException("No rows returned");
		return rows;
	}
	
	/**
	 * Returns list of Lounges of specified Sector
	 * @param sector wanted Sector
	 * @return list of Lounges in specified Sector
	 * @throws SQLException when sql throws its exception
	 * @throws EmptyResultListException when no lounges are returned
	 */
	public static List<Place> getLounges(Sector sector) throws SQLException, EmptyResultListException {
		List<Place> lounges = new ArrayList<Place>();
		
		String sql = "SELECT id, lounge_name FROM places "
				+ "WHERE sector_id = ? AND lounge_name IS NOT NULL";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			ps.setInt(1, sector.getId());
			try(ResultSet rs = ps.executeQuery()) {
				while(rs.next()) {
					Place lounge = new Place();
					lounge.setId(rs.getInt(1));
					lounge.setLoungeName(rs.getString(2));
					lounge.setSectorId(sector.getId());
					lounges.add(lounge);
				}
			}
		}
		if(lounges.isEmpty()) throw new EmptyResultListException("No lounges returned");
		return lounges;
	}
	
	/**
	 * Returns list of free lounges (type Place) of Event in Sector
	 * @param event wanted Event
	 * @param sector wanted Sector
	 * @return list of Places which are lounges and are in specified Sector whose Tickets for specified Event have isFree = true
	 * @throws SQLException when sql throws its exception
	 * @throws EmptyResultListException when no lounges are returned
	 */
	public static List<Place> getFreeLounges(Event event, Sector sector) throws SQLException, EmptyResultListException {
		List<Place> freeLounges = new ArrayList<Place>();
		String sql = "SELECT p.id, p.lounge_name FROM places p "
				+ "INNER JOIN tickets ti ON (ti.place_id = p.id) "
				+ "WHERE ti.event_id = ? AND p.sector_id = ? AND ti.is_free = true AND lounge_name IS NOT NULL";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			ps.setInt(1, event.getId());
			ps.setInt(2, sector.getId());
			try(ResultSet rs = ps.executeQuery()) {
				while(rs.next()) {
					Place lounge = new Place();
					lounge.setId(rs.getInt(1));
					lounge.setLoungeName(rs.getString(2));
					lounge.setSectorId(sector.getId());
					freeLounges.add(lounge);
				}
				
				if(freeLounges.isEmpty()) throw new EmptyResultListException("No free lounges returned");
			}
		}
		
		return freeLounges;
	}

	/**
	 * Returns list of Places for Sector in a row
	 * @param sectorId wanted Sector's id
	 * @param row wanted row's number
	 * @return list of Places in specified Sector in a specified row
	 * @throws SQLException when sql throws its exception
	 * @throws EmptyResultListException when no Places are returned
	 */
	public static List<Place> getSeats(int sectorId, int row) throws SQLException, EmptyResultListException {
		List<Place> seats = new ArrayList<Place>();
		String sql = "SELECT * FROM places "
				+ "WHERE sector_id = ? AND row_number = ?";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			ps.setInt(1, sectorId);
			ps.setInt(2, row);
			try(ResultSet rs = ps.executeQuery()) {
				while(rs.next()) {
					Place place = new Place();
					place.setId(rs.getInt(1));
					place.setSectorId(rs.getInt(2));
					place.setRowNumber(rs.getInt(3));
					place.setSeatNumber(rs.getInt(4));
					place.setLoungeName(rs.getString(5));
					seats.add(place);
				}
			}
		}
		
		if(seats.isEmpty()) throw new EmptyResultListException("No seats returned");
		return seats;
	}
	
	/**
	 * Returns map of number of free Places for Event in Sector
	 * @param eventId wanted Event's id
	 * @param sectorId wanted Sector's id
	 * @return map of free seats - row number with respective number of free seats
	 * @throws SQLException whens sql throws its exception
	 * @throws EmptyResultListException when no rows are returned
	 */
	public static Map<Integer, Integer> getFreeSeatsCount(int eventId, int sectorId) throws SQLException, EmptyResultListException {
		Map<Integer, Integer> counts = new HashMap<Integer, Integer>();
		String sql = "SELECT p.row_number, count(p.seat_number) FROM places p "
				+ "INNER JOIN tickets ti ON (ti.place_id = p.id) "
				+ "WHERE ti.event_id = ? AND ti.is_free = true AND p.sector_id = ? "
				+ "GROUP BY p.row_number";
		try (PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			ps.setInt(1, eventId);
			ps.setInt(2, sectorId);
			try(ResultSet rs = ps.executeQuery()) {
				while(rs.next()) {
					counts.put(rs.getInt(1), rs.getInt(2));
				}
			}
			if(counts.isEmpty()) throw new EmptyResultListException("No rows Returned");
			
			return counts;
		}
	}
	
	/**
	 * Returns list of free Places for Event in Sector with row number
	 * @param eventId wanted Event's id
	 * @param sectorId wanted sector's id
	 * @param row wanted seat's row number
	 * @return list of Places for specified Event in specified Sector with rowNumber = specified row
	 * @throws SQLException when sql throws its exception
	 * @throws EmptyResultListException when no Places are returned
	 * @throws MoreRowsReturnedException when one of classes' methods throws this exception
	 */
	public static List<Place> getFreeSeatsForEvent(int eventId, int sectorId, int row) throws SQLException, EmptyResultListException, MoreRowsReturnedException {
		List<Place> freeSeats = new ArrayList<Place>();
		String sql = "SELECT p.id FROM places p "
				+ "INNER JOIN tickets ti ON (ti.place_id = p.id) "
				+ "WHERE ti.event_id = ? AND ti.is_free = true AND p.sector_id = ? AND p.row_number = ?";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			ps.setInt(1, eventId);
			ps.setInt(2, sectorId);
			ps.setInt(3, row);
			try(ResultSet rs = ps.executeQuery()) {
				while(rs.next()) {
					Place place = PlaceFinder.getInstance().findById(rs.getInt(1));
					freeSeats.add(place);
				}
			}
		}
		
		if(freeSeats.isEmpty()) throw new EmptyResultListException("No free seats returned");
		
		return freeSeats;
	}
	
	/**
	 * Returns String containing seat number from list of Places
	 * @param places list of Places which seat numbers should be contained in returned String
	 * @return String containing seat numbers of specified list with commas in between
	 */
	public static String getSeatNumbersString(List<Place> places) {
		Collections.sort(places, Comparator.comparingInt(Place::getSeatNumber));
		String string = "";
		for(Place place : places) {
			string += place.getSeatNumber() + ",";
		}
		string = string.substring(0, string.length() - 1);
		return string;
	}
	
	/**
	 * Returns String containing lounge names from list of Places
	 * @param lounges list of Places which loungeNames should be contained in returned String
	 * @return String containing lounge names of specified list with commas in between 
	 */
	public static String getLoungeString(List<Place> lounges) {
		String string = "";
		for(Place lounge : lounges) {
			string += lounge.getLoungeName() + ",";
		}
		string = string.substring(0, string.length() - 1);
		return string;
	}
	
	/**
	 * Returns list of Places whose names are in list of String names
	 * @param from list of Places to choose from
	 * @param names of Places to choose from specified list of Places
	 * @return list of Places from specified list whose names are in specified list of names
	 */
	public static List<Place> getChosenPlaces(List<Place> from, List<String> names) {
		List<Place> chosen = new ArrayList<Place>();
		for(Place p : from) {
			if(p.getLoungeName() != null) {
				if(names.contains(p.getLoungeName())) chosen.add(p);
			} else {
				if(names.contains(String.valueOf(p.getSeatNumber()))) chosen.add(p);
			}
		}
		return chosen;
	}

	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
}
