package Transactions;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import Exceptions.EmptyResultListException;
import Exceptions.IdNotFoundException;
import Exceptions.MoreRowsReturnedException;
import main.DBContext;
import rdg.Customer;
import rdg.Reservation;
import rdg.Ticket;
import rdg.TicketFinder;

public class ReservationManager {

	/**
	 * Returns list of unconfirmed Reservations of a Customer
	 * @param customer whose Reservations we want to get
	 * @return list of Reservations whose customerId is specified Customer's id and isConfirmed is false
	 * @throws SQLException when sql throws its exception
	 * @throws EmptyResultListException when no reservations are returned
	 */
	public static List<Reservation> getUnconfirmedReservations(Customer customer) throws SQLException, EmptyResultListException {
		List<Reservation> reservations = new ArrayList<Reservation>();
		String sql = "SELECT * FROM reservations WHERE customer_id = ? AND is_confirmed = false";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			ps.setInt(1, customer.getId());
			try(ResultSet rs = ps.executeQuery()) {
				while(rs.next()) {
					Reservation reservation = new Reservation();
					reservation.setId(rs.getInt(1));
					reservation.setValidFrom(rs.getTimestamp(2).toLocalDateTime());
					reservation.setValidTill(rs.getTimestamp(3).toLocalDateTime());
					reservation.setConfirmed(rs.getBoolean(4));
					reservation.setPaid(rs.getBoolean(5));
					reservation.setCustomerId(rs.getInt(6));
					reservation.setTicketId(rs.getInt(7));
					reservations.add(reservation);
				}
				
				if(reservations.isEmpty()) throw new EmptyResultListException("No reservations found");
			}
		}
		return reservations;
	}
	
	/**
	 * Returns list of unpaid Reservations of a Customer
	 * @param customer whose Reservations we want to get
	 * @return list of Reservations whose customerId is specified Customer's id and isConfirmed is true and isPaid is false
	 * @throws SQLException when sql throws its exception
	 * @throws EmptyResultListException when no reservations are returned
	 */
	public static List<Reservation> getUnpaidReservations(Customer customer) throws SQLException, EmptyResultListException {
		List<Reservation> reservations = new ArrayList<Reservation>();
		String sql = "SELECT * FROM reservations WHERE customer_id = ? AND is_confirmed = true AND is_paid = false";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			ps.setInt(1, customer.getId());
			try(ResultSet rs = ps.executeQuery()) {
				while(rs.next()) {
					Reservation reservation = new Reservation();
					reservation.setId(rs.getInt(1));
					reservation.setValidFrom(rs.getTimestamp(2).toLocalDateTime());
					reservation.setValidTill(rs.getTimestamp(3).toLocalDateTime());
					reservation.setConfirmed(rs.getBoolean(4));
					reservation.setPaid(rs.getBoolean(5));
					reservation.setCustomerId(rs.getInt(6));
					reservation.setTicketId(rs.getInt(7));
					reservations.add(reservation);
				}
				
				if(reservations.isEmpty()) throw new EmptyResultListException("No reservations found");
			}
		}
		return reservations;
	}
	
	/**
	 * Returns list of Reservations whose id is in list of Ids
	 * @param from list of Reservations to choose from
	 * @param stringIds list of String values of Ids from Reservations to choose
	 * @return list of Reservations whose String value of Id is in stringIds
	 * @throws EmptyResultListException when no Reservation string values of ids are in stringIds
	 */
	public static List<Reservation> getChosenReservations(List<Reservation> from, List<String> stringIds) throws EmptyResultListException {
		List<Reservation> chosenReservations = new ArrayList<Reservation>();
		for(Reservation reservation : from) {
			if(stringIds.contains(String.valueOf(reservation.getId()))) chosenReservations.add(reservation);
		}
		if(chosenReservations.isEmpty()) throw new EmptyResultListException("No customer's reservations chosen");
		return chosenReservations;
	}
	
	/**
	 * Returns price of all Tickets of Reservations from list of Reservations together
	 * @param reservations list of wanted Reservations 
	 * @return sum of all Tickets' prices of Reservations from reservations
	 * @throws SQLException when sql throws its exception
	 * @throws IdNotFoundException when one of classes' methods returns this exception
	 * @throws MoreRowsReturnedException when one of classes' methods returns this exception
	 */
	public static Double getPriceForReservations(List<Reservation> reservations) throws SQLException, IdNotFoundException, MoreRowsReturnedException {
		Double price = 0.0;
		for(Reservation reservation : reservations) {
			Ticket ticket = TicketFinder.getInstance().findById(reservation.getTicketId());
			price += ticket.getPrice();
		}
		
		return price;
	}
	
}
