package Transactions;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import Exceptions.EmptyResultException;
import Exceptions.EmptyResultListException;
import Exceptions.IdNotFoundException;
import Exceptions.MoreRowsReturnedException;
import main.DBContext;
import rdg.Event;
import rdg.EventFinder;
import rdg.Sector;
import rdg.SectorFinder;

public class SectorManager {

	/**
	 * Returns list of Sectors which are available for an Event
	 * @return list of Sectors where isForEvent is true
	 * @throws SQLException when sql throws its exception
	 * @throws EmptyResultListException when no Sectors are returned
	 */
	public static List<Sector> getForEvent() throws SQLException, EmptyResultListException {
		List<Sector> sectors = new ArrayList<Sector>();
		String sql = "SELECT * FROM sectors WHERE is_for_event = true";
		
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			try(ResultSet rs = ps.executeQuery()){
				while(rs.next()) {
					Sector sector = new Sector();
					sector.setId(rs.getInt(1));
					sector.setName(rs.getString(2));
					sector.setForGame(rs.getBoolean(3));
					sector.setForEvent(rs.getBoolean(4));
					
					sectors.add(sector);
				}
				
				if(sectors.isEmpty()) throw new EmptyResultListException("No sectors returned"); 
				return sectors;
			}
		}
	}
	
	/**
	 * Returns list of Sectors which are available for an Event
	 * @return list of Sectors where isForEvent is true
	 * @throws SQLException when sql throws its exception
	 * @throws EmptyResultListException when no Sectors are returned
	 */
	public static List<Sector> getForGame() throws SQLException, EmptyResultListException {
		List<Sector> sectors = new ArrayList<Sector>();
		String sql = "SELECT * FROM sectors WHERE is_for_game = true";
		
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			try(ResultSet rs = ps.executeQuery()){
				while(rs.next()) {
					Sector sector = new Sector();
					sector.setId(rs.getInt(1));
					sector.setName(rs.getString(2));
					sector.setForGame(rs.getBoolean(3));
					sector.setForEvent(rs.getBoolean(4));
					
					sectors.add(sector);
				}
				
				if(sectors.isEmpty()) throw new EmptyResultListException("No sectors returned");
				return sectors;
			}
		}
	}
	
	/**
	 * Returns String containing names of Sectors from list of Sectors
	 * @param sectList list of wanted Sectors
	 * @return string containing names of Sectors from specified list with commas in between
	 */
	public static String listString(List<Sector> sectList) {
		String string = "";
		for(Sector s : sectList) string += s.getName()+",";
		string = string.substring(0, string.length() - 1); //posledna ciara prec
		return string;
	}
	
	/**
	 * Returns chosen Sector from list of Sectors based on name
	 * @param from list of Sectors to choose from
	 * @param name of wanted Sector
	 * @return Sector from specified list whose name is equals to specified name
	 * @throws EmptyResultException
	 */
	public static Sector getChosenSector(List<Sector> from, String name) throws EmptyResultException {
		Sector chosenSector = null;
		for(Sector s : from) {
			if(name.equals(s.getName())) {
				chosenSector = s;
				break;
			}
		}
		if(chosenSector == null) throw new EmptyResultException("Sector with this name not found");
		
		return chosenSector;
	}
	
	/**
	 * Returns list of Sectors from list of Sectors based on list of chosen Names
	 * @param from list of Sectors to choose from
	 * @param names list of names to choose from specified list from
	 * @return list of Sectors whose names are contained in names
	 */
	public static List<Sector> getChosenSectors(List<Sector> from, List<String> names) {
		List<Sector> chosenSectors = new ArrayList<Sector>();
		for(Sector s : from) {
			if(names.contains(s.getName())) chosenSectors.add(s);
		}
		return chosenSectors;
	}
	
	/**
	 * Returns list of Sectors assigned to an Event
	 * @param eventId wanted Event's Id
	 * @return list of Sectors where exists Assigned with Event's id and Sector's id
	 * @throws SQLException when sql throws its exception
	 * @throws IdNotFoundException when Event is not found
	 * @throws EmptyResultListException when no Sectors are returned
	 * @throws MoreRowsReturnedException when one of classes' methods throws this exception
	 */
	public static List<Sector> getSectors(int eventId) throws SQLException, IdNotFoundException, EmptyResultListException, MoreRowsReturnedException {
		List<Sector> sectors = new ArrayList<Sector>();
		Event event = EventFinder.getInstance().findById(eventId);
		if(event == null) throw new IdNotFoundException("Event with this id not found");
		
		String sql = "SELECT distinct(sector_id) FROM assigned WHERE event_id = ?";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			ps.setInt(1, event.getId());
			try(ResultSet rs = ps.executeQuery()) {
				while(rs.next()) {
					Sector sector = SectorFinder.getInstance().findById(rs.getInt(1));
					sectors.add(sector);
				}
			}
		}
		
		if(sectors.isEmpty()) throw new EmptyResultListException("No sectors found");
		
		return sectors;
	}
	
}
