package Transactions;

import java.sql.Array;
import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;

import org.postgresql.util.PSQLException;

import Exceptions.EmptyResultException;
import Exceptions.EmptyResultListException;
import Exceptions.IdNotFoundException;
import Exceptions.MoreRowsReturnedException;
import Exceptions.ProcessCancelledException;
import Exceptions.WrongInputException;
import main.DBContext;
import rdg.Customer;
import rdg.Event;
import rdg.EventFinder;
import rdg.Place;
import rdg.Reservation;
import rdg.Sector;
import rdg.Ticket;
import rdg.TicketFinder;

public class TicketManager {
	
	public TicketManager() {}
	
	/**
	 * Returns list of all reserved Tickets 
	 * Returns part of them - with sql query LIMIT and OFFSET
	 * @param limit for sql query
	 * @param offset for sql query
	 * @return list of Tickets where exists Reservation where ticketId is Ticket's id
	 * @throws SQLException when sql throws its exception
	 */
	public static List<Ticket> getReserved(int limit, int offset) throws SQLException {
		List<Ticket> reserved = new ArrayList<>();
		String sql = "SELECT t.id, is_free, is_sold, is_used, category, event_id, place_id "
				+ "from tickets t "
				+ "INNER JOIN reservations r ON (r.ticket_id = t.id)"
				+ "LIMIT ? OFFSET ?";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)){
			ps.setInt(1, limit);
			ps.setInt(2, offset);
			try(ResultSet rs = ps.executeQuery()){
				while(rs.next()) {
					Ticket ticket = new Ticket();
					ticket.setId(rs.getInt(1));
					ticket.setFree(rs.getBoolean(2));
					ticket.setSold(rs.getBoolean(3));
					ticket.setUsed(rs.getBoolean(4));
					ticket.setCategory(rs.getString(5));
					ticket.setEventID(rs.getInt(6));
					ticket.setPlaceID(rs.getInt(7));
					
					reserved.add(ticket);
				}
				if(!reserved.isEmpty()) return reserved;
			}
		}
		return null;
	}
	
	/**
	 * Returns list of all sold and not used Tickets 
	 * Returns part of them - with sql query LIMIT and OFFSET
	 * @param limit for sql query
	 * @param offset for sql query
	 * @return list of Tickets where exists isSold is true and isUsed is false
	 * @throws SQLException when sql throws its exception
	 */
	public static List<Ticket> getSoldNotUsed(int limit, int offset) throws SQLException {
		List<Ticket> soldNotUsed = new ArrayList<Ticket>();
		String sql = "SELECT * FROM tickets WHERE is_sold = true and is_used = false "
				+ "LIMIT ? OFFSET ?";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			ps.setInt(1, limit);
			ps.setInt(2, offset);
			try(ResultSet rs = ps.executeQuery()){
				while(rs.next()) {
					Ticket ticket = new Ticket();
					ticket.setId(rs.getInt(1));
					ticket.setFree(rs.getBoolean(2));
					ticket.setSold(rs.getBoolean(3));
					ticket.setUsed(rs.getBoolean(4));
					ticket.setCategory(rs.getString(5));
					ticket.setEventID(rs.getInt(6));
					ticket.setPlaceID(rs.getInt(7));
					
					soldNotUsed.add(ticket);
				}
				
				if(!soldNotUsed.isEmpty()) return soldNotUsed;
			}
		}
		return null;
	}
	
	/**
	 * Returns list of all sold and used Tickets 
	 * Returns part of them - with sql query LIMIT and OFFSET
	 * @param limit for sql query
	 * @param offset for sql query
	 * @return list of Tickets where exists isSold is true and isUsed is true
	 * @throws SQLException when sql throws its exception
	 */
	public static List<Ticket> getSoldAndUsed(int limit, int offset) throws SQLException {
		List<Ticket> soldAndUsed = new ArrayList<Ticket>();
		String sql = "SELECT * FROM tickets WHERE is_sold = true and is_used = true "
				+ "LIMIT ? OFFSET ?";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			ps.setInt(1, limit);
			ps.setInt(2, offset);
			try(ResultSet rs = ps.executeQuery()){
				while(rs.next()) {
					Ticket ticket = new Ticket();
					ticket.setId(rs.getInt(1));
					ticket.setFree(rs.getBoolean(2));
					ticket.setSold(rs.getBoolean(3));
					ticket.setUsed(rs.getBoolean(4));
					ticket.setCategory(rs.getString(5));
					ticket.setEventID(rs.getInt(6));
					ticket.setPlaceID(rs.getInt(7));
					
					soldAndUsed.add(ticket);
				}
				
				if(!soldAndUsed.isEmpty()) return soldAndUsed;
			}
		}
		return null;
	}
	
	/**
	 * Sets Ticket's isUsed true
	 * @param id of wanted Ticket
	 * @throws SQLException when sql throws its exception
	 * @throws IdNotFoundException when id is null
	 * @throws EmptyResultException when Ticket with id is not found
	 * @throws WrongInputException when Ticket's isUsed is true or isFree is true or isSold = false
	 * @throws MoreRowsReturnedException when ne of classes' methods throws this exception
	 */
	public static void setUsedByID(int id) throws SQLException, IdNotFoundException, EmptyResultException, WrongInputException, MoreRowsReturnedException {
		
		Ticket ticket = TicketFinder.getInstance().findById(id);
		if(ticket == null) {
			throw new IdNotFoundException("Ticket Id not found");
		}
		
		if(ticket.isUsed()) {
			throw new WrongInputException("Ticket is already used!");
		} else if(ticket.isFree()) {
			throw new WrongInputException("Ticket is still free!");
		} else if(!ticket.isSold()) {
			throw new WrongInputException("Ticket is not sold!");
		} 
		else {
			Event event = EventFinder.getInstance().findById(ticket.getEventID());
			if(event == null) throw new EmptyResultException("Event of this ticket not found");
			//dnes?
			if(!event.getDate().equals(Date.valueOf(LocalDate.now()))) {
				throw new WrongInputException("Event is not today!");
			}
			ticket.setUsed(true);
			ticket.update();
			System.out.println("Ticket updated as used!");
		}
	}
	
	/**
	 * Used to insert new Tickets into database - especially when new Event is created
	 * Returns the amount of newly created Tickets
	 * @param event for which event to create Tickets
	 * @param sector for which Sector Places to create Tickets
	 * @return the amount of newly created Tickets
	 * @throws SQLException when sql throws its exception
	 */
	public static int createTickets(Event event, Sector sector) throws SQLException {
		int pocet = -1;
		String sql = "INSERT INTO tickets(is_free, is_sold, is_used, event_id, place_id) "
				+ "SELECT true, false, false, ?, p.id "
				+ "FROM places p "
				+ "INNER JOIN sectors s ON (s.id = p.sector_id) "
				+ "WHERE s.id = ?";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)){
			ps.setInt(1, event.getId());
			ps.setInt(2, sector.getId());

			pocet = ps.executeUpdate();
			
			System.out.println("Successfully added " + pocet + " tickets for sector " + sector.getName());
		}
		
		return pocet;
		
	}
	
	/**
	 * Used to update Tickets' category as 'permanent' for specified event 
	 * 		where there exists permanent ticket for another game for the same place
	 * 		where Ticket's event is same type as chosen Event
	 * Used especially after creating new Event and creating new Tickets for it
	 * @param event wanted Event to update Tickets for
	 * @throws SQLException when sql throws its exception
	 * @throws EmptyResultListException when one of classes' methods throws this exception
	 */
	public static void updatePermanents(Event event) throws SQLException, EmptyResultListException {

		//zober miesta na ktore je permanentka a event ma rovnaky type (zc alebo playoff)
		List<Integer> placeIds = new ArrayList<Integer>();
		String sqlSelect = "SELECT DISTINCT(place_id) "
				+ "FROM tickets t "
				+ "INNER JOIN events e on (e.id = t.event_id) "
				+ "WHERE t.category = 'permanent' AND e.type = ?";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sqlSelect)){
			ps.setString(1, event.getType());
			try(ResultSet rs = ps.executeQuery()){
				while(rs.next()) {
					placeIds.add(rs.getInt(1));
				}
			}
		}
		
		//vytvor sql array z List<Integer>
		//pomoc z https://stackoverflow.com/questions/25600598/operator-does-not-exist-integer-integer-in-a-query-with-any
		List<Object> objListPlace = new ArrayList<Object>(placeIds);
		Object[] objArrayPlace = objListPlace.toArray();
		Array arrayPlace = DBContext.getConnection().createArrayOf("INTEGER", objArrayPlace);
		
		
		//executeUpdate s 
		String sqlUpdate = "UPDATE tickets SET "
				+ "(is_free, is_sold, is_used, category) "
			 + " = (false, true, false, 'permanent') "
			 + "WHERE event_id = ? "
			 + "AND place_id = ANY ( (SELECT ?)::int[] ) ";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sqlUpdate)){
			ps.setInt(1, event.getId());
			ps.setArray(2, arrayPlace);
			
			ps.executeUpdate();
			//int pocet = ps.executeUpdate();
			//System.out.println(pocet + " rows updated as permanents!");
		}
		
		
	}
	
	/**
	 * Used to create Reservations for chosen Places, Event, Customer, category
	 * Inserts one Reservation into database for each Place with same Event, Customer, Category
	 * Updates Tickets of specified Event and Place as reserved - isFree to false and sets category
	 * Returns list of newly created Reservations
	 * Is transaction with isolation level: REPEATABLE READ:
	 * 		if non-repeatable read occurs: cancel the transaction, throw ProcessCancelledException
	 * 		serialization anomaly should not occur
	 * @param places list of wanted places
	 * @param event wanted event
	 * @param customer wanted customer
	 * @param category wanted category for the Tickets
	 * @return list of created Reservations
	 * @throws SQLException when sql throws its exception
	 * @throws EmptyResultListException when no Tickets are found to be reserved
	 * @throws IdNotFoundException when classes' methods throw this exception
	 * @throws WrongInputException when Ticket is chosen that is not free
	 * @throws ProcessCancelledException when transaction non-repeatable read occurs
	 */
	public static List<Reservation> reserveTickets(List<Place> places, Event event, Customer customer, String category) throws SQLException, EmptyResultListException, IdNotFoundException, WrongInputException, ProcessCancelledException {
		
		Connection c = DBContext.getConnection();
		c.setTransactionIsolation(Connection.TRANSACTION_REPEATABLE_READ);
		c.setAutoCommit(false);

		List<Ticket> tickets = new ArrayList<Ticket>();
		List<Reservation> reservations = new ArrayList<Reservation>();
		
		try {
			//ziskaj vybrane listky
			for(Place place : places) {
				Ticket ticket = TicketFinder.getInstance().findByEventPlace(event.getId(), place.getId());
				if(!ticket.isFree()) throw new WrongInputException("Ticket is not free");
				tickets.add(ticket);
			}

			//kotrola listkov - existuje? je este volny?
			if(tickets.isEmpty()) throw new EmptyResultListException("No tickets to reserve found");
			for(Ticket ti : tickets) {
				if(!ti.isFree()) throw new IllegalArgumentException("Ticket " + ti.getId() + " is not free!");
			}
			
			//oznac listky ako obsadene (not free) a nastav kategoriu
			for(Ticket ti : tickets) {
				ti.setFree(false);
				ti.setCategory(category);
				ti.update();
			}


			//pridaj do rezervacii
			for(Ticket ti : tickets) {
				Reservation reservation = new Reservation();
				reservation.setCustomerId(customer.getId());
				reservation.setTicketId(ti.getId());
				reservation.insert();
				reservations.add(reservation);
			}

		} catch(PSQLException e) {
			throw new ProcessCancelledException("Transaction exception thrown");
		} finally {
			c.commit();
			c.setAutoCommit(true);
		}
		
		return reservations;

	}
	
	
	public static int midnightCheck() throws SQLException {
		int affected = 0;
		List<Integer> ids = findForCheck();
		List<Object> objList = new ArrayList<Object>(ids);
		Object[] objArray = objList.toArray();
		Array array = DBContext.getConnection().createArrayOf("INTEGER", objArray);
		String sql = "UPDATE tickets SET "
				+ "is_used = true "
				+ "WHERE id = ANY ( (SELECT ?)::int[] )";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			ps.setArray(1, array);
			affected = ps.executeUpdate();
		}
		return affected;
	}
	
	public static List<Integer> findForCheck() throws SQLException {
		List<Integer> ids = new ArrayList<>();
		String sql = "SELECT t.id FROM tickets t "
				+ "INNER JOIN events e on (e.id = t.event_id) "
				+ "WHERE t.is_used = false and e.date < (SELECT now()) ";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			try(ResultSet rs = ps.executeQuery()) {
				while(rs.next()) {
					ids.add(rs.getInt(1));
				}
			}
 		}
		System.out.println("Ids size:"+ids.size());
		return ids;
	}
	
	
	public static int midnightCheckTransaction() throws SQLException, EmptyResultException, ProcessCancelledException {
		Connection c = DBContext.getConnection();
		c.setAutoCommit(false);
		c.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);
		int affected = -1;
		try {
			affected = midnightCheck();
			
			if(affected > 0) return affected;
			else throw new EmptyResultException("No tickets affected by update");
		} catch(PSQLException e) {
			throw new ProcessCancelledException("Transaction exception thrown");
		} finally {
			c.commit();
			c.setAutoCommit(true);
		}
	}
	
	

}
