package rdg;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Time;
import java.util.ArrayList;
import java.util.List;

import Exceptions.EmptyResultException;
import Exceptions.EmptyResultListException;
import Exceptions.IdNotFoundException;
import Exceptions.MoreRowsReturnedException;
import Exceptions.WrongInputException;

import java.sql.Date;

import main.DBContext;

public class Event {
	private Integer id;
	private String name;
	private String type;
	private Date date;
	private Time time;
	
	public Integer getId() {
		return id;
	}
	public void setId(Integer id) {
		this.id = id;
	}
	
	public String getName() {
		return name;
	}
	public void setName(String name) {
		this.name = name;
	}
	
	public String getType() {
		return type;
	}
	public void setType(String type) {
		this.type = type;
	}
	
	public Date getDate() {
		return date;
	}
	public void setDate(Date date) {
		this.date = date;
	}
	
	public Time getTime() {
		return time;
	}
	public void setTime(Time time) {
		this.time = time;
	}
	
	/**
	 * Prints all values of an Event
	 */
	public void print() {
		System.out.println(id + " " + name + " " + type + " " + date + " " + time);
	}
	
	/**
	 * Inserts new event into database with current values
	 * @throws SQLException when sql update throw its exception
	 */
	public void insert() throws SQLException {
		String sql = "INSERT INTO events (name, type, date, time) VALUES (?,?,?,?)";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql, Statement.RETURN_GENERATED_KEYS)){
			ps.setString(1, name);
			ps.setString(2, type);
			ps.setDate(3, date);
			ps.setTime(4, time);
			
			ps.executeUpdate();
			
			try(ResultSet rs = ps.getGeneratedKeys()){
				rs.next();
				id = rs.getInt(1);
			}
		}
	}
	
	/**
	 * Updates current values of an event in database
	 * @throws SQLException when sql throws its exception
	 * @throws IdNotFoundException when id is null
	 */
	public void update() throws SQLException, IdNotFoundException {
		if(id == null) throw new IdNotFoundException("Event id not found");
		
		String sql = "UPDATE events SET name = ?, type = ?, date = ?, time = ? WHERE id = ?";
		try (PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)){
			ps.setString(1,  name);
			ps.setString(2, type);
			ps.setDate(3, date);
			ps.setTime(4, time);
			ps.setInt(5,  id);
			
			ps.executeUpdate();
		}
	}
	
	/**
	 * Deletes event from database
	 * @throws SQLException when sql throws its exception
	 * @throws IdNotFoundException when id is null
	 */
	public void delete() throws SQLException, IdNotFoundException {
		if(id == null) throw new IdNotFoundException("Event id not found");
		String sql = "DELETE FROM events WHERE id = ?";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)){
			ps.setInt(1, id);
			ps.executeUpdate();
		}
	}
	
	/**
	 * Returns the amount of all tickets from database assigned for this event
	 * @return amount of all Tickets with this event's id as eventId
	 * @throws SQLException when sql throws its exception
	 * @throws IdNotFoundException when id is null
	 */
	public int getAllTicketCount() throws SQLException, IdNotFoundException {
		if(id == null) throw new IdNotFoundException("Event id is null");
		
		String sql = "SELECT COUNT(id) FROM tickets WHERE event_id = ?";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)){
			ps.setInt(1, id);
			
			try(ResultSet rs = ps.executeQuery()){
				if(rs.next()) {
					int pocet = rs.getInt(1);
					if(rs.next()) System.out.println("More rows returned");
					return pocet;
				}
			}
			return -1;
		}
	}
	
	/**
	 * Returns the amount of occupied Tickets from database assigned for this event
	 * @return amount of all Tickets with this event's id as eventId and isFree is false
	 * @throws SQLException when sql throws its exception
	 * @throws IdNotFoundException when id is null
	 */
	public int getOccupiedTicketCount() throws SQLException, IdNotFoundException {
		if(id == null) throw new IdNotFoundException("Event id is null");
		
		String sql = "SELECT COUNT(id) FROM tickets WHERE event_id = ? and is_free = false";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)){
			ps.setInt(1, id);
			
			try(ResultSet rs = ps.executeQuery()){
				if(rs.next()) {
					int pocet = rs.getInt(1);
					if(rs.next()) System.out.println("More rows returned");
					return pocet;
				}
			}
			return -1;
		}
	}
	
	/**
	 * Returns the amount of reserved Tickets from database assigned for this event
	 * @return amount of all Tickets with this event's id as eventId, isFree is false and isSold is false
	 * @throws SQLException when sql throws its exception
	 * @throws IdNotFoundException when id is null
	 */
	public int getReservedTicketCount() throws SQLException, IdNotFoundException {
		if(id == null) throw new IdNotFoundException("Event id is null");
		
		String sql = "SELECT COUNT(id) FROM tickets WHERE event_id = ? and is_free = false and is_sold = false";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)){
			ps.setInt(1, id);
			
			try(ResultSet rs = ps.executeQuery()){
				if(rs.next()) {
					int pocet = rs.getInt(1);
					if(rs.next()) System.out.println("More rows returned");
					return pocet;
				}
			}
			return -1;
		}
	}
	
	/**
	 * Returns the amount of bought Tickets from database assigned for this event
	 * @return amount of all Tickets with this event's id as eventId, isFree is false, isSold is true
	 * @throws SQLException when sql throws its exception
	 * @throws IdNotFoundException when id is null
	 */
	public int getBoughtTicketCount() throws SQLException, IdNotFoundException {
		if(id == null) throw new IdNotFoundException("Event id is null");
		
		String sql = "SELECT COUNT(id) FROM tickets WHERE event_id = ? and is_free = false and is_sold = true";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)){
			ps.setInt(1, id);
			
			try(ResultSet rs = ps.executeQuery()){
				if(rs.next()) {
					int pocet = rs.getInt(1);
					if(rs.next()) System.out.println("More rows returned");
					return pocet;
				}
			}
			return -1;
		}
	}
	 
	/**
	 * Returns current sales of this Event
	 * Doesn't include sales from Permanent tickets
	 * @return sum of Tariffs of Tickets Assigned for this Event whose category is not 'permanent'
	 * @throws SQLException when sql throws its exception
	 * @throws MoreRowsReturnedException when more results of sales are returned
	 * @throws EmptyResultException when no result is returned
	 * @throws IdNotFoundException when id is null
	 */
	public Double getSales() throws SQLException, MoreRowsReturnedException, EmptyResultException, IdNotFoundException {
		if(id == null) throw new IdNotFoundException("Event id not set");
		
		//sales okream permanentiek
		String sql = "SELECT SUM(ta.price) FROM tariffs ta"
				+ " INNER JOIN assigned a on (a.tariff_id = ta.id)"
				+ " INNER JOIN tickets ti USING(event_id)"
				+ " INNER JOIN places p on (ti.place_id = p.id)"
				+ " WHERE ti.event_id = ? AND"
				+ " 	ti.category = ta.category AND"
				+ " 	p.sector_id = a.sector_id AND"
				+ "		ti.category != 'permanent'";
		
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)){
			ps.setInt(1, id);
			try(ResultSet rs = ps.executeQuery()){
				if(rs.next()) {
					double sales = rs.getDouble(1);
					if(rs.next()) throw new MoreRowsReturnedException("More rows returned");
					return sales;
				} else {
					throw new EmptyResultException("Empty result returned");
				}
			}
		}
	}

	/**
	 * Returns list of available categories of Tickets for specified Sector 
	 * @param sector is wanted Sector
	 * @return list of available categories for specified Sector
	 * @throws SQLException when sql throws its exception
	 * @throws EmptyResultListException when no categories are returrned
	 * @throws IdNotFoundException when id is null
	 * @throws WrongInputException whens sector is null
	 */
	public List<String> getSectorCategories(Sector sector) throws SQLException, EmptyResultListException, IdNotFoundException, WrongInputException {
		if(id == null) throw new IdNotFoundException("Event id not set");
		if(sector == null) throw new WrongInputException("Sector is null");
		List<String> categories = new ArrayList<String>();
		
		String sql = "SELECT ta.category FROM assigned a "
				+ "INNER JOIN tariffs ta ON (a.tariff_id = ta.id) "
				+ "WHERE event_id = ? AND sector_id = ?";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			ps.setInt(1, id);
			ps.setInt(2, sector.getId());
			try(ResultSet rs = ps.executeQuery()) {
				while(rs.next()) {
					categories.add(rs.getString(1));
				}
				
				if(categories.isEmpty()) throw new EmptyResultListException("No categories returned");
			}
		}
		
		return categories;
	}

}
