package rdg;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.time.LocalDateTime;

import Exceptions.IdNotFoundException;
import Exceptions.MoreRowsReturnedException;
import main.DBContext;

public class Reservation {
	private Integer id;
	private LocalDateTime validFrom;
	private LocalDateTime validTill;
	private Boolean isConfirmed;
	private Boolean isPaid;
	private Integer customerId;
	private Integer ticketId;
	
	public Integer getId() {
		return id;
	}
	public void setId(Integer id) {
		this.id = id;
	}
	
	public LocalDateTime getValidFrom() {
		return validFrom;
	}
	public void setValidFrom(LocalDateTime validFrom) {
		this.validFrom = validFrom;
	}
	
	public LocalDateTime getValidTill() {
		return validTill;
	}
	public void setValidTill(LocalDateTime validTill) {
		this.validTill = validTill;
	}
	
	public boolean isConfirmed() {
		return isConfirmed;
	}
	public void setConfirmed(boolean isConfirmed) {
		this.isConfirmed = isConfirmed;
	}
	
	public boolean isPaid() {
		return isPaid;
	}
	public void setPaid(boolean isPaid) {
		this.isPaid = isPaid;
	}
	
	public Integer getCustomerId() {
		return customerId;
	}
	public void setCustomerId(Integer customerId) {
		this.customerId = customerId;
	}
	
	public Integer getTicketId() {
		return ticketId;
	}
	public void setTicketId(Integer ticketId) {
		this.ticketId = ticketId;
	}
	
	/**
	 * Prints current values of a Reservation
	 */
	public void print() {
		System.out.println(id + " " + validFrom + " " + validTill + " " + isConfirmed + " " + isPaid + " " + customerId + " " + ticketId);
	}
	
	/**
	 * Returns values of interest of Event and Place
	 * using value ticket_id to get Event and Place from Ticket
	 * linked to this Reservation.
	 * Returns the above in a string.
	 * @return string containing event id, event valid from and till, place info
	 */
	public String toString() {
		String string = "";
		try {
			Ticket ticket = TicketFinder.getInstance().findById(ticketId);
			Event event = EventFinder.getInstance().findById(ticket.getEventID());
			Place place = PlaceFinder.getInstance().findById(ticket.getPlaceID());
			string += "id:" + id + " " + event.getName() + " valid from:" + validFrom + " valid till:" + validTill + 
				" confirmed:" + isConfirmed + " paid:" + isPaid + " category:" + ticket.getCategory();
			if(place.getLoungeName() == null) string += " row:" + place.getRowNumber() + " seat:" + place.getSeatNumber();
			else string += " lounge:" + place.getLoungeName();
			
		} catch (SQLException | MoreRowsReturnedException e) {
			e.printStackTrace();
		}
		
		return string;
		
	}
	
	/**
	 * Insert new Reservation into database with current values
	 * @throws SQLException when sql update throws its exception
	 */
	public void insert() throws SQLException {
		
		if(validFrom == null) validFrom = LocalDateTime.now();
		if(validTill == null) validTill = validFrom.plusMinutes(30);
		if(isConfirmed == null) isConfirmed = false;
		if(isPaid == null) isPaid = false;
		
		String sql = "INSERT INTO reservations(valid_from, valid_till, is_confirmed, is_paid, customer_id, ticket_id) "
				+ "VALUES (?,?,?,?,?,?)";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql, Statement.RETURN_GENERATED_KEYS)) {
			ps.setTimestamp(1, Timestamp.valueOf(validFrom));
			ps.setTimestamp(2, Timestamp.valueOf(validTill)); 
			ps.setBoolean(3, isConfirmed());
			ps.setBoolean(4, isPaid());
			ps.setInt(5, customerId);
			ps.setInt(6, ticketId);
			
			ps.executeUpdate();
			
			try(ResultSet rs = ps.getGeneratedKeys()) {
				rs.next();
				id = rs.getInt(1);
			}
		}
	}
	
	public void update() throws IdNotFoundException, SQLException {
		if(id == null) throw new IdNotFoundException("Reservation id not found");
		
		String sql = "UPDATE reservations SET valid_from = ?, valid_till = ?, is_confirmed = ?, is_paid = ?, "
				+ "customer_id = ?, ticket_id = ? WHERE id = ?";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			ps.setTimestamp(1, Timestamp.valueOf(validFrom));
			ps.setTimestamp(2, Timestamp.valueOf(validTill));
			ps.setBoolean(3, isConfirmed);
			ps.setBoolean(4, isPaid);
			ps.setInt(5, customerId);
			ps.setInt(6, ticketId);
			ps.setInt(7, id);
			
			ps.executeUpdate();
		}
	}
	
	public void confirm() throws IdNotFoundException, SQLException {
		if(isConfirmed()) throw new IllegalArgumentException("Reservation is already confirmed!");
		
		isConfirmed = true;
		validTill = validFrom.plusDays(3);
		
		update();
	}
}
