package rdg;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import Exceptions.IdNotFoundException;
import Exceptions.MoreRowsReturnedException;
import main.DBContext;

public class Ticket {
	private Integer id;
	private boolean isFree;
	private boolean isSold;
	private boolean isUsed;
	private String category;
	private Integer eventID;
	private Integer placeID;
	
	public void setId(Integer id) {
		this.id = id;
	}
	public Integer getId() {
		return id;
	}
	
	public void setFree(boolean isFree) {
		this.isFree = isFree;
	}
	public boolean isFree() {
		return isFree;
	}
	
	public void setSold(boolean isSold) {
		this.isSold = isSold;
	}
	public boolean isSold() {
		return isSold;
	}
	
	public void setUsed(boolean isUsed) {
		this.isUsed = isUsed;
	}
	public boolean isUsed() {
		return isUsed;
	}
	
	public void setCategory(String category) {
		this.category = category;
	}
	public String getCategory() {
		return category;
	}
	
	public void setEventID(Integer eventID) {
		this.eventID = eventID;
	}
	public Integer getEventID() {
		return eventID;
	}
	
	public void setPlaceID(Integer placeID) {
		this.placeID = placeID;
	}
	public Integer getPlaceID() {
		return placeID;
	}
	
	/**
	 * Prints current values of a Ticket
	 */
	public void print() {
		System.out.println(id + " " + isFree + " " + isSold + " " + isUsed + " " + category + " " + eventID + " " + placeID);
		
	}
	
	/**
	 * Prints ticket with the information about its event and place
	 * @throws SQLException when sql update throws its exception
	 * @throws IdNotFoundException when one of eventId, placeId, sectorId is null
	 * @throws MoreRowsReturnedException when more than one results of a events, place, sector were returned
	 */
	public void printNice() throws SQLException, IdNotFoundException, MoreRowsReturnedException {
		Place place = PlaceFinder.getInstance().findById(placeID);
		if(place == null) throw new IdNotFoundException("Place id not found");
		
		Event event = EventFinder.getInstance().findById(eventID);
		if(event == null) throw new IdNotFoundException("Event id not found");
		
		Sector sector = SectorFinder.getInstance().findById(place.getSectorId());
		if(sector == null) throw new IdNotFoundException("Sector id not found");
		
		String vysl = "id:" + id + " ";
		vysl += "Event:" + event.getName() + " ";
		vysl += "Sector:" + sector.getName() + " ";
		
		if(place.getLoungeName() == null) {
			vysl += "Row:" + place.getRowNumber() + " ";
			vysl += "Seat:" + place.getSeatNumber();
		} else {
			vysl += "Lounge:" + place.getLoungeName();
		}
		System.out.println(vysl);
		
	}
	
	/**
	 * Updates current values of a Ticket in database
	 * @throws SQLException when sql update throws its exception
	 * @throws IdNotFoundException when id is null
	 */
	public void update() throws IdNotFoundException, SQLException {
		if(id == null) throw new IdNotFoundException("Ticket id not found");
		
		String sql = "UPDATE tickets SET is_free = ?, is_sold = ?, is_used = ?, category = ?, event_id = ?, place_id = ? "
				+ "where id = ?";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			ps.setBoolean(1, isFree);
			ps.setBoolean(2, isSold);
			ps.setBoolean(3, isUsed);
			ps.setString(4, category);
			ps.setInt(5, eventID);
			ps.setInt(6, placeID);
			ps.setInt(7, id);
			
			ps.executeUpdate();
		}
	}
	
	/**
	 * Returns the price of this Ticket based on its category and placeID
	 * Uses classes of Assigned and Tariff to reach this. 
	 * @return price of this ticket in double
	 * @throws IdNotFoundException when one of needed classes'id is null
	 * @throws SQLException when sql query throws its exception
	 * @throws MoreRowsReturnedException when one of the classes finders throws its exception
	 */
	public Double getPrice() throws IdNotFoundException, SQLException, MoreRowsReturnedException {
		if(id == null) throw new IdNotFoundException("Ticket id not found");
		if(category == null) throw new IllegalArgumentException("Category not set");
		
		Double price = null;
		Place place = PlaceFinder.getInstance().findById(placeID);
		
		String sql = "SELECT ta.price FROM tariffs ta "
				+ "INNER JOIN assigned a ON (ta.id = a.tariff_id) "
				+ "WHERE a.event_id = ? AND a.sector_id = ? AND ta.category = ?";
		try(PreparedStatement ps = DBContext.getConnection().prepareStatement(sql)) {
			ps.setInt(1, eventID);
			ps.setInt(2, place.getSectorId());
			ps.setString(3, category);
			try(ResultSet rs = ps.executeQuery()) {
				if(rs.next()) {
					price = rs.getDouble(1);
				}
				if(rs.next()) {
					throw new MoreRowsReturnedException("More prices returned");
				}
				return price;
			}
		}
	}
}
