package ui;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.sql.SQLException;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import Exceptions.EmptyResultException;
import Exceptions.EmptyResultListException;
import Exceptions.IdNotFoundException;
import Exceptions.InsufficientFundsException;
import Exceptions.MoreRowsReturnedException;
import Exceptions.ProcessCancelledException;
import Exceptions.WrongInputException;
import Transactions.PaymentsManager;
import Transactions.PlaceManager;
import Transactions.ReservationManager;
import Transactions.SectorManager;
import Transactions.TicketManager;
import rdg.Admin;
import rdg.AdminFinder;
import rdg.Customer;
import rdg.CustomerFinder;
import rdg.Event;
import rdg.EventFinder;
import rdg.Place;
import rdg.Reservation;
import rdg.Sector;
import rdg.Ticket;
import rdg.TicketFinder;

public class TicketsMenu extends Menu{
	
	private Integer customerId;
	
	@Override
	public void print() {
		System.out.println("*****************************");
		System.out.println("0. exit");
		System.out.println("1. find ticket by id");
		System.out.println("2. show reserved");
		System.out.println("3. show sold and not used");
		System.out.println("4. show sold and used");
		System.out.println("5. set used by id");
		System.out.println("6. reserve tickets");
		System.out.println("7. confirm reservations");
		System.out.println("8. buy tickets online");
		System.out.println("9. buy tickets at box office");
		System.out.println("*****************************");
	}
	

	
	@Override
	public void handle(String option) {
		try {
			switch(option) {
			default: System.out.println("Unknown command");break;
			case "0": mainMenu();break;
			case "1": printNiceId();break;
			case "2": showReserved();break;
			case "3": showSoldNotUsed();break;
			case "4": showSoldAndUsed();break;
			case "5": setUsed();break;
			case "6": reserveTickets();break;
			case "7": confirmReservation();break;
			case "8": buyTicketsOnline();break;
			case "9": buyTicketsBoxOffice();break;
			}
		}catch(SQLException | IOException | IdNotFoundException | EmptyResultListException | NumberFormatException | EmptyResultException | WrongInputException | MoreRowsReturnedException | InsufficientFundsException | ProcessCancelledException e) {
			System.out.println("Error:");
			e.printStackTrace();
		}
		
	}
	//0. back to main menu
	public void mainMenu() throws IOException {
		exit();
		MainMenu mMenu = new MainMenu();
		mMenu.run();
	}
	//1. print Nice by Id
	public void printNiceId() throws SQLException, IdNotFoundException, NumberFormatException, IOException, MoreRowsReturnedException {
		BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
		
		System.out.println("Enter ticket id:");
		int id = Integer.valueOf(br.readLine());
		
		Ticket ticket = TicketFinder.getInstance().findById(id);
		if(ticket == null) throw new IdNotFoundException("Ticket id not found");
		
		ticket.print();
		System.out.println("--------------");
		ticket.printNice();
		
	}
	//2. show reserved tickets
	public void showReserved() throws SQLException, IdNotFoundException, EmptyResultListException, NumberFormatException, IOException, MoreRowsReturnedException {
		BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
		
		System.out.println("Enter offset:");
		int offset = Integer.valueOf(br.readLine());
		
		List<Ticket> reserved = TicketManager.getReserved(10, offset);
		
		if(reserved == null) throw new EmptyResultListException("Empty list returned");
		for(int i = 0; i < 10; i++) {
			reserved.get(i).printNice();
		}
	}
	//3. show sold and not used
	public void showSoldNotUsed() throws SQLException, EmptyResultListException, IdNotFoundException, NumberFormatException, IOException, MoreRowsReturnedException {
		BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
		
		System.out.println("Enter offset:");
		int offset = Integer.valueOf(br.readLine());
		
		List<Ticket> soldNotUsed = TicketManager.getSoldNotUsed(10, offset);
		if(soldNotUsed == null) throw new EmptyResultListException("Empty list returned");
		for(int i = 0; i < 10; i++) {
			soldNotUsed.get(i).printNice();
		}
	}
	//4. show sold and used
	public void showSoldAndUsed() throws SQLException, EmptyResultListException, IdNotFoundException, NumberFormatException, IOException, MoreRowsReturnedException {
		BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
		
		System.out.println("Enter offset:");
		int offset = Integer.valueOf(br.readLine());
		
		List<Ticket> soldAndUsed = TicketManager.getSoldAndUsed(10, offset);
		if(soldAndUsed == null) throw new EmptyResultListException("Empty list returned");
		for(int i = 0; i < 10; i++) {
			soldAndUsed.get(i).printNice();
		}
	}
	//5. set used for ticket by id
	public void setUsed() throws SQLException, IdNotFoundException, NumberFormatException, IOException, EmptyResultException, WrongInputException, MoreRowsReturnedException {
		BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
		
		System.out.println("Enter ticket id:");
		int id = Integer.valueOf(br.readLine());
		
		TicketManager.setUsedByID(id);
	}
	
	
	//6. reserve tickets
	public List<Reservation> reserveTickets() throws NumberFormatException, IOException, SQLException, IdNotFoundException, EmptyResultListException, EmptyResultException, ProcessCancelledException, WrongInputException, MoreRowsReturnedException {
		BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
		List<Reservation> reservations;
		
		//enter your id
		int customerId;
		if(this.customerId == null) {
			System.out.println("Enter your id:");
			customerId = Integer.valueOf(br.readLine());
		} else customerId = this.customerId;
		
		
		//find customer
		Customer customer = CustomerFinder.getInstance().findById(customerId);
		System.out.println("Good day " + customer.getFirstName() + " " + customer.getLastName());
		
		//enter event id
		System.out.println("Enter event id:");
		int eventId = Integer.valueOf(br.readLine());
		
		//find event
		Event event = EventFinder.getInstance().findById(eventId);
		if(event == null) throw new IdNotFoundException("Event with this id not found.");
		
		//list available sectoros to choose for this event
		List<Sector> sectors = SectorManager.getSectors(event.getId());
		System.out.println("Sectors available for " + event.getName() + ":");
		String sectList = SectorManager.listString(sectors);
		System.out.println(sectList);
		
		//enter sector
		String inputSect = br.readLine();
		Sector sector = SectorManager.getChosenSector(sectors, inputSect);
		
		//if sector is VIP
		if(sector.getName().length() > 3 && sector.getName().substring(0, 3).equals("VIP")) {
			//list free lounges
			System.out.println("Sector " + sector.getName() + " VIP sector" + sector.getId());
			List<Place> lounges = PlaceManager.getFreeLounges(event, sector);
			String loungeString = PlaceManager.getLoungeString(lounges);
			System.out.println(loungeString);
			
			//choose lounges
			System.out.println("Choose lounges:     napr: L1,L3");
			String inputLounge = br.readLine();
			List<Place> chosenLounges = PlaceManager.getChosenPlaces(lounges, Arrays.asList(inputLounge.split(",")));
			
			//choose category
			System.out.println("Choose id category from:");
			List<String> categories = event.getSectorCategories(sector);
			for(String cat : categories) {
				System.out.println(categories.indexOf(cat) + " " + cat);
			}
			int inputCategory = Integer.valueOf(br.readLine());
			String category = categories.get(inputCategory);
			
			//confirmation
			System.out.println("Confirm: Reserve these lounges?     enter = yes   |   write anything (+ enter )for no");
			System.out.println("Event: " + event.getName() + " Sector:" + sector.getName());
			String out = "Lounges: ";
			for(Place p : chosenLounges) out += p.getLoungeName() + ",";
			out += "   Category:" + category;
			System.out.println(out);
			String inputConfirm = br.readLine();
			if(inputConfirm.equals("")) {
				reservations = TicketManager.reserveTickets(chosenLounges, event, customer, category);
				System.out.println("Successfully reserved! You have 30 minutes to confirm reservations!");
			} else {
				throw new ProcessCancelledException("Reservation process cancelled by user!");
			}	
			
		} // if sector is basic
		else {
			//print rows + number of empty seats
			System.out.println("Chosen sector: " + sector.getName());
			List<Integer> rows = PlaceManager.getRows(sector.getId());
			Collections.sort(rows);
			System.out.println("Choose row:");
			Map<Integer, Integer> rowCounts = PlaceManager.getFreeSeatsCount(event.getId(), sector.getId());
			for(Integer row : rowCounts.keySet()) {
				System.out.println("Row no. " + row + " has "
						+ rowCounts.get(row) + "empty seats out of "
						+ PlaceManager.getSeats(sector.getId(), row).size());
			}
			//enter row number
			int rowNumber = Integer.valueOf(br.readLine());
			
			//list empty seat numbers
			List<Place> emptySeats = PlaceManager.getFreeSeatsForEvent(event.getId(), sector.getId(), rowNumber);
			
			//choose seats
			System.out.println("Choose seat numbers in row " + rowNumber + "     napr: 2,15,34");
			System.out.println(PlaceManager.getSeatNumbersString(emptySeats));
			
			String inputSeats = br.readLine();
			List<Place> chosenPlaces = PlaceManager.getChosenPlaces(emptySeats, Arrays.asList(inputSeats.split(",")));
			
			//choose category
			System.out.println("Choose category from:");
			List<String> categories = event.getSectorCategories(sector);
			for(String cat : categories) {
				System.out.println(categories.indexOf(cat) + " " + cat);
			}
			int inputCategory = Integer.valueOf(br.readLine());
			String category = categories.get(inputCategory);
			
			//confirmation
			System.out.println("Confirm: Reserve these seats?     enter = yes   |   write anything (+ enter )for no");
			System.out.println("Event: " + event.getName() + " Sector:" + sector.getName() + " Row:" + rowNumber);
			String out = "Seats: ";
			for(Place p : chosenPlaces) out += p.getSeatNumber() + ",";
			out += "   Category:" + category;
			
			System.out.println(out);
			String inputConfirm = br.readLine();
			if(inputConfirm.equals("")) {
				reservations = TicketManager.reserveTickets(chosenPlaces, event, customer, category);
				System.out.println("Successfully reserved!");
				if(this.customerId == null) System.out.println("You have 30 minutes to confirm reservations!");
			} else {
				throw new ProcessCancelledException("Reservation process cancelled by user!");
			}
		}
		return reservations;

	}
	
	// 7. confirm reservation
	public void confirmReservation() throws NumberFormatException, IOException, SQLException, EmptyResultListException, IdNotFoundException, ProcessCancelledException {
		BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
		
		//enter your id
		System.out.println("Enter your id:");
		int customerId = Integer.valueOf(br.readLine());
		
		//find customer
		Customer customer = CustomerFinder.getInstance().findById(customerId);
		System.out.println("Good day " + customer.getFirstName() + " " + customer.getLastName());
		
		//find reservations
		List<Reservation> reservations = ReservationManager.getUnconfirmedReservations(customer);
		for(Reservation r : reservations) {
			System.out.println(r.toString());
		}
		
		//choose reservation
		System.out.println("Enter reservation IDs to confirm       napr.: 123,456   empty = confirm all   \"no\" to refuse");
		List<Reservation> chosenReservations;
		String inputRes = br.readLine();
		if(inputRes.equals("")) chosenReservations = reservations;
		else if(inputRes.equals("no")) {
			throw new ProcessCancelledException("Reservations confirmation process cancelled by user!");
		}
		else chosenReservations = ReservationManager.getChosenReservations(reservations, Arrays.asList(inputRes.split(",")));
		
		//cofirm chosen reservations
		for(Reservation reservation : chosenReservations) {
			reservation.confirm();
		}
		
		System.out.println("Successfully confirmed! You have 3 days to pay for your tickets.");
	}
	
	//8. buy ticket online
	public void buyTicketsOnline() throws NumberFormatException, IOException, SQLException, IdNotFoundException, EmptyResultListException, EmptyResultException, MoreRowsReturnedException, WrongInputException, InsufficientFundsException, ProcessCancelledException {
		BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
		
		//enter your id
		System.out.println("Enter your id:");
		int customerId = Integer.valueOf(br.readLine());
		this.customerId = customerId;
		
		//find customer
		Customer customer = CustomerFinder.getInstance().findById(customerId);
		System.out.println("Good day " + customer.getFirstName() + " " + customer.getLastName());
		
		//pay for reserved?
		System.out.println("Do you already have reservations?");
		System.out.println("0. yes     1. no");
		int answer = Integer.valueOf(br.readLine());
		
		List<Reservation> reservations;
		//new reservations + immediately confirm
		if(answer == 1) {
			reservations = reserveTickets();
			for(Reservation r : reservations) r.confirm();
		}
		//get unpaid reservations
		else {
			reservations = ReservationManager.getUnpaidReservations(customer);
		}
		//get price for reservations
		Double priceSum = ReservationManager.getPriceForReservations(reservations);
		
		//print info
		System.out.println("Reservations:");
		for(Reservation reservation : reservations) {
			Double priceOne = TicketFinder.getInstance().findById(reservation.getTicketId()).getPrice();
			System.out.println(reservation.toString() + " price:" + priceOne);
		}
		System.out.println("Cost together: " + priceSum);
		
		//choose reservations to pay
		System.out.println("Your wallet: " + customer.getWallet() + " Points:" + customer.getPoints());
		System.out.println("Enter reservation id to pay for:        napr:1234,2580    enter for all      \"no\" to exit");
		String choice = br.readLine();
		List<Reservation> chosenReservations;
		if(choice.equals("no")) {
			throw new ProcessCancelledException("Payment process cancelled by user!");
		} else if (choice.equals("")) chosenReservations = reservations;
		else {
			chosenReservations = ReservationManager.getChosenReservations(reservations, Arrays.asList(choice.split(",")));
		}
		//final info - final price
		Double finalPrice = 0.0;
		for(Reservation reservation : chosenReservations) {
			//check if valid
			if(reservation.getValidTill().compareTo(LocalDateTime.now()) < 0) {
				throw new IllegalArgumentException("This reservation was only valid till " + reservation.getValidTill().toString());
			}
			Double price = TicketFinder.getInstance().findById(reservation.getTicketId()).getPrice();
			finalPrice += price;
			System.out.println(reservation.toString() + " price:" + price);
		}
		System.out.println("Together: " + finalPrice);
		//cofirm / proceed to payment
		System.out.println("Proceed to payment?         enter to procees     write anything (+enter) to cancell");
		String confirmation = br.readLine();
		if(confirmation.equals("")) {
			//payment to admin id = 1
			Admin admin = AdminFinder.getInstance().findById(1);
			//TRANSACTION
			PaymentsManager.payOnlineForReservations(customer, admin, chosenReservations);
			int pointsGained = PaymentsManager.givePoints(customer, finalPrice);
			System.out.println("Successfull payment! You were awarded " + pointsGained + " points!");
		} else {
			throw new ProcessCancelledException("Payment process cancelled by user!");
		}
	}
	
	//9. buy tickets at box office
	public void buyTicketsBoxOffice() throws NumberFormatException, IOException, SQLException, IdNotFoundException, EmptyResultListException, EmptyResultException, ProcessCancelledException, WrongInputException, MoreRowsReturnedException {
		BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
		
		//enter event id
		System.out.println("Enter event id:");
		int eventId = Integer.valueOf(br.readLine());
				
		//find event
		Event event = EventFinder.getInstance().findById(eventId);
		if(event == null) throw new IdNotFoundException("Event with this id not found.");
				
		//list available sectoros to choose for this event
		List<Sector> sectors = SectorManager.getSectors(event.getId());
		System.out.println("Sectors available for " + event.getName() + ":");
		String sectList = SectorManager.listString(sectors);
		System.out.println(sectList);
				
		//enter sector
		String inputSect = br.readLine();
		Sector sector = SectorManager.getChosenSector(sectors, inputSect);
		
		//if sector is VIP
		if(sector.getName().length() > 3 && sector.getName().substring(0, 3).equals("VIP")) {
			//list free lounges
			System.out.println("Sector " + sector.getName() + " VIP sector" + sector.getId());
			List<Place> lounges = PlaceManager.getFreeLounges(event, sector);
			String loungeString = PlaceManager.getLoungeString(lounges);
			System.out.println(loungeString);

			//choose lounges
			System.out.println("Choose lounges:     napr: L1,L3");
			String inputLounge = br.readLine();
			List<Place> chosenLounges = PlaceManager.getChosenPlaces(lounges, Arrays.asList(inputLounge.split(",")));

			//choose category
			System.out.println("Choose id category from:");
			List<String> categories = event.getSectorCategories(sector);
			for(String cat : categories) {
				System.out.println(categories.indexOf(cat) + " " + cat);
			}
			int inputCategory = Integer.valueOf(br.readLine());
			String category = categories.get(inputCategory);
			
			//confirmation
			System.out.println("Confirm: Buy these lounges?     enter = yes   |   write anything (+ enter )for no");
			System.out.println("Event: " + event.getName() + " Sector:" + sector.getName());
			String out = "Lounges: ";
			for(Place p : chosenLounges) out += p.getLoungeName() + ",";
			out += "   Category:" + category;
			//get tickets
			Double price = 0.0;
			List<Ticket> tickets = new ArrayList<Ticket>();
			for(Place p : chosenLounges) {
				out += p.getSeatNumber() + ",";
				//get tickets
				Ticket ticket = TicketFinder.getInstance().findByEventPlace(event.getId(), p.getId());
				ticket.setCategory(category);
				price += ticket.getPrice();
				tickets.add(ticket);
			}
			out += " Price:" + price;
			System.out.println(out);
			String inputConfirm = br.readLine();
			if(inputConfirm.equals("")) {
				Admin admin = AdminFinder.getInstance().findById(2);
				PaymentsManager.payBOForTickets(admin, tickets);
				System.out.println("Successfuly bought tickets!");
			} else {
				throw new ProcessCancelledException("Payment process cancelled by user!");
			}
		}
		else {
			//print rows + number of empty seats
			System.out.println("Chosen sector: " + sector.getName());
			List<Integer> rows = PlaceManager.getRows(sector.getId());
			Collections.sort(rows);
			System.out.println("Choose row:");
			Map<Integer, Integer> rowCounts = PlaceManager.getFreeSeatsCount(event.getId(), sector.getId());
			for(Integer row : rowCounts.keySet()) {
				System.out.println("Row no. " + row + " has "
						+ rowCounts.get(row) + "empty seats out of "
						+ PlaceManager.getSeats(sector.getId(), row).size());
			}
			//enter row number
			int rowNumber = Integer.valueOf(br.readLine());
			
			//list empty seat numbers
			List<Place> emptySeats = PlaceManager.getFreeSeatsForEvent(event.getId(), sector.getId(), rowNumber);
			
			//choose seats
			System.out.println("Choose seat numbers in row " + rowNumber + "     napr: 2,15,34");
			System.out.println(PlaceManager.getSeatNumbersString(emptySeats));
			
			String inputSeats = br.readLine();
			List<Place> chosenPlaces = PlaceManager.getChosenPlaces(emptySeats, Arrays.asList(inputSeats.split(",")));
			
			//choose category
			System.out.println("Choose category from:");
			List<String> categories = event.getSectorCategories(sector);
			for(String cat : categories) {
				System.out.println(categories.indexOf(cat) + " " + cat);
			}
			int inputCategory = Integer.valueOf(br.readLine());
			String category = categories.get(inputCategory);
			
			//confirmation
			System.out.println("Confirm: Buy these seats?     enter = yes   |   write anything (+ enter )for no");
			System.out.println("Event: " + event.getName() + " Sector:" + sector.getName() + " Row:" + rowNumber);
			String out = "Seats: ";
			//get tickets
			Double price = 0.0;
			List<Ticket> tickets = new ArrayList<Ticket>();
			for(Place p : chosenPlaces) {
				out += p.getSeatNumber() + ",";
				//get tickets
				Ticket ticket = TicketFinder.getInstance().findByEventPlace(event.getId(), p.getId());
				ticket.setCategory(category);
				price += ticket.getPrice();
				tickets.add(ticket);
			}
			out += "   Category:" + category;
			out += " Price:" + price;
			System.out.println(out);
			String inputConfirm = br.readLine();
			if(inputConfirm.equals("")) {
				Admin admin = AdminFinder.getInstance().findById(2);
				PaymentsManager.payBOForTickets(admin, tickets);
				System.out.println("Successfuly bought tickets!");
			} else {
				throw new ProcessCancelledException("Payment process cancelled by user!");
			}
		}
	}
	
}
