import java.io.PrintWriter
import java.util.*

/**
 * Vertex
 *
 * @property label
 * @constructor Create empty Vertex
 */
data class Vertex(var label: String)

/**
 * Edge
 *
 * @property label
 * @constructor Create empty Edge
 */
data class Edge(var label: Pair<Vertex, Vertex>)

/**
 * Graph
 *
 * @constructor Create empty Graph
 */
open class Graph {
    private val adjVertices: MutableMap<Vertex, MutableList<Vertex>> = mutableMapOf()

    /**
     * Get vertices
     *
     * @return
     */
    fun getVertices(): MutableSet<Vertex> {
        val list = mutableSetOf<Vertex>()
        for (vertex in adjVertices)
            list.add(vertex.key)
        return list
    }

    /**
     * Get edges
     *
     * @return
     */
    fun getEdges(): MutableSet<Edge> {
        val list = mutableSetOf<Edge>()
        for (edge in adjVertices)
            for (vertex in edge.value)
                if (!list.contains(Edge(Pair(edge.key, vertex))))
                    list.add(Edge(Pair(edge.key, vertex)))
        return list
    }

    private fun addVertex(label: Vertex) {
        adjVertices.putIfAbsent(label, ArrayList())
    }

    /**
     * Remove vertex
     *
     * @param label
     */
    fun removeVertex(label: Vertex) {
        adjVertices.values.stream().forEach { e: MutableList<Vertex> -> e.remove(label) }
        adjVertices.remove(label)
    }

    /**
     * Add edge
     *
     * @param label
     */
    fun addEdge(label: Edge) {
        val v1 = Vertex(label.label.first.label)
        val v2 = Vertex(label.label.second.label)
        if (adjVertices[v1]?.contains(v2) == false)
            adjVertices[v1]?.add(v2)
        if (adjVertices[v2]?.contains(v1) == false)
            adjVertices[v2]?.add(v1)
    }

    /**
     * Remove edge
     *
     * @param label
     */
    fun removeEdge(label: Edge) {
        val v1 = Vertex(label.label.first.label)
        val v2 = Vertex(label.label.second.label)
        val eV1: MutableList<Vertex>? = adjVertices[v1]
        val eV2: MutableList<Vertex>? = adjVertices[v2]
        eV1?.remove(v2)
        eV2?.remove(v1)
    }

    /**
     * Create graph
     *
     * @param vertices
     * @param edges
     * @return
     */
    fun createGraph(vertices: List<Vertex>, edges: List<Edge> = emptyList()): Graph {
        val graph = Graph()

        for (i in vertices)
            graph.addVertex(i)

        for (j in edges)
            graph.addEdge(j)

        return graph
    }

    private fun getAdjVertices(label: Vertex): MutableList<Vertex>? {
        return adjVertices[label]
    }

    /**
     * Depth first traversal
     *
     * @param graph
     * @param root
     * @return
     */
    fun depthFirstTraversal(graph: Graph, root: Vertex): Set<Vertex> {
        val visited: MutableSet<Vertex> = LinkedHashSet()
        val stack = Stack<Vertex>()
        stack.push(root)
        while (!stack.isEmpty()) {
            val vertex = stack.pop()
            if (!visited.contains(vertex)) {
                visited.add(vertex)
                for (v in graph.getAdjVertices(vertex)!!) {
                    stack.push(v)
                }
            }
        }
        return visited
    }

    /**
     * Breadth first traversal
     *
     * @param graph
     * @param root
     * @return
     */
    fun breadthFirstTraversal(graph: Graph, root: Vertex): Set<Vertex> {
        val visited: MutableSet<Vertex> = LinkedHashSet()
        val queue: Queue<Vertex> = LinkedList()
        queue.add(root)
        visited.add(root)
        while (!queue.isEmpty()) {
            val vertex = queue.poll()
            for (v in graph.getAdjVertices(vertex)!!) {
                if (!visited.contains(v)) {
                    visited.add(v)
                    queue.add(v)
                }
            }
        }
        return visited
    }

    /**
     * Write graph to file
     *
     * @param writer
     */
    fun writeGraphToFile(writer: PrintWriter) {
        for (edge in getEdges()) {
            writer.append("${edge.label.first.label} ${edge.label.second.label}")
            if (edge == getEdges().last())
                writer.append("\n")
            else
                writer.append("  ")
        }
    }

    private fun colorGraph(color: IntArray, pos: Int, c: Int): Boolean {
        if (color[pos] != -1 && color[pos] != c)
            return false

        // color this pos as c and all its neighbours as 1-c
        color[pos] = c
        var ans = true
        for (i in 0 until adjVertices.size) {
            if (adjVertices[Vertex(pos.toString())]?.contains(Vertex(i.toString())) == true) {
                if (color[i] == -1)
                    ans = ans and colorGraph(color, i, 1 - c)
                if (color[i] != -1 && color[i] != 1 - c)
                    return false
            }
            if (!ans)
                return false
        }
        return true
    }

    /**
     * Is bipartite
     *
     * @return
     */
    fun isBipartite(): Boolean {
        val color = IntArray(adjVertices.size)
        for (i in 0 until adjVertices.size)
            color[i] = -1
        val pos = 0

        return colorGraph(color, pos, 1)
    }


    // A utility function to find the vertex with minimum key value, from the set of vertices
    private fun minKey(key: IntArray, mstSet: Array<Boolean?>): Int {
        var min = Int.MAX_VALUE
        var minIndex = -1
        for (v in 0 until adjVertices.size)
            if (mstSet[v] == false && key[v] < min) {
                min = key[v]
                minIndex = v
            }
        return minIndex
    }


    private fun primMST(): IntArray {
        val parent = IntArray(adjVertices.size)
        val key = IntArray(adjVertices.size)
        val mstSet = arrayOfNulls<Boolean>(adjVertices.size)

        for (i in 0 until adjVertices.size) {
            key[i] = Int.MAX_VALUE
            mstSet[i] = false
        }

        key[0] = 0
        parent[0] = -1

        for (count in 0 until adjVertices.size - 1) {
            val u = minKey(key, mstSet)
            if (u == -1)
                return IntArray(0)
            mstSet[u] = true

            for (v in 0 until adjVertices.size)
                if (adjVertices[Vertex(u.toString())]?.contains(
                        Vertex(v.toString())) == true && mstSet[v] == false) {
                    parent[v] = u
                    key[v] = 1
                }
        }

        return parent
    }

    /**
     * Is connected
     *
     * @return
     */
    fun isConnected(): Boolean {
        val numberOfVertices = getVertices().size
        if (numberOfVertices - 1 > getEdges().size)
            return false
        else if (numberOfVertices - 1 > primMST().size)
            return false
        return true
    }
}