// @ Luk Gajdoech 2019

#include "PointCloudScan.h"
#include "Segmentation.h"

#include "Engine.h"
#include "DrawDebugHelpers.h"

#include "Loaders/PointCloudLoader.h"
#include "Loaders/COGSPointCloud.h"
#include "Structure/Octree.h"
#include "Structure/OctreeNodePredicate.h"
#include "Mode/BachelorMode.h"

UPointCloudScan::UPointCloudScan()
{
  PrimaryComponentTick.bCanEverTick = true;
  octree = CreateDefaultSubobject<UOctree>(TEXT("Octree"));
  segmentation = CreateDefaultSubobject<USegmentation>(TEXT("Segmentation"));
  segmentation->parentScan = this;
}

void UPointCloudScan::BeginDestroy()
{
  Super::BeginDestroy();
  if (GetPointCloud() != nullptr)
  {
    GetPointCloud()->ConditionalBeginDestroy();
    SetPointCloud(nullptr);
  }
  if (cogsPointCloud != nullptr)
  {
    cogsPointCloud->ConditionalBeginDestroy();
    cogsPointCloud = nullptr;
  }
  if (octree != nullptr)
  {
    octree->ConditionalBeginDestroy();
    octree = nullptr;
  }
  if (segmentation != nullptr) {
    segmentation->ConditionalBeginDestroy();
    segmentation = nullptr;
  }
}

void UPointCloudScan::TickComponent(float DeltaTime, ELevelTick TickType, FActorComponentTickFunction *ThisTickFunction)
{
  Super::TickComponent(DeltaTime, TickType, ThisTickFunction);
  octree->Draw();
}

void UPointCloudScan::LoadPointCloudData()
{
  if (GetPointCloud() == nullptr) { return; }
  bDisplayingIntensities = false;
  bDisplayingNormals = false;
  ResetRelativeTransform();
  selectionIndices.Empty();
  octree->Initialize(this);
}

void UPointCloudScan::ResetPointCloudData()
{
  if (GetPointCloud() == nullptr) { return; }
  DeselectAll();
  for (int32 i = 0; i < NumberOfPoints(); i++)
  {
    GetPointCloud()->GetPointCloudData()[i].SetEnabled(true);
  }
  GetPointCloud()->Rebuild(true);
}

void UPointCloudScan::Translate(FVector offset)
{
  FTransform transform = GetRelativeTransform();
  transform.SetTranslation(transform.GetTranslation() + offset);
  SetRelativeTransform(transform);
}

void UPointCloudScan::Rotate(FRotator rotation)
{
  FTransform transform = GetRelativeTransform();
  transform.SetRotation(transform.GetRotation() * FQuat(rotation));
  SetRelativeTransform(transform);
}

void UPointCloudScan::Scale(FVector scale)
{
  FTransform transform = GetRelativeTransform();
  transform.SetScale3D(transform.GetScale3D() * scale);
  SetRelativeTransform(transform);
}

void UPointCloudScan::SetColor(FColor color)
{
  if (GetPointCloud() == nullptr) { return; }
  GetPointCloud()->Color = color;
  GetPointCloud()->Rebuild(true);
}

void UPointCloudScan::DeletePointsSelected()
{
  if (GetPointCloud() == nullptr) { return; }
  for (int32 &i : selectionIndices)
  {
    GetPointCloud()->GetPointCloudData()[i].SetEnabled(false, true);
  }
  GetPointCloud()->Rebuild(true);
}

void UPointCloudScan::ColorPoints(TSet<int32> &indices, FColor color)
{
  if (GetPointCloud() == nullptr) { return; }
  for (int32 &i : indices)
  {
    if (i >= 0 && i < NumberOfPoints())
    {
      GetPointCloud()->GetPointCloudData()[i].Color = color;
    }
  }
  GetPointCloud()->Rebuild(true);
}

void UPointCloudScan::ColorPoints(FColor color, bool bRemainSelected)
{
  if (GetPointCloud() == nullptr) { return; }
  for (int32 i = 0; i < NumberOfPoints(); i++)
  {
    if (bRemainSelected && selectionIndices.Contains(i)) 
    {
      continue;
    }
    GetPointCloud()->GetPointCloudData()[i].Color = color;
  }
  GetPointCloud()->Rebuild(true);
  bDisplayingIntensities = false;
  bDisplayingNormals = false;
}

bool UPointCloudScan::ColorIntensities(bool bRemainSelected)
{
  if (GetPointCloud() == nullptr || cogsPointCloud == nullptr)
  {
    return false;
  }
  TArray<float> intensities = cogsPointCloud->GetIntensities();
  if (NumberOfPoints() != intensities.Num())
  {
    return false;
  }
  for (int32 i = 0; i < NumberOfPoints(); i++)
  {
    if (bRemainSelected && selectionIndices.Contains(i)) 
    {
      continue;
    }
    float intensity = intensities[i];
    GetPointCloud()->GetPointCloudData()[i].Color = FColor(intensity, intensity, intensity);
  }
  GetPointCloud()->Rebuild(true);
  bDisplayingIntensities = true;
  bDisplayingNormals = false;
  return true;
}

bool UPointCloudScan::ColorNormals(bool bRemainSelected)
{
  if (GetPointCloud() == nullptr || cogsPointCloud == nullptr)
  {
    return false;
  }
  TArray<FVector> normals = cogsPointCloud->GetNormals();
  if (NumberOfPoints() != normals.Num())
  {
    return false;
  }
  for (int32 i = 0; i < NumberOfPoints(); i++)
  {
    if (bRemainSelected && selectionIndices.Contains(i))
    {
      continue;
    }
    FVector normal = normals[i];
    GetPointCloud()->GetPointCloudData()[i].Color = FColor(normal.X, normal.Y, normal.Z);
  }
  GetPointCloud()->Rebuild(true);
  bDisplayingNormals = true;
  bDisplayingIntensities = false;
  return true;
}

void UPointCloudScan::DeselectAll()
{
  if (GetPointCloud() == nullptr || selectionIndices.Num() == 0) { return; }
  if (bDisplayingIntensities)
  {
    ColorIntensities();
  }
  else if (bDisplayingNormals)
  {
    ColorNormals();
  }
  else
  {
    ColorPoints(selectionIndices, FColor::White);
  }
  selectionIndices.Empty();
}

void UPointCloudScan::DeselectPoints(TSet<int32> &indices)
{
  if (GetPointCloud() == nullptr || selectionIndices.Num() == 0) 
  { 
    return; 
  }
  for (int32 &index : indices)
  {
    selectionIndices.Remove(index);
  }
  if (bDisplayingIntensities)
  {
    ColorIntensities(true);
  }
  else if (bDisplayingNormals)
  {
    ColorNormals(true);
  }
  else
  {
    ColorPoints(indices, FColor::White);
  }
}

void UPointCloudScan::SelectPoints(TSet<int32> &indices)
{
  selectionIndices.Append(indices);
  ColorPoints(indices, FColor::Orange);
}

TSet<int32> UPointCloudScan::LocationNeighborhood(FVector &location, float distance) const
{
  if (GetPointCloud() == nullptr) { return TSet<int32>(); }
  TSet<int32> indices;

  TArray<UOctreeNode*> queue;
  queue.Add(octree->GetRoot());
  while (queue.Num() > 0) {
    UOctreeNode *current = queue.Pop(false);
    if (current->IsLeaf())
    {
      for (const uint32_t &pointIndex : current->pointsIndices)
      {
        FVector pointLocation = TransformedPosition(pointIndex);
        if (FVector::Dist(pointLocation, location) <= distance)
        {
          indices.Add(pointIndex);
        }
      }
    }
    else
    {
      for (UOctreeNode *node : current->childNodes)
      {
        if (node->MinimalDistanceTo(location) < distance)
        {
          queue.Add(node);
        }
      }
    }
  }
  return indices;
}

TSet<int32> UPointCloudScan::LocationNeighborhoodIterative(FVector &location, float distance) const
{
  if (GetPointCloud() == nullptr) return TSet<int32>();
  TSet<int32> indices;
  for (int32 i = 0; i < NumberOfPoints(false); i++)
  {
    FVector otherLocation = GetPointCloud()->GetPointCloudData()[i].Location;
    if (FVector::Dist(location, otherLocation) < distance)
    {
      indices.Add(i);
    }
  }
  return indices;
}

TSet<int32> UPointCloudScan::PointNeighborhood(int32 pointNumber, float distance) const
{
  if (GetPointCloud() == nullptr || pointNumber < 0 || pointNumber >= NumberOfPoints())
  {
    return TSet<int32>();
  }
  return LocationNeighborhood(GetPointCloud()->GetPointCloudData()[pointNumber].Location, distance);
}

int32 UPointCloudScan::FindPointIndex(const FVector &location) const
{
  if (GetPointCloud() == nullptr) { return -1; }
  int32 index = 0;
  float min = INFINITY;
  for (int32 i = 0; i < NumberOfPoints(); i++)
  {
    FVector oLoc = TransformedPosition(i);
    float distance = FVector::Dist(location, oLoc);
    if (distance < min)
    {
      min = distance;
      index = i;
    }
  }
  return index;
}

float UPointCloudScan::ScanMaxExtent() const
{
  if (GetPointCloud() == nullptr) { return 0; }
  float res = -INFINITY;
  for (FPointCloudPoint const &p : GetPointCloud()->GetPointCloudData())
  {
    FVector const &location = TransformedPosition(p.Location);
    if (location.X > res) { res = location.X; }
    if (location.Y > res) { res = location.Y; }
    if (location.Z > res) { res = location.Z; }
  }
  return res;
}

float UPointCloudScan::ScanMinExtent() const
{
  if (GetPointCloud() == nullptr) { return 0; }
  float res = INFINITY;
  for (FPointCloudPoint const &p : GetPointCloud()->GetPointCloudData())
  {
    FVector const &location = TransformedPosition(p.Location);
    if (location.X < res) { res = location.X; }
    if (location.Y < res) { res = location.Y; }
    if (location.Z < res) { res = location.Z; }
  }
  return res;
}

bool UPointCloudScan::Import(FString fileName)
{
  if (cogsPointCloud != nullptr)
  {
    cogsPointCloud->ConditionalBeginDestroy();
    cogsPointCloud = nullptr;
  }
  FString extension = UPointCloudLoader::GetExtension(fileName);
  if (extension.IsEmpty())
  {
    return false;
  }
  if (extension.Equals(".xyz") || extension.Equals(".txt"))
  {
    SetPointCloud(UPointCloudLoader::Load(fileName, Cast<ABachelorMode>(GetWorld()->GetAuthGameMode())));
  }
  else if (extension.Equals(".cogs"))
  {
    cogsPointCloud = UCOGSPointCloud::Load(fileName);
    SetPointCloud(cogsPointCloud->Convert(this));
  }
  LoadPointCloudData();
  return true;
}

bool UPointCloudScan::Export()
{

  if (GetPointCloud() != nullptr)
  {
    FTransform transform = GetRelativeTransform();
    ResetRelativeTransform();
    if (cogsPointCloud != nullptr)
    {
      cogsPointCloud->Save(this);
    }
    else
    {
      UPointCloudLoader::Save(this);
    }
    SetRelativeTransform(transform);
    return true;
  }
  return false;
}

void  UPointCloudScan::ApplyTransformation()
{
  if (GetPointCloud() == nullptr) { return; }
  TArray<FPointCloudPoint> newPoints = TransformedPositions();
  GetPointCloud()->SetPointCloudData(newPoints);
  ResetRelativeTransform();
  octree->Initialize(this);
}

TArray<FPointCloudPoint> UPointCloudScan::TransformedPositions() const
{
  TArray<FPointCloudPoint> newPoints;
  if (GetPointCloud() == nullptr)
  {
    return newPoints;
  }

  for (FPointCloudPoint p : GetPointCloud()->GetPointCloudData())
  {
    p.OriginalLocation = TransformedPosition(p.Location) / 100;
    p.OriginalLocation.X = -p.OriginalLocation.X;
    newPoints.Add(p);
  }

  return newPoints;
}

FVector UPointCloudScan::TransformedPosition(int32 pointIndex) const
{
  if (GetPointCloud() == nullptr || pointIndex < 0 || pointIndex >= NumberOfPoints())
  {
    return FVector::ZeroVector;
  }
  return TransformedPosition(GetPointCloud()->GetPointCloudData()[pointIndex].Location);
}

FVector UPointCloudScan::TransformedPosition(const FVector &originalPosition) const
{
  FTransform transform = GetRelativeTransform();
  return transform.TransformPosition(originalPosition);
}

int32 UPointCloudScan::NumberOfPoints(bool bCountOnlyEnabled) const
{
  if (GetPointCloud() == nullptr) { return 0; }
  return GetPointCloud()->GetPointCount(bCountOnlyEnabled);
}

void UPointCloudScan::Segmentation()
{
  segmentation->Make();
}
