// @ Luk Gajdoech 2019

#pragma once

#include "CoreMinimal.h"
#include "UObject/NoExportTypes.h"
#include "Octree.generated.h"

class UOctreeNode;

/**
 * \brief Octree, should be used as a component of the APointCloudScan.
 */
UCLASS()
class BACHELORPROTOTYPE_API UOctree : public UObject
{
  GENERATED_BODY()

public:

  /**
   * Toggles the rendering of the octree nodes.
   */
  UFUNCTION(BlueprintCallable, Category = "Octree Function")
  void ToggleVisibility() { bDraw = !bDraw; }

  /**
   * Must be called after instantiation to properly initialize this octree over a given APointCloudScan
   * @param scan Octree will be built over this scan.
   */
  UFUNCTION(BlueprintCallable, Category = "Octree Function")
  void Initialize(class UPointCloudScan *scan);

  /**
   * Called every tick by the parent APointCloudScan, draws the octree node when bDraw is set to true.
   */
  UFUNCTION(BlueprintCallable, Category = "Octree Function")
  void Draw() const;

  /**
   * Return number of nodes of this octree.
   */
  UFUNCTION(BlueprintCallable, Category = "Octree Function")
  int32 NodeCount() const;

  /**
   * Return the closest node of this octree intersected by the ray.
   * @param rayOrigin Vector location of the origin of the ray, in the local coordinates of the point cloud.
   * @param rayDirection Normalized directional vector of the ray, in the local coordinates of the point cloud.
   */
  UOctreeNode *ClosestIntersectedNode(FVector rayOrigin, FVector rayDirection);

  /**
   * Calls the recursive finding process on the root node of this octree, returning the leaf node containing the given location, or nullptr, of the location is not within this octree.
   * @param location Vector location in the local coordinates of the point cloud.
   */
  UOctreeNode *FindNode(FVector location);

  /**
   * Static function, returns the node from the given array, closest to the location of the ray origin.
   * @param rayOrigin Vector location of the origin of the ray, in the local coordinates of the point cloud.
   * @param nodes Reference to the array of nodes which will be searched.
   */
  static UOctreeNode *FindClosestNode(FVector rayOrigin, TArray<UOctreeNode *> &nodes);

  /**
   * Gather the leaf nodes of this octree into the given list, intersected by the ray.
   * @param rayOrigin Vector location of the origin of the ray, in the local coordinates of the point cloud.
   * @param rayDirection Normalized directional vector of the ray, in the local coordinates of the point cloud.
   * @param nodes Reference to the array, to which the intersected nodes will be gathered.
   */
  void GatherIntersectedNodes(FVector rayOrigin, FVector rayDirection, TArray<UOctreeNode *> &nodes);

  /**
   * Returns the reference to the root of this octree.
   */
  class UOctreeNode *GetRoot();

  /**
   * Radius of the sphere which represents a point of the scan.
   */
  float sphereRadius;

  /**
   * Pointer to the scan, over which is this octree built.
   */
  UPROPERTY()
  class UPointCloudScan *parentScan;

private:

  bool bDraw = false;

  bool bInitialized = false;

  UPROPERTY()
  class UOctreeNode *root;

  float OptimalRadius(float neighborDistance);

};


