// @ Luk Gajdoech 2019

#pragma once

#include "CoreMinimal.h"
#include "Input.h"
#include "GameFramework/Pawn.h"
#include "User.generated.h"

/**
 * \brief Representation of the user.
 */
UCLASS()
class BACHELORPROTOTYPE_API AUser : public APawn
{
  GENERATED_BODY()

public:

  /**
   * Constructor managing the following actions:
   * - sets this class to accept the input from the user
   * - creates the root scene component for this class
   * - setups the only scene camera
   * - creates and setups the motion controllers
   * - setups the interaction component, which handles the interaction with in-world placed GUI
   * - creates the widget panel attached to the left controller
   * - instantiates the UInput class
   * - instantiates the USettings class for graphics settings of the application
   */
  AUser();

  /**
   * Overriden virtual method of the parent APawn classed, called at the start of the program. Gets the reference to the instances of PointCloudEdit and BachelorMode classes.
   */
  virtual void BeginPlay() override;
  
  /**
   * Overriden virtual method of the parent APawn classed, called internally every tick of the program. Calls the Update() method of the Input class and handles the movement of the user.
   * @param DeltaTime number of second since the last tick.
   */
  virtual void Tick(float DeltaTime) override;

  /**
   * Overriden virtual method of the parent APawn classed, called at the start of the program, binding the user actions.
   */
  virtual void SetupPlayerInputComponent(class UInputComponent *PlayerInputComponent) override;

  /**
   * Called by the Input class when the selection sequence is triggered using the Ray selection mode. 
   * @param rayOrigin Vector of the origin of the ray, passed as a const reference to safe memory.
   * @param rayEnd Normalized directional vector of the ray, passed as a const reference to safe memory.
   * @param bSelect Boolean switch, deciding whether the intersected points should be added or erased from the selection.
   * @param bDrawDebugLine Whether a debug line of the ray should be rendered.
   */
  void RayInteraction(const FVector &rayOrigin, const FVector &rayEnd, bool bSelect, bool bDrawDebugLine = false);

  /**
   * Called by the Input class when the selection sequence is triggered using the Sphere volume selection mode.
   * @param sphereLocation Location of the sphere, which is usually the location of the motion controller, passed as a const reference to safe memory.
   * @param bSelect Boolean switch, deciding whether the intersected points should be added or erased from the selection.
   */
  void SphereInteraction(const FVector &sphereLocation, bool bSelect);

  /**
   * Resets the position and the camera of the user.
   */
  UFUNCTION(BluePrintCallable, Category = "User Functions")
  void ResetPositionAndCamera();

  /**
   * Gets the pointer to the settings object, managing the graphics settings of the application.
   */
  UFUNCTION(BluePrintPure, Category = "User Functions")
  class USettings *GetSettings();

  /**
   * Gets the pointer to the input object.
   */
  UFUNCTION(BluePrintPure, Category = "User Functions")
  class UInput *GetInput();

  /**
   * Gets the pointer to left controller object.
   */
  class UMotionControllerComponent *GetLeftController();

  /**
   * Gets the pointer to right controller object
   */
  class UMotionControllerComponent *GetRightController();

private:
  FVector LocalPointCloudPosition(const FVector &worldPosition) const;

  void SetupMotionControllers();

  void SetupWidget();

  FORCEINLINE	void Move(float axisValue) { currentVelocity.X = axisValue * 2; };

  FORCEINLINE	void Side(float axisValue) { currentVelocity.Y = axisValue * 2; };

  FORCEINLINE	void Yaw(float axisValue) { currentRotation.Yaw = axisValue / 2; };

  FORCEINLINE	void Pitch(float axisValue) { currentRotation.Pitch = axisValue / 2; };

  void MouseLeftPressed();

  void MouseLeftReleased();

  void RightTriggerPressed();

  void RightTriggerReleased();

  void LeftTriggerPressed();

  void LeftTriggerReleased();

  FORCEINLINE void MouseWheel(float axisValue) { input->lastMouseWheel = axisValue; };

  FORCEINLINE void MouseRightPressed() { input->bMouseRightPressed = true; input->UpdateControllers(); };

  FORCEINLINE void MouseRightReleased() { input->bMouseRightPressed = false; };

  FORCEINLINE void MouseMiddlePressed() { input->bMouseMiddlePressed = true; input->UpdateControllers(); };

  FORCEINLINE void MouseMiddleReleased() { input->bMouseMiddlePressed = false; };

  FORCEINLINE void LeftShoulderPressed() { input->bLeftShoulderPressed = true; input->UpdateControllers(); };

  FORCEINLINE void LeftShoulderReleased() { input->bLeftShoulderPressed = false; };

  FORCEINLINE void RightShoulderPressed() { input->bRightShoulderPressed = true; input->UpdateControllers(); };

  FORCEINLINE void RightShoulderReleased() { input->bRightShoulderPressed = false; };

  void ResetTransform();

  FVector currentVelocity;

  FRotator currentRotation;

  UPROPERTY()
  class USettings *settings;

  UPROPERTY()
  class APointCloudEdit *pointCloud;

  UPROPERTY()
  UInput *input;

  UPROPERTY()
  class UWidgetInteractionComponent *interaction;

  UPROPERTY()
  class UWidgetComponent *widget;

  UPROPERTY()
  class UCameraComponent *camera;

  UPROPERTY()
  class UMotionControllerComponent *leftController;

  UPROPERTY()
  class UMotionControllerComponent *rightController;
};
