// @ Luk Gajdoech 2019

#pragma once

#include "CoreMinimal.h"
#include "PointCloudComponent.h"
#include "PointCloud.h"
#include "PointCloudScan.generated.h"

/**
 * \brief Single point cloud scan, should be used as a component of APointCloudEdit.
 */
UCLASS()
class BACHELORPROTOTYPE_API UPointCloudScan : public UPointCloudPluginComponent
{
  GENERATED_BODY()

public:

  /**
   * Initializes the UOctree and USegmentation subobjects.
   */
  UPointCloudScan();

  /**
   * Overriden destructor, releases octree, point cloud data and segmentation from the memory.
   */
  virtual void BeginDestroy() override;

  /**
   * Overriden method of the UActorComponent, called internally every tick of the program. Calls the Draw() method of UOctree, which renders the nodes if enabled.
   */
  virtual void TickComponent(float DeltaTime, ELevelTick TickType, FActorComponentTickFunction *ThisTickFunction) override;

  /**
   * Resets the point cloud data, deselecting all enabling all points.
   */
  void ResetPointCloudData();

  /**
   * Must be called after new point cloud data is loaded, resets the selection array and reinitializes the octree.
   */
  void LoadPointCloudData();

  /**
   * Apply a color filter over this scan. 
   * NOTE: this does not recolor the points themselves, @see ColorPoints(FColor color, bool bRemainSelected = false) for that functionality.
   * @param color Color to be applied.
   */
  void SetColor(FColor color);

  /**
   * Gets the maximal extent of this scan, which is the greatest coordinate from all 3 axes.
   */
  float ScanMaxExtent() const;

  /**
   * Gets the minimal extent of this scan, which is the smallest coordinate from all 3 axes.
   */
  float ScanMinExtent() const;

  /**
   * Returns the index of a point closest to the given location.
   * @param location Vector of location in local coordinates of the point cloud passed as a reference to safe memory.
   */
  int32 FindPointIndex(const FVector &location) const;

  /**
   * Returns a set of point indices of the cloud data of this scan within the given distance from the point at given index.
   * @param pointNumber Index of the point which surrounding will be searched.
   * @param distance Radius of the neighborhood.
   */
  TSet<int32> PointNeighborhood(int32 pointNumber, float distance) const;

  /**
   * Returns a set of point indices of the cloud data of this scan within the given distance from the given location.
   * NOTE: This implementation uses a custom NNS implementation using heap to effectively traverse the octree.
   * @param pointNumber Vector of location in local coordinate system of the point cloud which surrounding will be searched.
   * @param distance Radius of the neighborhood.
   */
  TSet<int32> LocationNeighborhood(FVector &location, float distance) const;

  /**
   * Returns a set of point indices of the cloud data of this scan within the given distance from the given location.
   * NOTE: This is a iterative implementation, which always traverse all points. @see TSet<int32> PointNeighborhood(int32 pointNumber, float distance) for a possibly better performing solution.
   * @param pointNumber Vector of location in local coordinate system of the point cloud which surrounding will be searched.
   * @param distance Radius of the neighborhood.
   */
  TSet<int32> LocationNeighborhoodIterative(FVector &location, float distance) const;

  /**
   * Makes the segmentation of the point cloud data assigned to this scan using the USegmentation subobject.
   */
  void Segmentation();

  /**
   * Visualize the intensity data of points as gray-scale colors.
   * NOTE: Available only for point clouds imported from COGS format, since XYZ files do not contain intensity data.
   * @param bRemainSelected Optinal parameter, whether the selected points should remain colored in orange.
   */
  bool ColorIntensities(bool bRemainSelected = false);

  /**
   * Visualize the normals data, which have been normalized into 0-255 range for RGB values.
   * NOTE: Available only for point clouds imported from COGS format, since XYZ files do not contain normals data.
   * @param bRemainSelected Optinal parameter, whether the selected points should remain colored in orange.
   */
  bool ColorNormals(bool bRemanSelected = false);

  /**
   * Color the set of points with given indices to the given color.
   * @param indices Indices of points to be recolored, passed as a reference to safe memory.
   * @param color New color for the points.
   */
  void ColorPoints(TSet<int32> &indices, FColor color);

  /**
   * Color all points of the point cloud data to the given color.
   * @param color New color for the points.
   * @param bRemainSelected Optinal parameter, whether the selected points should remain colored in orange.
   */
  void ColorPoints(FColor color, bool bRemainSelected = false);

  /**
   * Select the points with given indices, adding them to the selection and coloring them in orange.
   * @param indices Indices of points to be selected, passed as a reference to safe memory.
   */
  void SelectPoints(TSet<int32> &indices);

  /**
   * Deselect the points with given indices, removing them from selection and coloring them back to their original color.
   * @param indices Indices of points to be deselected, passed as a reference to safe memory.
   */
  void DeselectPoints(TSet<int32> &indices);

  /**
   * Deselect all points, effectively emptying the selection and coloring all points back to their original color.
   */
  void DeselectAll();

  /**
   * Returns array of points of the point cloud data contained within this scan with the shader transformations applied.
   */
  TArray<FPointCloudPoint> TransformedPositions() const;

  /**
   * Returns the vector location of the point at given index, with the shader transform applied.
   * @param pointIndex Index of the point of which location will be returned.
   */
  FVector TransformedPosition(int32 pointIndex) const;

  /**
   * Returns the vector location, with the shader transform applied.
   * @param originalPosition Position to be transformed by the shader transform, passed in as a reference to safe memory.
   */
  FVector TransformedPosition(const FVector &originalPosition) const;

  /**
   * Returns the total number of points of the point cloud data assigned to this scan.
   * @param bCountOnlyEnabled Optional parameter, whether the method should count only the enabled points.
   */
  UFUNCTION(BluePrintCallable, Category = "Scan Functions")
  int32 NumberOfPoints(bool bCountOnlyEnabled = false) const;

  /**
   * Delete the selected points of the point cloud data assigned to this scan.
   * NOTE: Actual deletion of the points from the structure would be destructive, slow and unreversable. Therefore the points are only disabled, but still remain in the memory.
   */
  UFUNCTION(BluePrintCallable, Category = "Scan Functions")
  void DeletePointsSelected();

  /**
   * Translate the points of the point cloud data assigned to this scan.
   * NOTE: This transformation occurs only on the shader level, without changing the actual coordinates. These are changed only during export or when @see ApplyTransformation() is called.
   * @param offset Translation vector to be applied.
   */
  UFUNCTION(BluePrintCallable, Category = "Scan Functions")
  void Translate(FVector offset);

  /**
   * Rotate the points of the point cloud data assigned to this scan.
   * NOTE: This transformation occurs only on the shader level, without changing the actual coordinates. These are changed only during export or when @see ApplyTransformation() is called.
   * @param rotation Instance of FRotator to be applied.
   */
  UFUNCTION(BluePrintCallable, Category = "Scan Functions")
  void Rotate(FRotator rotation);

  /**
   * Scale the point cloud data assigned to this scan.
   * NOTE: This transformation occurs only on the shader level, without changing the actual coordinates. These are changed only during export or when @see ApplyTransformation() is called.
   * @param scale Scale vector to be applied.
   */
  UFUNCTION(BluePrintCallable, Category = "Scan Functions")
  void Scale(FVector scale);

  /**
   * Apply the shader transformations, rewriting the coordinates of points of the cloud data assigned to this scan.
   * NOTE: This modification is permanent.
   */
  UFUNCTION(BluePrintCallable, Category = "Scan Functions")
  void ApplyTransformation();

  /**
   * Import the point cloud file located in the Content folder, while automatically deciding whether it is a COGS or XYZ file based on the file ending.
   */
  UFUNCTION(BluePrintCallable, Category = "Scan Functions")
  bool Import(FString fileName);

  /**
   * Export the point cloud data assigned to this scan, adding the EXPORT_ prefix to the original name of the data file.
   */
  UFUNCTION(BluePrintCallable, Category = "Scan Functions")
  bool Export();

  /**
   * Pointer to the UOctree subobject of this scan.
   */
  UPROPERTY(VisibleAnywhere, BlueprintReadOnly, Category = "Scan Properties")
  class UOctree *octree;

  /**
   * Pointer to the UCOGSPointCloud adapter, if this scan has a point cloud data imported from COGS file, nullptr otherwise.
   */
  UPROPERTY()
  class UCOGSPointCloud *cogsPointCloud;

  /**
  * Identification number of this UPointCloudScan instance.
  */
  int idNumber;

private:

  bool bDisplayingIntensities = false;

  bool bDisplayingNormals = false;

  UPROPERTY()
  TSet<int32> selectionIndices;

  UPROPERTY()
  class USegmentation *segmentation;

};
