// @ Luk Gajdoech 2019

#include "PointCloudEdit.h"
#include "PointCloudScan.h"
#include "PointCloud.h"
#include "ScanSelectionAnimator.h"
#include "Anchor.h"
#include "RenderMethod.h"

#include "Engine.h"
#include "Components/TextRenderComponent.h"

#include "Loaders/COGSPointCloud.h"
#include "Loaders/PointCloudLoader.h"
#include "Structure/Octree.h"
#include "Structure/OctreeNodePredicate.h"

const int UNDEFINED_INDEX = -1;
const float SELECTION_MULTIPLIER = 0.1f;

APointCloudEdit::APointCloudEdit()
{
  PrimaryActorTick.bCanEverTick = true;
  RootComponent = CreateDefaultSubobject<USceneComponent>(TEXT("Root"));
  animator = CreateDefaultSubobject<UScanSelectionAnimator>(TEXT("Animator"));
  animator->PointCloud = this;
  renderMethod = CreateDefaultSubobject<URenderMethod>(TEXT("RenderMethod"));
}

void APointCloudEdit::BeginPlay()
{
  Super::BeginPlay();
  AAnchor* anchor = GetWorld()->SpawnActor<AAnchor>();
  anchor->PointCloud = this;
}

void APointCloudEdit::Tick(float DeltaTime)
{
  Super::Tick(DeltaTime);
  animator->Animate(DeltaTime);
}

float APointCloudEdit::MinExtent() const
{
  float min = INFINITY;
  for (UPointCloudScan *scan : scans)
  {
    float scanMin = scan->ScanMinExtent();
    if (scanMin < min)
    {
      min = scanMin;
    }
  }
  return min;
}

float APointCloudEdit::MaxExtent() const
{
  float max = -INFINITY;
  for (UPointCloudScan *scan : scans)
  {
    float scanMax = scan->ScanMaxExtent();
    if (scanMax > max)
    {
      max = scanMax;
    }
  }
  return max;
}

void APointCloudEdit::CreateScanSlot()
{
  UPointCloudScan *scan = NewObject<UPointCloudScan>(this, UPointCloudScan::StaticClass());
  scan->RegisterComponent();
  scan->SetMobility(EComponentMobility::Movable);
  scan->AttachTo(RootComponent);
  scan->SetColor(FColor::White);
  scan->idNumber = scans.Num();
  scans.Add(scan);
}

int32 APointCloudEdit::NumberOfPoints()
{
  int32 number = 0;
  for (UPointCloudScan *scan : scans)
  {
    number += scan->NumberOfPoints();
  }
  return number;
}

int32 APointCloudEdit::NumberOfNodes()
{
  int32 number = 0;
  for (UPointCloudScan *scan : scans)
  {
    if (scan->octree != nullptr) {
      number += scan->octree->NodeCount();
    }
  }
  return number;
}

void APointCloudEdit::ColorsRandom()
{
  for (UPointCloudScan *scan : scans)
  {
    scan->ColorPoints(FColor::White, true);
    scan->SetColor(FColor::MakeRandomColor());
  }
}

void APointCloudEdit::ColorsWhite()
{
  for (UPointCloudScan *scan : scans)
  {
    scan->ColorPoints(FColor::White, true);
    scan->SetColor(FColor::White);
  }
}

bool APointCloudEdit::ColorsIntensities()
{
  for (UPointCloudScan *scan : scans)
  {
    if (scan->ColorIntensities(true) == false)
    {
      return false;
    }
  }
  return true;
}

bool APointCloudEdit::ColorsNormals()
{
  for (UPointCloudScan *scan : scans)
  {
    if (scan->ColorNormals(true) == false)
    {
      return false;
    }
  }
  return true;
}

void APointCloudEdit::ResetPointCloudData()
{
  for (UPointCloudScan *scan : scans)
  {
    scan->ResetPointCloudData();
  }
}

void APointCloudEdit::ResetTransform()
{
  SetActorScale3D(FVector(1));
  SetActorRotation(FRotator::ZeroRotator);
  SetActorLocation(FVector(-250, -500, 70));
}

void APointCloudEdit::DeselectAll()
{
  for (UPointCloudScan *scan : scans)
  {
    scan->DeselectAll();
  }
}

void APointCloudEdit::Segmentation()
{
  for (UPointCloudScan *scan : scans)
  {
    scan->Segmentation();
  }
}

void APointCloudEdit::ApplyTransformations()
{
  for (UPointCloudScan *scan : scans)
  {
    scan->ApplyTransformation();
  }
}

void APointCloudEdit::AutoAlign()
{
  if (scans.Num() > 0)
  {
    for (UPointCloudScan *scan : scans)
    {
      scan->ResetRelativeTransform();
    }
    FVector average = AveragePosition();
    for (UPointCloudScan *scan : scans)
    {
      scan->Translate(-average);
    }
  }
}

FVector APointCloudEdit::AveragePosition() const
{
  FVector average = FVector::ZeroVector;
  int32 count = 0;
  for (UPointCloudScan *scan : scans)
  {
    if (scan->GetPointCloud() != nullptr)
    {
      for (FPointCloudPoint const &p : scan->GetPointCloud()->GetPointCloudData())
      {
        average += p.Location;
      }
      count += scan->NumberOfPoints();
    }
  }
  average /= count;
  return average;
}

void APointCloudEdit::SplitScans()
{
  static bool toggle = false;
  if (toggle)
  {
    for (UPointCloudScan *s : scans)
    {
      s->ResetRelativeTransform();
    }
    toggle = false;
  }
  else
  {
    FRotator rot = FRotator(0, -(360 / scans.Num()), 0);
    FVector mov = FVector(0, 20, 0);
    for (UPointCloudScan *s : scans)
    {
      s->Translate(mov);
      mov = rot.RotateVector(mov);
    }
    toggle = true;
  }
}

TArray<FString> APointCloudEdit::ListScanFiles()
{
  TArray<FString> files;
  files.Append(UPointCloudLoader::ListFiles());
  files.Append(UCOGSPointCloud::ListFiles());
  return files;
}

bool APointCloudEdit::ImportFolder(FString folderDir)
{
  TArray<FString> files;
  IFileManager &fileManager = IFileManager::Get();
  FString path1 = FPaths::ProjectContentDir() + "COGSFiles/" + folderDir;
  FString path2 = FPaths::ProjectContentDir() + "XYZFiles/" + folderDir;
  FString fullPath = "";
  if (fileManager.DirectoryExists(*path1))
  {
    fullPath = path1;
  }
  else if (fileManager.DirectoryExists(*path2))
  {
    fullPath = path2;
  }
  if (!fullPath.Equals(""))
  {
    for (UPointCloudScan *s : scans)
    {
      s->ConditionalBeginDestroy();
    }
    scans.Empty();
    TArray<FString> files;
    fileManager.FindFiles(files, *fullPath);
    int scanIndex = 0;
    for (FString &file : files)
    {
      if (scanIndex >= scans.Num())
      {
        CreateScanSlot();
      }
      scans[scanIndex++]->Import(folderDir + "/" + file);
    }
    return true;
  }
  return false;
}

FVector APointCloudEdit::RayHit(FVector &rayOrigin, FVector &rayDirection)
{
  TArray<UOctreeNode *> nodeList;
  for (UPointCloudScan *scan : scans)
  {
    if (scan->IsVisible() && scan->GetPointCloud() != nullptr)
    {
      scan->octree->GatherIntersectedNodes(rayOrigin, rayDirection, nodeList);
    }
  }
  UOctreeNode *bestNode = nullptr;
  FVector pointLocation = FVector::ZeroVector;
  OctreeNodePredicate::referencePosition = rayOrigin;
  nodeList.Heapify(OctreeNodePredicate());
  while (nodeList.Num() != 0 && pointLocation.Equals(FVector::ZeroVector))
  {
    nodeList.HeapPop(bestNode, OctreeNodePredicate());
    pointLocation = bestNode->IntersectedPointLocation(rayOrigin, rayDirection);
  }
  return pointLocation;
}

bool APointCloudEdit::NeighborhoodInteraction(FVector &location, bool select)
{
  if (!location.Equals(FVector::ZeroVector))
  {
    for (UPointCloudScan *scan : scans)
    {
      if (scan->IsVisible())
      {
        float scaleFactor = 1 / GetActorScale3D().X;
        TSet<int32> selection = scan->LocationNeighborhood(location, selectionRadius * SELECTION_MULTIPLIER * scaleFactor);
        if (select)
        {
          scan->SelectPoints(selection);
        }
        else
        {
          scan->DeselectPoints(selection);
        }
      }
    }
    return true;
  }
  return false;
}

int32 APointCloudEdit::GetActiveScan()
{
  return activeScan;
}

void APointCloudEdit::SetActiveScan(int32 scan)
{
  if (scan != activeScan) 
  {
    if (scan >= 0 && scan < scans.Num())
    {
      activeScan = scan;
      animator->Initialize();
    }
  }
}

void APointCloudEdit::ApplyRenderMethod()
{
  if (renderMethod->size == 0) 
  {
    for (UPointCloudScan *scan : scans)
    {
      if (scan->GetPointCloud() != nullptr)
      {
        if (renderMethod->bLit)
        {
          scan->GetPointCloud()->SetRenderingMethod(EPointCloudRenderMethod::Point_Lit_RGB, false);
        }
        else
        {
          scan->GetPointCloud()->SetRenderingMethod(EPointCloudRenderMethod::Point_Unlit_RGB, false);
        }
        scan->GetPointCloud()->Rebuild(true);
      }
    }
  }
  else
  {
    for (UPointCloudScan *scan : scans)
    {
      if (scan->GetPointCloud() != nullptr)
      {
        if (renderMethod->bLit)
        {
          scan->GetPointCloud()->SetRenderingMethod(EPointCloudRenderMethod::Sprite_Lit_RGB, false);
        }
        else
        {
          scan->GetPointCloud()->SetRenderingMethod(EPointCloudRenderMethod::Sprite_Unlit_RGB, false);
        }
        scan->GetPointCloud()->SpriteSize = FVector2D(renderMethod->size/7.0f, renderMethod->size/7.0f);
        scan->GetPointCloud()->SpriteMask = EPointCloudSpriteMask::Circle;
        scan->GetPointCloud()->Rebuild(true);
      }
    }
  }

}


