// @ Luk Gajdoech 2019

#pragma once

#include "CoreMinimal.h"
#include "GameFramework/Actor.h"
#include "PointCloudEdit.generated.h"

/**
 * \brief Parent actor holding a list of UPointCloudScan instances.
 */
UCLASS()
class BACHELORPROTOTYPE_API APointCloudEdit : public AActor
{
  GENERATED_BODY()

public:

  /**
   * Initializes the root component of this actor, UScanSelectionAnimator and URenderMethod subobjects.
   */
  APointCloudEdit();

  /**
   * Overriden method of the AActor, internally called at the start of the program, spawns the AActor.
   */
  virtual void BeginPlay() override;

  /**
   * Overriden method of the AActor, internally called every tick of the program. Progress the animation, if the user changed the active scan.
   */
  virtual void Tick(float DeltaTime) override;

  /**
   * Returns the total number of octree nodes, summing the nodes of ocrees of all the scans. 
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  int32 NumberOfNodes();

  /**
   * Returns the total number of toints, summing the points of all cloud datas of all scans.
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  int32 NumberOfPoints();

  /**
   * Apply the current render method, must be called after the method is changed.
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  void ApplyRenderMethod();

  /**
   * Assign random color to each scan.
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  void ColorsRandom();

  /**
   * Color each scan in default white.
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  void ColorsWhite();

  /**
   * Visualize intensities on all scans.
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  bool ColorsIntensities();

  /**
   * Visualize normals on all scans.
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  bool ColorsNormals();

  /**
   * Reset point cloud data of all scans.
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  void ResetPointCloudData();

  /**
   * Reset the shader transform of this actor, which is applied by using motion controllers or mouse.
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  void ResetTransform();

  /**
   * Deselect all points from all scans.
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  void DeselectAll();

  /**
   * Run the segmentation on all scans.
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  void Segmentation();

  /**
   * Apply individual shader transformations on all scans.
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  void ApplyTransformations();

  /**
   * Create a new scan slot, effectively adding a new instance of UPointCloudScan to the array of scans.
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  void CreateScanSlot();

  /**
   * Align this object to the centre of the local coordinate system.
   * Performed by calculating the average of all points from all scans and offseting the object by the negative value of this vector, forcing the average to be zero.
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  void AutoAlign();

  /**
   * Tries to move the scan apart from each other, by calculating divergent offset vectors.
   * NOTE: Does not work optimaly for all point clouds, try the Owl dataset to see the optimal result
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  void SplitScans();

  /**
   * List all the scan file names from both the COGSFiles and XYZFiles folders.
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  TArray<FString> ListScanFiles();

  /**
   * Check if the subfolder exists within the XYZFiles or COGSFiles. If yes, import all the point cloud data files from it as single scans, deleting the previous scan slots.
   * @param folderDir Name of the folder to be imported.
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  bool ImportFolder(FString folderDir);

  /**
   * Returns the vector location of the first intersected point of all the scans in the local coordinate system.
   * @param rayOrigin Vector of the origin of the ray in the local coordinate system, passed as a const reference to safe memory.
   * @param rayDirection Normalized directional vector of the ray in the local coordinate system, passed as a const reference to safe memory.
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  FVector RayHit(FVector &rayOrigin, FVector &rayDirection);

  /**
   * Interacts with the neighborhood of the given location, either selecting or deselecting the contained points.
   * @param location Vector location in the local coordinate system to be queried.
   * @param bSelect Call with true to select points, false to deselect.
   */
  bool NeighborhoodInteraction(FVector &location, bool bSelect);

  /**
   * Returns the array index of the active scan.
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  int32 GetActiveScan();

  /**
   * Sets a new active scan.
   * @param scan Index of the new active scan.
   */
  UFUNCTION(BluePrintCallable, Category = "PointCloud Functions")
  void SetActiveScan(int32 scan);

  /**
   * Gets the minimal extent of the object, which is the smallest coordinate from all three axes of all the scans.
   */
  float MinExtent() const;

  /**
   * Gets the maximal extent of the object, which is the greatest coordinate from all three axes of all the scans.
   */
  float MaxExtent() const;

  /**
   * Array of scans of this object.
   */
  UPROPERTY(VisibleAnywhere, BluePrintReadOnly, Category = "PointCloud Properties")
  TArray<class UPointCloudScan *> scans;

  /**
   * Current selection radius.
   */
  UPROPERTY(EditAnywhere, BluePrintReadWrite, Category = "PointCloud Properties")
  int32 selectionRadius = 10; 

  /**
   * Current render method.
   */
  UPROPERTY(EditAnywhere, BluePrintReadWrite, Category = "PointCloud Properties")
  class URenderMethod *renderMethod;

private:

  UPROPERTY()
  class UScanSelectionAnimator* animator;

  int32 activeScan = 0;

  FVector AveragePosition() const;

};
