// @ Luk Gajdoech 2019

#include "User.h"
#include "Settings.h"

#include "Engine.h"
#include "ConstructorHelpers.h"
#include "Camera/CameraComponent.h"
#include "Components/StaticMeshComponent.h"
#include "Components/WidgetInteractionComponent.h"
#include "Components/WidgetComponent.h"
#include "Components/InputComponent.h"
#include "MotionControllerComponent.h"

#include "Data/PointCloudEdit.h"
#include "Data/PointCloudPreview.h"
#include "Mode/BachelorMode.h"

AUser::AUser()
{
  PrimaryActorTick.bCanEverTick = true;
  AutoPossessPlayer = EAutoReceiveInput::Player0;

  RootComponent = CreateDefaultSubobject<USceneComponent>(TEXT("RootComponent"));

  camera = CreateDefaultSubobject<UCameraComponent>(TEXT("Camera"));
  camera->SetupAttachment(RootComponent);

  SetupMotionControllers();

  interaction = CreateDefaultSubobject<UWidgetInteractionComponent>(TEXT("WidgetInteraction"));
  interaction->SetupAttachment(rightController);

  SetupWidget();

  settings = CreateDefaultSubobject<USettings>(TEXT("Settings"));
  input = CreateDefaultSubobject<UInput>(TEXT("Input"));
  input->SetUser(this);
}

void AUser::SetupWidget()
{
  ConstructorHelpers::FClassFinder<UUserWidget> widgetClass(TEXT("/Game/Blueprints/Hand_BP"));
  widget = CreateDefaultSubobject<UWidgetComponent>(TEXT("UserWidget"));
  widget->SetWidgetClass(widgetClass.Class);
  widget->SetupAttachment(leftController);
  widget->SetDrawSize(FVector2D(600, 950));
  widget->SetRelativeScale3D(FVector(0.2, 0.2, 0.2));
  widget->SetRelativeRotation(FRotator(90, 90, 0));
  widget->SetRelativeLocation(FVector(-70, 0, 0));
}

void AUser::SetupMotionControllers()
{
  USceneComponent *motionControllersOffset = CreateDefaultSubobject<USceneComponent>(TEXT("ControllerOffset"));
  motionControllersOffset->SetupAttachment(RootComponent);
  motionControllersOffset->AddLocalOffset(FVector(100, 0, 0));
  leftController = CreateDefaultSubobject<UMotionControllerComponent>(TEXT("LeftController"));
  rightController = CreateDefaultSubobject<UMotionControllerComponent>(TEXT("RightController"));
  leftController->SetupAttachment(motionControllersOffset);
  rightController->SetupAttachment(motionControllersOffset);
  rightController->SetTrackingSource(EControllerHand::Right);
  leftController->SetTrackingSource(EControllerHand::Left);

  ConstructorHelpers::FObjectFinder<UStaticMesh> controllerMesh(TEXT("/Engine/EditorMeshes/Axis_Guide"));
  leftController->bDisplayDeviceModel = true;
  rightController->bDisplayDeviceModel = true;
  leftController->SetDisplayModelSource(FName("Custom"));
  rightController->SetDisplayModelSource(FName("Custom"));
  leftController->SetCustomDisplayMesh(controllerMesh.Object);
  rightController->SetCustomDisplayMesh(controllerMesh.Object);
  leftController->SetRelativeScale3D(FVector(0.001f));
  rightController->SetRelativeScale3D(FVector(0.001f));
}

void AUser::SetupPlayerInputComponent(UInputComponent *PlayerInputComponent)
{
  Super::SetupPlayerInputComponent(PlayerInputComponent);

  InputComponent->BindAxis("Side", this, &AUser::Side);
  InputComponent->BindAxis("Move", this, &AUser::Move);
  InputComponent->BindAxis("Yaw", this, &AUser::Yaw);
  InputComponent->BindAxis("Pitch", this, &AUser::Pitch);
  InputComponent->BindAxis("MouseWheel", this, &AUser::MouseWheel);

  PlayerInputComponent->BindAction("MouseLeft", IE_Pressed, this, &AUser::MouseLeftPressed);
  PlayerInputComponent->BindAction("MouseLeft", IE_Released, this, &AUser::MouseLeftReleased);
  PlayerInputComponent->BindAction("MouseRight", IE_Pressed, this, &AUser::MouseRightPressed);
  PlayerInputComponent->BindAction("MouseRight", IE_Released, this, &AUser::MouseRightReleased);
  PlayerInputComponent->BindAction("MouseMiddle", IE_Pressed, this, &AUser::MouseMiddlePressed);
  PlayerInputComponent->BindAction("MouseMiddle", IE_Released, this, &AUser::MouseMiddleReleased);
  PlayerInputComponent->BindAction("LeftTrigger", IE_Pressed, this, &AUser::LeftTriggerPressed);
  PlayerInputComponent->BindAction("LeftTrigger", IE_Released, this, &AUser::LeftTriggerReleased);
  PlayerInputComponent->BindAction("LeftShoulder", IE_Pressed, this, &AUser::LeftShoulderPressed);
  PlayerInputComponent->BindAction("LeftShoulder", IE_Released, this, &AUser::LeftShoulderReleased);
  PlayerInputComponent->BindAction("RightTrigger", IE_Pressed, this, &AUser::RightTriggerPressed);
  PlayerInputComponent->BindAction("RightTrigger", IE_Released, this, &AUser::RightTriggerReleased);
  PlayerInputComponent->BindAction("RightShoulder", IE_Pressed, this, &AUser::RightShoulderPressed);
  PlayerInputComponent->BindAction("RightShoulder", IE_Released, this, &AUser::RightShoulderReleased);

  PlayerInputComponent->BindAction("ResetPosition", IE_Pressed, this, &AUser::ResetPositionAndCamera);
  PlayerInputComponent->BindAction("ResetTransform", IE_Pressed, this, &AUser::ResetTransform);
  PlayerInputComponent->BindAction("Panels", IE_Pressed, input, &UInput::Panels);

  APointCloudPreview* preview = *TActorIterator<APointCloudPreview>(GetWorld(), APointCloudPreview::StaticClass());
  if (preview != nullptr)
  {
    PlayerInputComponent->BindAction("Switch", IE_Pressed, preview, &APointCloudPreview::SwitchPointCloud);
  }
}

void AUser::ResetTransform()
{
  pointCloud->ResetTransform();
}

void AUser::MouseLeftPressed()
{
  input->bMouseLeftPressed = true;
  interaction->PressPointerKey(EKeys::LeftMouseButton);
  if (input->mode == EMode::Default) {
    pointCloud->DeselectAll();
  }
}

void AUser::MouseLeftReleased()
{
  input->bMouseLeftPressed = false;
  interaction->ReleasePointerKey(EKeys::LeftMouseButton);
}

void AUser::LeftTriggerPressed()
{
  input->bLeftTriggerPressed = true;
  input->UpdateControllers();
  if (input->mode == EMode::Default) {
    pointCloud->DeselectAll();
  }
};

void AUser::LeftTriggerReleased()
{
  input->bLeftTriggerPressed = false;
};

void AUser::RightTriggerPressed()
{
  interaction->PressPointerKey(EKeys::LeftMouseButton);
}

void AUser::RightTriggerReleased()
{
  interaction->ReleasePointerKey(EKeys::LeftMouseButton);
}

void AUser::BeginPlay()
{
  Super::BeginPlay();
  pointCloud = *TActorIterator<APointCloudEdit>(GetWorld(), APointCloudEdit::StaticClass());
}

void AUser::Tick(float DeltaTime)
{
  Super::Tick(DeltaTime);
  input->Update();
  if (!currentVelocity.IsZero())
  {
    AddActorLocalOffset(currentVelocity * DeltaTime * 40);
  }
  if (!currentRotation.IsZero())
  {
    AddActorLocalRotation(currentRotation * DeltaTime * 40);
  }
}

void AUser::RayInteraction(const FVector &rayOrigin, const FVector &rayEnd, bool select, bool bDrawDebugLine)
{
  FVector transformedRayOrigin = LocalPointCloudPosition(rayOrigin);
  FVector transformedRayEnd = LocalPointCloudPosition(rayEnd);
  FVector rayDirection = (transformedRayEnd - transformedRayOrigin);
  rayDirection.Normalize();
  FVector hitLocation = pointCloud->RayHit(transformedRayOrigin, rayDirection);
  pointCloud->NeighborhoodInteraction(hitLocation, select);
  if (bDrawDebugLine)
  {
    DrawDebugLine(GetWorld(), rayOrigin, rayEnd, FColor::Green, true, 10, 0, 1);
  }
}

void AUser::SphereInteraction(const FVector &sphereLocation, bool select)
{
  FVector transformedLocation = LocalPointCloudPosition(sphereLocation);
  pointCloud->NeighborhoodInteraction(transformedLocation, select);
}

void AUser::ResetPositionAndCamera()
{
  SetActorLocation(FVector(0, 0, 70));
  SetActorRotation(FRotator(0, -90, 0));
  camera->ResetRelativeTransform();
}

USettings * AUser::GetSettings()
{
  return settings;
}

UInput * AUser::GetInput()
{
  return input;
}

UMotionControllerComponent * AUser::GetLeftController()
{
  return leftController;
}

UMotionControllerComponent * AUser::GetRightController()
{
  return rightController;
}

FVector AUser::LocalPointCloudPosition(const FVector &worldPosition) const
{
  FTransform actorTransform = pointCloud->GetActorTransform();
  return actorTransform.InverseTransformPosition(worldPosition);
}