// Fill out your copyright notice in the Description page of Project Settings.

#include "PointCloudLoader.h"
#include "PointCloud.h"

#include "Engine.h"
#include "Core/Public/Misc/FileHelper.h"

#include "Data/PointCloudScan.h"
#include "Mode/BachelorMode.h"

FString UPointCloudLoader::GetExtension(FString fileName)
{
	int32 index;
	fileName.FindChar('.', index);
	if (index == INDEX_NONE) 
	{
		return "";
	}
	FString extension = "";
	for (index; index < fileName.Len(); index++)
	{
		extension += fileName[index];
	}
	return extension;
}

void UPointCloudLoader::Save(UPointCloudScan* scan)
{
	FString ExportPath = FPaths::ProjectContentDir() + "XYZFiles/" + "EXPORT_" + scan->GetPointCloud()->GetName();
	FString coords = "";
	for (FPointCloudPoint p : scan->TransformedPositions()) 
	{
		if (p.IsEnabled()) 
		{
      FVector location = p.OriginalLocation;
			FString x = FString::SanitizeFloat(location.X);
			FString y = FString::SanitizeFloat(location.Y);
			FString z = FString::SanitizeFloat(location.Z);
			if (x.Len() < 8 || (x.Len() < 9 && x[0] == '-')) {
				int count = 8 + (int)(x[0] == '-') - x.Len();
				for (int i = 0; i < count; i++) { x += "0"; }
			}
			if (y.Len() < 8 || (y.Len() < 9 && y[0] == '-')) {
				int count = 8 + (int)(y[0] == '-') - y.Len();
				for (int i = 0; i < count; i++) { y += "0"; }
			}
			if (z.Len() < 8 || (z.Len() < 9 && z[0] == '-')) {
				int count = 8 + (int)(z[0] == '-') - z.Len();
				for (int i = 0; i < count; i++) { z += "0"; }
			}
			coords += x + " " + y + " " + z + "\r\n";
		}
	}
	FFileHelper::SaveStringToFile(coords, *ExportPath);
}

UPointCloud* UPointCloudLoader::LoadFromFile(FString fileName, ABachelorMode* gameMode) 
{
	TArray<FPointCloudPoint> points;
	TArray<FString> coords;
	FString pathToFile = FPaths::ProjectContentDir() + "XYZFiles/" + fileName;
	FString input;
	FFileHelper::LoadFileToString(input, *pathToFile);
	FString line = "";
	for (char c: input) 
	{
		if (c == '\n') 
		{
			coords.Empty();
			line.ParseIntoArray(coords, TEXT(" "), true);
			FVector pos(FCString::Atof(*coords[0]), FCString::Atof(*coords[1]), FCString::Atof(*coords[2]));
			points.Emplace(pos, FColor::White, true);
			line = "";
		}
		line.AppendChar(c);
	}
	UPointCloud* pointCloud = NewObject<UPointCloud>();
	pointCloud->Rename(*fileName);
	pointCloud->SetPointCloudData(points);
	DefaultSettings(pointCloud);
	return pointCloud;
};

UPointCloud* UPointCloudLoader::LoadFromMemory(FString name, ABachelorMode* gameMode)
{
	for (UPointCloud* pointCloud : gameMode->preloadedPointClouds) 
	{
		if (pointCloud->GetName().Equals(name)) 
		{
			UPointCloud* newPointCloud = NewObject<UPointCloud>();
			newPointCloud->Rename(*name);
			newPointCloud->SetPointCloudData(pointCloud->GetPointCloudData());
			DefaultSettings(newPointCloud);
			UE_LOG(LogTemp, Log, TEXT("Loaded From Memory!"));
			return newPointCloud;
		}
	}
	return nullptr;
}

void UPointCloudLoader::DefaultSettings(UPointCloud* pointCloud)
{
	pointCloud->bEnableLOD = false;
	pointCloud->bDebugLOD = false;
	pointCloud->Offset = EPointCloudOffset::None;
	pointCloud->SetRenderingMethod(EPointCloudRenderMethod::Point_Unlit_RGB);
	pointCloud->Rebuild(true);
}

UPointCloud* UPointCloudLoader::Load(FString name, ABachelorMode* gameMode) 
{
	UPointCloud* pointcloud = LoadFromMemory(name, gameMode);
	if (pointcloud != nullptr) 
	{
		return pointcloud;
	} 
	return LoadFromFile(name, gameMode);
}

void UPointCloudLoader::Preload(ABachelorMode* gameMode) 
{
	gameMode->preloadedPointClouds.Empty();
	TArray<FString> files = ListFiles();
	for (FString fileName : files) 
	{
    if (GetExtension(fileName).Len() != 0) {
      UPointCloud* pointCloud = LoadFromFile(fileName, gameMode);
      gameMode->preloadedPointClouds.Add(pointCloud);
    }
	}
}

TArray<FString> UPointCloudLoader::ListFiles(bool debugPrint) 
{
	TArray<FString> files;
	IFileManager &fileManager = IFileManager::Get();
	FString pathToFolder = FPaths::ProjectContentDir() + "XYZFiles/";

  fileManager.FindFilesRecursive(files, *pathToFolder, TEXT("*"), true, true);
  TArray<FString> strippedFiles;
  for (FString fileName : files) {
    int index = fileName.Find("XYZFiles/");
    FString newFileName = fileName.RightChop(index + 9);
    strippedFiles.Add(newFileName);
  }

	if (debugPrint) 
	{
		UE_LOG(LogTemp, Log, TEXT("Number of Files: %d"), files.Num());
		for (FString fileName : files) 
		{
			UE_LOG(LogTemp, Log, TEXT("%s"), *fileName);
		}
	}
	return strippedFiles;
}