// Fill out your copyright notice in the Description page of Project Settings.

#include "PointCloudEdit.h"
#include "PointCloudComponent.h"
#include "EngineUtils.h"
#include "Engine/TextRenderActor.h"
#include "Components/TextRenderComponent.h"
#include "Components/StaticMeshComponent.h"
#include "ConstructorHelpers.h"
#include "Engine/GameEngine.h"

// Sets default values
APointCloudEdit::APointCloudEdit()
{
 	// Set this actor to call Tick() every frame.  You can turn this off to improve performance if you don't need it.
	PrimaryActorTick.bCanEverTick = true;

	root_ = CreateDefaultSubobject<USceneComponent>(TEXT("Root"));
	RootComponent = root_;

	ConstructorHelpers::FObjectFinder<UStaticMesh> cyl(TEXT("/Engine/BasicShapes/Cylinder"));
	collider_ = CreateAbstractDefaultSubobject<UStaticMeshComponent>(TEXT("Collider"));
	collider_->SetupAttachment(RootComponent);
	collider_->SetStaticMesh(cyl.Object);
	collider_->SetVisibility(false);

	ConstructorHelpers::FObjectFinder<UPointCloud> def(TEXT("/Game/PointClouds/default"));

	for (int i = 0; i < NUMBER_OF_SCANS; i++) {
		UPointCloud* pc = CreateDefaultSubobject<UPointCloud>(*FString("Pointcloud" + FString::FromInt(i)));
		pointclouds_.Add(pc);
		UPointCloudScan* scan = CreateDefaultSubobject<UPointCloudScan>(*FString("Scan" + FString::FromInt(i)));
		scan->SetupAttachment(RootComponent);
		scan->SetPointCloud(def.Object);
		scan->SetMobility(EComponentMobility::Movable);
		scans_.Add(scan);
	}
}

// Called when the game starts or when spawned
void APointCloudEdit::BeginPlay()
{
	Super::BeginPlay();
	for (int i = 0; i < NUMBER_OF_SCANS; i++) {
		scans_[i]->SetPointCloud(pointclouds_[i]);
		scans_[i]->SetColor(FColor::White);
	}
}

// Called every frame
void APointCloudEdit::Tick(float DeltaTime)
{
	Super::Tick(DeltaTime);
}

void APointCloudEdit::EndPlay(const EEndPlayReason::Type EndPlayReason)
{
	Super::EndPlay(EndPlayReason);
}


void APointCloudEdit::SetVisibility(bool value) {
	for (int i = 0; i < NUMBER_OF_SCANS; i++) {
		scans_[i]->SetVisibility(value);
	}
}

int32 APointCloudEdit::GetNumberOfPoints() {
	int32 number = 0;
	for (int i = 0; i < NUMBER_OF_SCANS; i++) {
		number += scans_[i]->GetScanNumberOfPoints();
	}
	GEngine->AddOnScreenDebugMessage(-1, 5.0f, FColor::Purple, TEXT("Total point count: " + FString::FromInt(number)));
	return number;
}

void APointCloudEdit::SetSpriteSize(float size) {
	for (int i = 0; i < NUMBER_OF_SCANS; i++) {
		scans_[i]->SetSpriteSize(size);
	}
}

void APointCloudEdit::GenerateColors() {
	FColor colors[3] = { FColor::Blue, FColor::Green, FColor::Red };
	for (int i = 0; i < NUMBER_OF_SCANS; i++) {
		scans_[i]->SetColor(colors[i]);
	}
}

void APointCloudEdit::ResetColors() {
	for (int i = 0; i < NUMBER_OF_SCANS; i++) {
		scans_[i]->SetColor(FColor::White);
	}
}

void APointCloudEdit::ResetPointCloudData() {
	for (int i = 0; i < NUMBER_OF_SCANS; i++) {
		scans_[i]->ResetPointCloudData();
	}
}

void APointCloudEdit::ToggleVisibility(int scan_number) {
	bool old = scans_[scan_number]->IsVisible();
	scans_[scan_number]->SetVisibility(!old);
}

void APointCloudEdit::FindScanColorPointNeighborhood(float x, float y, float z, FColor color) {
	int32* scan_indexes = new int32[NUMBER_OF_SCANS];
	for (int i = 0; i < NUMBER_OF_SCANS; i++) {
		scan_indexes[i] = scans_[i]->FindPointNumber(x, y, z);
	}
	FVector pLoc(x, y, z);
	int best_scan = 0;
	float min_distance = INFINITY;
	for (int i = 0; i < NUMBER_OF_SCANS; i++) {
		int32 point_index = scan_indexes[i];
		FVector oLoc = scans_[i]->GetPointCloud()->GetPointCloudData()[point_index].Location;
		if (FVector::Dist(pLoc, oLoc) < min_distance) {
			min_distance = FVector::Dist(pLoc, oLoc);
			best_scan = i;
		}
	}
	scans_[best_scan]->ColorPointNeighborhood(scan_indexes[best_scan], 10, color);
	UE_LOG(LogTemp, Log, TEXT("Best Scan: %d"), best_scan);
	GEngine->AddOnScreenDebugMessage(-1, 5.0f, FColor::Blue, TEXT("Best Scan: " + FString::FromInt(best_scan)));

	delete[] scan_indexes;
}

void APointCloudEdit::ExperimentalColors() {
	int32 quarter_num = scans_[0]->edit_points.Num() / 4;

	if (current_quarter == 0) {
		scans_[0]->SetColorOfPoints(3 * quarter_num, 4 * quarter_num, FColor::White);
	}
	else {
		scans_[0]->SetColorOfPoints((current_quarter - 1) * quarter_num, current_quarter * quarter_num, FColor::White);
	}

	scans_[0]->SetColorOfPoints(current_quarter * quarter_num, (current_quarter + 1) * quarter_num, FColor::Emerald);
	//scans_[0]->PrintAverageCoordsInSection(current_quarter * quarter_num, (current_quarter + 1) * quarter_num);

	if (++current_quarter > 3) {
		current_quarter = 0;
	}
}

void APointCloudEdit::IterateSpriteSizes() {
	if (sprite_size == 0.3f) {
		sprite_size = 1.0f;
	}
	else if (sprite_size == 1.0f) {
		sprite_size = 2.0f;
	} 
	else if (sprite_size == 2.0f) {
		sprite_size = 3.0f;
	}
	else if (sprite_size == 3.0f) {
		sprite_size = 0.3f;
	}

	SetSpriteSize(sprite_size);
}

void APointCloudEdit::CoolCompose() {
	int strenght = 20;
	FVector one = FVector(0, 1, 0);
	FVector two = FVector(0.77f, -0.33f, 0);
	FVector three = FVector(-0.77f, -0.33f, 0);

	one *= strenght;
	two *= strenght;
	three *= strenght;

	if (compose_flag)
	{
		scans_[0]->AddLocalOffset(-one);
		scans_[1]->AddLocalOffset(-two);
		scans_[2]->AddLocalOffset(-three);
		compose_flag = false;
	}
	else
	{
		scans_[0]->AddLocalOffset(one);
		scans_[1]->AddLocalOffset(two);
		scans_[2]->AddLocalOffset(three);
		compose_flag = true;
	}

}


