// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "GenericOctree.h"
#include "PointCloud.h"
#include "Components/ActorComponent.h"
#include "UObject/NoExportTypes.h"
#include "PointCloudOctreeCustom.generated.h"

USTRUCT(BlueprintType)
struct FOctreeElement
{
	GENERATED_BODY()

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Octree Element Struct")
	FPointCloudPoint Point;

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Octree Element Struct")
	FBoxSphereBounds BoxSphereBounds;

	FOctreeElement()
	{ }
};

struct FOctreeSematics
{
	enum { MaxElementsPerLeaf = 512 };
	enum { MinInclusiveElementsPerNode = 8 };
	enum { MaxNodeDepth = 6 };

	typedef TInlineAllocator<MaxElementsPerLeaf> ElementAllocator;

	/**
	* Get the bounding box of the provided octree element. In this case, the box
	* is merely the point specified by the element.
	*
	* @param	Element	Octree element to get the bounding box for
	*
	* @return	Bounding box of the provided octree element
	*/
	FORCEINLINE static FBoxSphereBounds GetBoundingBox(const FOctreeElement& Element)
	{
		return Element.BoxSphereBounds;
	}

	FORCEINLINE static bool AreElementsEqual(const FOctreeElement& A, const FOctreeElement& B)
	{
		return A.Point.Location == B.Point.Location;
	}

	FORCEINLINE static void SetElementId(const FOctreeElement& Element, FOctreeElementId Id)
	{ }

};

typedef TOctree<FOctreeElement, FOctreeSematics> FPCOctree;

UCLASS()
class BACHELORPROTOTYPE_API UPointCloudOctreeCustom : public UObject
{
	GENERATED_BODY()

public:
	// Sets default values for this component's properties
	UPointCloudOctreeCustom() : OctreeData(nullptr) { };

	/**
	* Used in conjunction with a constructor to initialize the object.
	* @param NewBounds	Intial size of the Octree
	* @param inDrawDebugInfo	Whether or not to display debug boundaries
	*/
	UFUNCTION(BlueprintCallable, Category = Octree)
	void Initialize(class UPointCloudScan* scan);

	/**
	* Adds a spefic element to the Octree
	* @param NewOctreeElement	FOctreeElement to be added.
	*/
	UFUNCTION(BlueprintCallable, Category = Octree)
	void AddOctreeElement(const FOctreeElement& inNewOctreeElement);

	/** Get number of nodes	*/
	UFUNCTION(BlueprintCallable, Category = Octree)
	int GetNodeCount();

	/**
	* Returns elements within the specified region.
	* @param inBoundingBoxQuery	Box to query Octree.
	* @return TArray of Elements
	*/
	UFUNCTION(BlueprintCallable, Category = Octree)
	TArray<FOctreeElement> GetElementsWithinBounds(const FBoxSphereBounds& inBoundingBoxQuery);

	/**
	* Returns elements within the specified region.
	* @param inBoundingBoxQuery	Box to query Octree.
	* @return TArray of Elements
	*/
	TArray<FOctreeElement> GetElementsWithinBounds(const FBoxCenterAndExtent& inBoundingBoxQuery);

	/** Draws Debug information at runtime */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = Debug)
	bool bDrawDebugInfo = false;

	void Draw();

	FPCOctree* OctreeData;

	UPROPERTY()
	class UPointCloudScan* scan;

	bool bInitialized;
};

