// Fill out your copyright notice in the Description page of Project Settings.

#include "ModelPC.h"
#include "PointCloudActor.h"
#include "PointCloudComponent.h"
#include "PointCloud.h"
#include "EngineUtils.h"
#include "User.h"
#include "Engine/TextRenderActor.h"
#include "Kismet/KismetMathLibrary.h"
#include "Components/TextRenderComponent.h"
#include "Blueprint/UserWidget.h"
#include "Components/Button.h"
#include "Components/TextBlock.h"
#include "ConstructorHelpers.h"

// Sets default values
AModelPC::AModelPC()
{
	// Set this actor to call Tick() every frame.  You can turn this off to improve performance if you don't need it.
	PrimaryActorTick.bCanEverTick = true;

	static ConstructorHelpers::FObjectFinder<UPointCloud> pc1_ob(TEXT("/Game/PointClouds/bottle"));
	static ConstructorHelpers::FClassFinder<UUserWidget> menu_ob(TEXT("/Game/Blueprints/Options_BP"));

	menu_class = menu_ob.Class;
	pc_component = CreateDefaultSubobject<UPointCloudComponent>(TEXT("PointCloudComponent"));
	pc_component->SetPointCloud(pc1_ob.Object);
	RootComponent = pc_component;
}

// Called when the game starts or when spawned
void AModelPC::BeginPlay()
{
	Super::BeginPlay();
	user = *TActorIterator<AUser>(GetWorld(), AUser::StaticClass());
	if (menu_class) {
		menu = CreateWidget<UUserWidget>(GetWorld(), menu_class);
	}
	LoadPointCloudsFromScene();
	SpawnTextLabel();
	LoadButtonsLabels();
	SetupButtons();
}

// Called every frame
void AModelPC::Tick(float DeltaTime)
{
	Super::Tick(DeltaTime);
	Show();
}

void AModelPC::SpawnTextLabel() {
	FVector MyLocation = GetActorLocation();
	FVector LabelLocation = MyLocation + FVector(0, 0, 60);
	text_label = GetWorld()->SpawnActor<ATextRenderActor>(ATextRenderActor::StaticClass(), LabelLocation, FRotator(0, 0, 0));
	text_label->GetTextRender()->SetHorizontalAlignment(EHTA_Center);
	text_label->GetTextRender()->SetText(GetClass()->GetName());
	text_label->GetTextRender()->SetTextRenderColor(FColor::White);
	text_label->SetActorScale3D(FVector(1, 1, 1));
	text_label->SetActorHiddenInGame(true);
}

void AModelPC::LoadPointCloudsFromScene() {
	UWorld* world = GetWorld();
	for (TActorIterator<APointCloudActor> It(world, APointCloudActor::StaticClass()); It; ++It) {
		APointCloudActor* pc_actor = *It;
		UPointCloudComponent* pc_comp = pc_actor->GetPointCloudComponent();
		UPointCloud* pc = pc_comp->GetPointCloud();
		if (pc_actor != nullptr) {
			pointclouds.Add(pc);
		}
	}
}

void AModelPC::SetPointCloudByIndex(int index) {
	if (index >= pointclouds.Num()) return;
	pc_index_ = index;
	UPointCloud* pc = pointclouds[index];
	pc_component->SetPointCloud(pc);
}

void AModelPC::SwitchPointCloud() {
	pc_index_++;
	if (pc_index_ >= pointclouds.Num()) {
		pc_index_ = 0;
	}
	if (pointclouds.Num() != 0 && pointclouds[pc_index_] != nullptr) {
		pc_component->SetPointCloud(pointclouds[pc_index_]);
	}
}

int AModelPC::FindPointCloudIndex(UPointCloud* pc) {
	int index = 0;
	while (index++ < pointclouds.Num()) {
		if (pointclouds[index] == pc) {
			break;
		}
	}
	return index;
}

void AModelPC::SetPointCloudByName(FString desired_name) {
	int i = 0;
	for (UPointCloud* pc : pointclouds) {
		FString name = pc->GetFName().ToString();
		if (name.Contains(desired_name)) {
			SetPointCloudByIndex(i);
			return;
		}
		i++;
	}
}

float AModelPC::DistanceFromUser() {
	if (user == nullptr) return INFINITY;

	FVector user_location = user->GetActorLocation();
	FVector my_location = GetActorLocation();
	FVector diff = user_location - my_location;
	float distance = 0;
	diff.ToDirectionAndLength(*(new FVector()), distance);
	return distance;
}

void AModelPC::LabelFaceUser() {
	if (user == nullptr) return;

	FVector user_location = user->GetActorLocation();
	FVector my_location = GetActorLocation();
	FRotator look_rotation = UKismetMathLibrary::FindLookAtRotation(user_location, my_location);
	look_rotation += FRotator(0, 180, 0);
	text_label->SetActorRotation(look_rotation);
}

void AModelPC::Show() {
	if (DistanceFromUser() < 400) {
		LabelFaceUser();
		text_label->SetActorHiddenInGame(false);
		if (menu != nullptr && !menu->IsInViewport()) 
			menu->AddToViewport();
	}
	else
	{
		text_label->SetActorHiddenInGame(true);
		if (menu != nullptr && menu->IsInViewport()) 
			menu->RemoveFromViewport();
	}
}

bool AModelPC::IsUserLooking() {
	FVector user_location = user->GetActorLocation();
	FVector my_location = GetActorLocation();

	FRotator user_rotation = user->GetActorRotation();
	FRotator look_rotation = UKismetMathLibrary::FindLookAtRotation(user_location, my_location);

	if (FMath::Abs(look_rotation.Yaw - user_rotation.Yaw) < 10) {
		return true;
	}

	return false;
}

void AModelPC::SetupButtons() {
	UButton* button = (UButton*)menu->GetWidgetFromName(TEXT("SwitchButton"));
	if (button != nullptr) {
		button->OnClicked.AddDynamic(this, &AModelPC::ButtonSwitch);
	}
	button = (UButton*)menu->GetWidgetFromName(TEXT("Button0"));
	if (button != nullptr) {
		button->OnClicked.AddDynamic(this, &AModelPC::Button0);
	}
	button = (UButton*)menu->GetWidgetFromName(TEXT("Button1"));
	if (button != nullptr) {
		button->OnClicked.AddDynamic(this, &AModelPC::Button1);
	}
	button = (UButton*)menu->GetWidgetFromName(TEXT("Button2"));
	if (button != nullptr) {
		button->OnClicked.AddDynamic(this, &AModelPC::Button2);
	}
}

void AModelPC::ButtonSwitch() {
	SwitchPointCloud();
}

void AModelPC::Button0() {
	SetPointCloudByIndex(0);
}

void AModelPC::Button1() {
	SetPointCloudByIndex(1);
}

void AModelPC::Button2() {
	SetPointCloudByIndex(2);
}

void AModelPC::LoadButtonsLabels() {
	if (menu == nullptr) return;
	for (int i = 0; i < 3; i++) {
		FString text = "Text_" + FString::FromInt(i);
		UTextBlock* label = Cast<UTextBlock>(menu->GetWidgetFromName(FName(*text)));
		if (label == nullptr) return;
		if (i >= pointclouds.Num()) {
			label->SetText(FText::FromString(""));
		}
		else {
			label->SetText(FText::FromString(pointclouds[i]->GetName()));
		}
	}
}