// Fill out your copyright notice in the Description page of Project Settings.

#include "PointCloudScan.h"
#include "PointCloudOctreeCustom.h"
#include "Engine/GameEngine.h"



UPointCloudScan::UPointCloudScan() {
	PrimaryComponentTick.bCanEverTick = true;
}

// Called every frame
void UPointCloudScan::TickComponent(float DeltaTime, ELevelTick TickType, FActorComponentTickFunction* ThisTickFunction)
{
	Super::TickComponent(DeltaTime, TickType, ThisTickFunction);
	octree->Draw();
}

void UPointCloudScan::BeginPlay()
{
	Super::BeginPlay();
	octree = NewObject<UPointCloudOctreeCustom>(this, UPointCloudOctreeCustom::StaticClass());
}

void UPointCloudScan::EndPlay(const EEndPlayReason::Type EndPlayReason)
{
	Super::EndPlay(EndPlayReason);
	ResetPointCloudData();
}

void UPointCloudScan::LoadPointCloudData() {
	original_points = GetPointCloud()->GetPointCloudData();
	edit_points = GetPointCloud()->GetPointCloudData();
	GetPointCloud()->SetPointCloudData(original_points);
}

void UPointCloudScan::DeletePoints(int32 from, int32 to) {
	if (original_points.Num() == 0) {
		LoadPointCloudData();
	}
	if (from < 0 || to >= edit_points.Num()) {
		return;
	}
	edit_points.RemoveAt(from, to - from, true);
	GetPointCloud()->SetPointCloudData(edit_points);
}

void UPointCloudScan::ResetPointCloudData() {
	if (original_points.Num() != 0) {
		edit_points = original_points;
		GetPointCloud()->SetPointCloudData(original_points);
	}
}

void UPointCloudScan::PrintAverageCoordsInSection(int32 from, int32 to) {
	if (from < 0 || to >= GetPointCloud()->GetPointCloudData().Num()) {
		return;
	}
	float sumX, sumY, sumZ;
	sumX = sumY = sumZ = 0;
	for (int32 i = from; i < to; i++) {
		sumX += GetPointCloud()->GetPointCloudData()[i].Location.X;
		sumY += GetPointCloud()->GetPointCloudData()[i].Location.Y;
		sumZ += GetPointCloud()->GetPointCloudData()[i].Location.Z;
	}
	UE_LOG(LogTemp, Log, TEXT("AvgX: %f, AvgY: %f, AvgZ: %f"), sumX / (to - from), sumY / (to - from), sumZ / (to - from));
}

void UPointCloudScan::PrintPointsCoords() {
	for (FPointCloudPoint p : GetPointCloud()->GetPointCloudData()) {
		UE_LOG(LogTemp, Log, TEXT("%s"), *p.ToString());
	}
	UE_LOG(LogTemp, Log, TEXT("TOTAL NUMBER OF POINTS: %d"), edit_points.Num());
}

void UPointCloudScan::SetSpriteSize(float size) {
	GetPointCloud()->SpriteSize = FVector2D(size, size);
}

void UPointCloudScan::SetColor(FColor color) {
	GetPointCloud()->Color = color;
	GetPointCloud()->Rebuild();
}

void UPointCloudScan::SetColorOfPoints(int32 from, int32 to, FColor color) {
	if (from < 0 || to >= edit_points.Num()) {
		return;
	}
	for (int32 i = from; i < to; i++) {
		edit_points[i].Color = color;
	}
	GetPointCloud()->SetPointCloudData(edit_points);
}

void UPointCloudScan::ColorPointNeighborhood(int32 point_number, int distance, FColor color) {
	if (original_points.Num() == 0) {
		LoadPointCloudData();
	}
	int32 n = edit_points.Num();
	FVector pLoc = edit_points[point_number].Location;
	FVector oLoc;
	for (int32 i = 0; i < n; i++) {
		oLoc = edit_points[i].Location;
		if (FVector::Dist(pLoc, oLoc) < distance) {
			edit_points[i].Color = color;
		}
	}
	GetPointCloud()->SetPointCloudData(edit_points);
}

int32 UPointCloudScan::FindPointNumber(float X, float Y, float Z) {
	float min = INFINITY;
	FVector fLoc = FVector(X, Y, Z);
	FVector oLoc;
	int32 number = 0;
	int32 n = GetPointCloud()->GetPointCloudData().Num();
	for (int32 i = 0; i < n; i++) {
		oLoc = GetPointCloud()->GetPointCloudData()[i].Location;
		if (FVector::Dist(fLoc, oLoc) < min) {
			min = FVector::Dist(fLoc, oLoc);
			number = i;
		}
	}
	return number;
}

void UPointCloudScan::SetRenderMethod(EPointCloudRenderMethod method) {
	GetPointCloud()->SetRenderingMethod(method);
	GetPointCloud()->Rebuild();
}

float UPointCloudScan::GetMaxExtent() {
	float res = -INFINITY;
	for (FPointCloudPoint p : GetPointCloud()->GetPointCloudData()) {
		if (p.Location.X > res) res = p.Location.X;
		if (p.Location.Y > res) res = p.Location.Y;
		if (p.Location.Z > res) res = p.Location.Z;
	}
	return res;
}

float UPointCloudScan::GetMinExtent() {
	float res = INFINITY;
	for (FPointCloudPoint p : GetPointCloud()->GetPointCloudData()) {
		if (p.Location.X < res) res = p.Location.X;
		if (p.Location.Y < res) res = p.Location.Y;
		if (p.Location.Z < res) res = p.Location.Z;
	}
	return res;
}

void UPointCloudScan::ChangeScanSourcePath(FString FileName) {
	FString SourcePath = FPaths::GameContentDir() + "XYZFiles/" + FileName;
	UE_LOG(LogTemp, Log, TEXT("%s"), *SourcePath);
	GEngine->AddOnScreenDebugMessage(-1, 5.0f, FColor::Green, TEXT("Loading Scan: " + FileName));
	GetPointCloud()->SetNewSourcePath(SourcePath, false);
	Reimport();
}

void UPointCloudScan::Reimport(){
	original_points.Empty();
	edit_points.Empty();
	GetPointCloud()->Reimport();
	GetPointCloud()->SetRenderingMethod(EPointCloudRenderMethod::Point_Lit_RGB);
	GetPointCloud()->Rebuild();
}

int32 UPointCloudScan::GetScanNumberOfPoints() {
	int32 res = GetPointCloud()->GetPointCount(true);
	FString print = GetName() + " point count: " + FString::FromInt(res);
	GEngine->AddOnScreenDebugMessage(-1, 5.0f, FColor::Yellow, TEXT(""+print));
	return res;
}

void UPointCloudScan::ToggleOctreeVisibility() {
	octree->bDrawDebugInfo = !octree->bDrawDebugInfo;
}