// Fill out your copyright notice in the Description page of Project Settings.

#include "User.h"
#include "Camera/CameraComponent.h"
#include "Components/StaticMeshComponent.h"
#include "Components/WidgetInteractionComponent.h"
#include "Components/InputComponent.h"
#include "Engine/GameEngine.h"
#include "PointCloudEdit.h"
#include "Engine.h"
#include "EngineUtils.h"
#include "ConstructorHelpers.h"

// Sets default values
AUser::AUser()
{
	// Set this pawn to call Tick() every frame.  You can turn this off to improve performance if you don't need it.
	PrimaryActorTick.bCanEverTick = true;

	// Set this pawn to be controlled by the lowest-numbered player
	AutoPossessPlayer = EAutoReceiveInput::Player0;

	RootComponent = CreateDefaultSubobject<USceneComponent>(TEXT("RootComponent"));

	camera = CreateDefaultSubobject<UCameraComponent>(TEXT("Camera"));
	camera->SetupAttachment(RootComponent);

	interaction = CreateDefaultSubobject<UWidgetInteractionComponent>(TEXT("WidgetInteraction"));
	interaction->SetupAttachment(RootComponent);

	UStaticMeshComponent* pointer = CreateDefaultSubobject<UStaticMeshComponent>(TEXT("Pointer"));
	static ConstructorHelpers::FObjectFinder<UStaticMesh> plane_mesh_ob(TEXT("/Engine/BasicShapes/Plane"));
	static ConstructorHelpers::FObjectFinder<UMaterial> material_ob(TEXT("/Engine/BasicShapes/BasicShapeMaterial"));

	pointer->SetStaticMesh(plane_mesh_ob.Object);
	pointer->SetMaterial(0, material_ob.Object);
	pointer->SetupAttachment(RootComponent);
	pointer->SetRelativeLocation(FVector(50, 0, 0));
	pointer->SetRelativeScale3D(FVector(0.005f, 0.005f, 0.005f));
	pointer->SetRelativeRotation(FRotator(90, 0, 0));
	pointer->SetCastShadow(false);
}

// Called when the game starts or when spawned
void AUser::BeginPlay()
{
	Super::BeginPlay();
}

// Called every frame
void AUser::Tick(float DeltaTime)
{
	Super::Tick(DeltaTime);

	/*
	if (!CurrentVelocity.IsZero())
	{
		FVector NewLocation = GetActorLocation() + (CurrentVelocity * DeltaTime);
		SetActorLocation(NewLocation);
	}
	*/

	if (!current_velocity_.IsNearlyZero()) {
		AddActorLocalOffset(current_velocity_, false);
	}
	if (!current_rotation_.IsNearlyZero()) {
		AddActorLocalRotation(current_rotation_, false);
	}
}

// Called to bind functionality to input
void AUser::SetupPlayerInputComponent(UInputComponent* PlayerInputComponent)
{
	Super::SetupPlayerInputComponent(PlayerInputComponent);
	// Respond every frame to the values of our two movement axes, "MoveX" and "MoveY".
	// PlayerInputComponent->BindAxis("MoveX", this, &AUser::Move_XAxis);
	// PlayerInputComponent->BindAxis("MoveY", this, &AUser::Move_YAxis);
	// PlayerInputComponent->BindAction("Rotate", IE_Pressed, this, &AUser::RotateModel);
	// PlayerInputComponent->BindAction("Rotate", IE_Repeat, this, &AUser::RotateModel);

	PlayerInputComponent->BindKey(EKeys::W, IE_Pressed, this, &AUser::MoveForward);
	PlayerInputComponent->BindKey(EKeys::S, IE_Pressed, this, &AUser::MoveBackward);
	PlayerInputComponent->BindKey(EKeys::A, IE_Pressed, this, &AUser::MoveLeft);
	PlayerInputComponent->BindKey(EKeys::D, IE_Pressed, this, &AUser::MoveRight);
	PlayerInputComponent->BindKey(EKeys::Q, IE_Pressed, this, &AUser::YawLeft);
	PlayerInputComponent->BindKey(EKeys::E, IE_Pressed, this, &AUser::YawRight);
	PlayerInputComponent->BindKey(EKeys::R, IE_Pressed, this, &AUser::PitchUp);
	PlayerInputComponent->BindKey(EKeys::F, IE_Pressed, this, &AUser::PitchDown);

	PlayerInputComponent->BindKey(EKeys::W, IE_Released, this, &AUser::StopMovement);
	PlayerInputComponent->BindKey(EKeys::S, IE_Released, this, &AUser::StopMovement);
	PlayerInputComponent->BindKey(EKeys::A, IE_Released, this, &AUser::StopMovement);
	PlayerInputComponent->BindKey(EKeys::D, IE_Released, this, &AUser::StopMovement);
	PlayerInputComponent->BindKey(EKeys::Q, IE_Released, this, &AUser::StopRotation);
	PlayerInputComponent->BindKey(EKeys::E, IE_Released, this, &AUser::StopRotation);
	PlayerInputComponent->BindKey(EKeys::R, IE_Released, this, &AUser::StopRotation);
	PlayerInputComponent->BindKey(EKeys::F, IE_Released, this, &AUser::StopRotation);

	PlayerInputComponent->BindKey(EKeys::LeftMouseButton, IE_Pressed, this, &AUser::LeftMouseButtonPressed);
	PlayerInputComponent->BindKey(EKeys::LeftMouseButton, IE_Released, this, &AUser::LeftMouseButtonReleased);
}

void AUser::LeftMouseButtonPressed() {
	interaction->PressPointerKey(EKeys::LeftMouseButton);
	TraceFromMouseCursor();
	
}

void AUser::TraceFromPointer() {
	FHitResult hit_result;
	FVector start_trace = camera->GetComponentLocation();
	FVector forward_vector = camera->GetForwardVector();
	FVector end_trace = ((forward_vector * 1000) + start_trace);
	FCollisionQueryParams CollisionParams;
	CollisionParams.AddIgnoredActor(this);

	if (GetWorld()->LineTraceSingleByChannel(hit_result, start_trace, end_trace, ECC_Visibility, CollisionParams))
	{
		UE_LOG(LogTemp, Log, TEXT("You hit: %s"), *hit_result.GetActor()->GetName());
		CheckPointCloudHit(hit_result);
	}
}

void AUser::TraceFromMouseCursor() {
	FHitResult hit_result;
	APlayerController* controller = GetWorld()->GetFirstPlayerController();
	controller->GetHitResultUnderCursor(ECC_Visibility, true, hit_result);

	if (hit_result.GetActor() != nullptr) 
	{
		UE_LOG(LogTemp, Log, TEXT("You hit: %s"), *hit_result.GetActor()->GetName());
		CheckPointCloudHit(hit_result);
	}

}

void AUser::CheckPointCloudHit(FHitResult hit_result) {
	if (hit_result.GetActor()->IsA(APointCloudEdit::StaticClass())) {
		DrawDebugLine(GetWorld(), hit_result.TraceStart, hit_result.TraceEnd, FColor::Green, true, 10, 0, 1);
		APointCloudEdit* pc = (APointCloudEdit*)hit_result.GetActor();
		FVector l = hit_result.ImpactPoint;
		UE_LOG(LogTemp, Log, TEXT("HitX: %f, HitY: %f, HitZ: %f"), l.X, l.Y, l.Z);
		FVector la = l - pc->GetActorLocation();
		FRotator r = pc->GetActorRotation().GetInverse();
		FMatrix rm = FRotationMatrix::Make(r);
		la = rm.TransformPosition(la);
		UE_LOG(LogTemp, Log, TEXT("HitAX: %f, HitAY: %f, HitAZ: %f"), la.X, la.Y, la.Z);
		pc->FindScanColorPointNeighborhood(la.X, la.Y, la.Z, FColor::Yellow);
	}
}

void AUser::LeftMouseButtonReleased() {
	interaction->ReleasePointerKey(EKeys::LeftMouseButton);
}


/*
void AUser::Move_XAxis(float AxisValue)
{
// Move at 100 units per second forward or backward
CurrentVelocity.X = FMath::Clamp(AxisValue, -1.0f, 1.0f) * 100.0f;
//UE_LOG(LogTemp, Log, TEXT("Hi"));
}

void AUser::Move_YAxis(float AxisValue)
{
// Move at 100 units per second right or left
CurrentVelocity.Y = FMath::Clamp(AxisValue, -1.0f, 1.0f) * 100.0f;
}
*/
