// Fill out your copyright notice in the Description page of Project Settings.


#include "PointCloudCustomOctree.h"
#include "DrawDebugHelpers.h"
#include "EngineUtils.h"
#include "Engine/World.h"


// Sets default values
APointCloudCustomOctree::APointCloudCustomOctree(): OctreeData(nullptr)
{
 	// Set this actor to call Tick() every frame.  You can turn this off to improve performance if you don't need it.
	PrimaryActorTick.bCanEverTick = true;

	static ConstructorHelpers::FClassFinder<AActor> actor_ob(TEXT("/Game/Blueprints/OctreeElement_BP"));
	ElementClass = actor_ob.Class;

	Root = CreateDefaultSubobject<USceneComponent>(TEXT("Root"));
	RootComponent = Root;
}

// Called when the game starts or when spawned
void APointCloudCustomOctree::BeginPlay()
{
	Super::BeginPlay();
	FVector dim1 = FVector(-100, -100, -100) + GetActorLocation();
	FVector dim2 = FVector(100, 100, 100) + GetActorLocation();
	FBox box(dim1, dim2);
	Initialize(box, true);

	for (int i = 0; i < 8; i++) {

		FVector randomLocation(FMath::RandRange(-100, 100), FMath::RandRange(-100, 100), FMath::RandRange(-100, 100));
		AActor* actor = GetWorld()->SpawnActor<AActor>(ElementClass, randomLocation + GetActorLocation(), FRotator(0, 0, 0));
		FBoxSphereBounds bounds = actor->GetComponentsBoundingBox();
		FTestOctreeElement element;
		element.MyActor = actor;
		element.BoxSphereBounds = bounds;
		AddOctreeElement(element);
	}

}

void APointCloudCustomOctree::Initialize(const FBox& inNewBounds, const bool& inDrawDebugInfo)
{
	bInitialized = true;
	bDrawDebugInfo = inDrawDebugInfo;
	OctreeData = new FSimpleOctree(inNewBounds.GetCenter(), inNewBounds.GetExtent().GetMax()); // const FVector & InOrigin, float InExtent
}

// Called every frame
void APointCloudCustomOctree::Tick(float DeltaTime)
{
	Super::Tick(DeltaTime);

	if (bInitialized && bDrawDebugInfo)
	{

		int nodeCount = 0;

		// Go through the nodes of the octree
		for (FSimpleOctree::TConstIterator<> NodeIt(*OctreeData); NodeIt.HasPendingNodes(); NodeIt.Advance())
		{
			const FSimpleOctree::FNode& CurrentNode = NodeIt.GetCurrentNode();
			const FOctreeNodeContext& CurrentContext = NodeIt.GetCurrentContext();
			const FBoxCenterAndExtent& CurrentBounds = CurrentContext.Bounds;

			nodeCount++;

			FOREACH_OCTREE_CHILD_NODE(ChildRef)
			{
				if (CurrentNode.HasChild(ChildRef))
				{
					NodeIt.PushChild(ChildRef);
				}
			}

			// Draw Node Bounds
			DrawDebugBox(GetWorld(), CurrentBounds.Center, CurrentBounds.Extent, FColor().Blue, false, 0.0f);
		}
		//UE_LOG(LogTemp, Log, TEXT("Node Count: %d"), nodeCount);
	}
}

void APointCloudCustomOctree::AddOctreeElement(const FTestOctreeElement& inNewOctreeElement)
{
	OctreeData->AddElement(inNewOctreeElement);
}

TArray<FTestOctreeElement> APointCloudCustomOctree::GetElementsWithinBounds(const FBoxSphereBounds& inBoundingBoxQuery)
{
	FBoxCenterAndExtent boundingBoxQuery = FBoxCenterAndExtent(inBoundingBoxQuery);
	return GetElementsWithinBounds(boundingBoxQuery);
}

TArray<FTestOctreeElement> APointCloudCustomOctree::GetElementsWithinBounds(const FBoxCenterAndExtent& inBoundingBoxQuery)
{
	// Iterating over a region in the octree and storing the elements
	int count = 0;
	TArray<FTestOctreeElement> octreeElements;

	for (FSimpleOctree::TConstElementBoxIterator<> OctreeIt(*OctreeData, inBoundingBoxQuery);
		OctreeIt.HasPendingElements();
		OctreeIt.Advance())
	{
		octreeElements.Add(OctreeIt.GetCurrentElement());
		count++;
	}
	// UE_LOG(LogTemp, Log, TEXT("%d elements in %s"), count, *boundingBoxQuery.Extent.ToString());

	return octreeElements;
}



