// Fill out your copyright notice in the Description page of Project Settings.

#include "PointCloudOctreeCustom.h"
#include "PointCloudScan.h"
#include "DrawDebugHelpers.h"
#include "Engine/GameEngine.h"

void UPointCloudOctreeCustom::Initialize(UPointCloudScan* scan)
{
	this->scan = scan;
	bInitialized = true;
	bDrawDebugInfo = true;
	float minExtent = scan->GetMinExtent();
	float maxExtent = scan->GetMaxExtent();

	FVector dim1 = FVector(minExtent, minExtent, minExtent) + scan->GetOwner()->GetActorLocation();
	FVector dim2 = FVector(maxExtent, maxExtent, maxExtent) + scan->GetOwner()->GetActorLocation();
	FBox box(dim1, dim2);
	OctreeData = new FPCOctree(box.GetCenter(), box.GetExtent().GetMax());

	for (FPointCloudPoint p: scan->GetPointCloud()->GetPointCloudData()) {
		FVector location = p.Location;
		FRotator r = scan->GetOwner()->GetActorRotation();
		FMatrix rm = FRotationMatrix::Make(r);
		location = rm.InverseTransformPosition(location);
		location += scan->GetOwner()->GetActorLocation();
		FBoxSphereBounds bounds(location, FVector(0.1, 0.1, 0.1), 0.1);
		FOctreeElement element;
		element.Point = p;
		element.BoxSphereBounds = bounds;
		AddOctreeElement(element);
	}
}

void UPointCloudOctreeCustom::Draw() {

	if (bInitialized && bDrawDebugInfo)
	{
		// Go through the nodes of the octree
		for (FPCOctree::TConstIterator<> NodeIt(*OctreeData); NodeIt.HasPendingNodes(); NodeIt.Advance())
		{
			const FPCOctree::FNode& CurrentNode = NodeIt.GetCurrentNode();
			const FOctreeNodeContext& CurrentContext = NodeIt.GetCurrentContext();
			const FBoxCenterAndExtent& CurrentBounds = CurrentContext.Bounds;

			FOREACH_OCTREE_CHILD_NODE(ChildRef)
			{
				if (CurrentNode.HasChild(ChildRef))
				{
					NodeIt.PushChild(ChildRef);
				}
			}

			// Draw Node Bounds
			// UE_LOG(LogTemp, Log, TEXT("Center: %s Extent: %s"), *CurrentBounds.Center.ToString(), *CurrentBounds.Extent.ToString());
			DrawDebugBox(GetWorld(), CurrentBounds.Center, CurrentBounds.Extent, FColor().Red, false, 0.0f);
		}
	}
}

void UPointCloudOctreeCustom::AddOctreeElement(const FOctreeElement& inNewOctreeElement)
{
	OctreeData->AddElement(inNewOctreeElement);
}

TArray<FOctreeElement> UPointCloudOctreeCustom::GetElementsWithinBounds(const FBoxSphereBounds& inBoundingBoxQuery)
{
	FBoxCenterAndExtent boundingBoxQuery = FBoxCenterAndExtent(inBoundingBoxQuery);
	return GetElementsWithinBounds(boundingBoxQuery);
}

TArray<FOctreeElement> UPointCloudOctreeCustom::GetElementsWithinBounds(const FBoxCenterAndExtent& inBoundingBoxQuery)
{
	// Iterating over a region in the octree and storing the elements
	int count = 0;
	TArray<FOctreeElement> octreeElements;

	for (FPCOctree::TConstElementBoxIterator<> OctreeIt(*OctreeData, inBoundingBoxQuery);
		OctreeIt.HasPendingElements();
		OctreeIt.Advance())
	{
		octreeElements.Add(OctreeIt.GetCurrentElement());
		count++;
	}
	// UE_LOG(LogTemp, Log, TEXT("%d elements in %s"), count, *boundingBoxQuery.Extent.ToString());

	return octreeElements;
}

int UPointCloudOctreeCustom::GetNodeCount() {
	int node_count = 0;
	if (bInitialized)
	{
		// Go through the nodes of the octree
		for (FPCOctree::TConstIterator<> NodeIt(*OctreeData); NodeIt.HasPendingNodes(); NodeIt.Advance())
		{
			const FPCOctree::FNode& CurrentNode = NodeIt.GetCurrentNode();
			const FOctreeNodeContext& CurrentContext = NodeIt.GetCurrentContext();
			const FBoxCenterAndExtent& CurrentBounds = CurrentContext.Bounds;
			node_count++;
			FOREACH_OCTREE_CHILD_NODE(ChildRef)
			{
				if (CurrentNode.HasChild(ChildRef))
				{
					NodeIt.PushChild(ChildRef);
				}
			}
		}
	}
	UE_LOG(LogTemp, Log, TEXT("Node Count: %d"), node_count);
	GEngine->AddOnScreenDebugMessage(-1, 5.0f, FColor::Cyan, TEXT("Octree node count: " + FString::FromInt(node_count)));
	return node_count;
}


