// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "GenericOctree.h"
#include "GameFramework/Actor.h"
#include "PointCloudCustomOctree.generated.h"

USTRUCT(BlueprintType)
struct FTestOctreeElement
{
	GENERATED_BODY()

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Octree Element Struct")
	AActor* MyActor;

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Octree Element Struct")
	FBoxSphereBounds BoxSphereBounds;

	FTestOctreeElement()
	{
		MyActor = nullptr;
	}
};

struct FTestOctreeSematics
{
	enum { MaxElementsPerLeaf = 2 }; // 16
	enum { MinInclusiveElementsPerNode = 7 };
	enum { MaxNodeDepth = 12 };

	typedef TInlineAllocator<MaxElementsPerLeaf> ElementAllocator;

	/**
	* Get the bounding box of the provided octree element. In this case, the box
	* is merely the point specified by the element.
	*
	* @param	Element	Octree element to get the bounding box for
	*
	* @return	Bounding box of the provided octree element
	*/
	FORCEINLINE static FBoxSphereBounds GetBoundingBox(const FTestOctreeElement& Element)
	{
		return Element.BoxSphereBounds;
	}

	FORCEINLINE static bool AreElementsEqual(const FTestOctreeElement& A, const FTestOctreeElement& B)
	{
		return A.MyActor == B.MyActor;
	}

	FORCEINLINE static void SetElementId(const FTestOctreeElement& Element, FOctreeElementId Id)
	{
	}

};

typedef TOctree<FTestOctreeElement, FTestOctreeSematics> FSimpleOctree;

UCLASS()
class BACHELORPROTOTYPE_API APointCloudCustomOctree : public AActor
{
	GENERATED_BODY()
	
public:	
	// Sets default values for this actor's properties
	APointCloudCustomOctree();

	/**
	* Used in conjunction with a constructor to initialize the object.
	* @param NewBounds	Intial size of the Octree
	* @param inDrawDebugInfo	Whether or not to display debug boundaries
	*/
	UFUNCTION(BlueprintCallable, Category = Octree)
	void Initialize(const FBox& inNewBounds, const bool& inDrawDebugInfo);

	/**
	* Adds a spefic element to the Octree
	* @param NewOctreeElement	FOctreeElement to be added.
	*/
	UFUNCTION(BlueprintCallable, Category = Octree)
	void AddOctreeElement(const FTestOctreeElement& inNewOctreeElement);

	/**
	* Returns elements within the specified region.
	* @param inBoundingBoxQuery	Box to query Octree.
	* @return TArray of Elements
	*/
	UFUNCTION(BlueprintCallable, Category = Octree)
	TArray<FTestOctreeElement> GetElementsWithinBounds(const FBoxSphereBounds& inBoundingBoxQuery);

	/**
	* Returns elements within the specified region.
	* @param inBoundingBoxQuery	Box to query Octree.
	* @return TArray of Elements
	*/
	TArray<FTestOctreeElement> GetElementsWithinBounds(const FBoxCenterAndExtent& inBoundingBoxQuery);

	/** Draws Debug information at runtime */
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = Debug)
	bool bDrawDebugInfo = false;

private:

	FSimpleOctree* OctreeData;
	bool bInitialized;

	UPROPERTY(EditAnywhere)
	TSubclassOf<class AActor> ElementClass;

	UPROPERTY()
	USceneComponent* Root;

protected:
	// Called when the game starts or when spawned
	virtual void BeginPlay() override;

	// Called every frame
	virtual void Tick(float DeltaTime) override;	
	
};
