package root.ui;

import java.io.BufferedReader;
import java.io.IOException;
import java.sql.*;
import java.time.LocalDate;
import java.util.List;

import root.Main;
import root.rdg.*;
import root.ts.*;

/**
 *
 * @author luk
 */
public class MainMenu extends Menu {
    private static final int PAGE_SIZE = 30;

    public MainMenu() {
        System.out.print(Menu.ANSI_BLUE);
        System.out.println(" __    __           _     _          __      ___      _        _                    \n" +
                "/ / /\\ \\ \\___  _ __| | __| |   ___  / _|    /   \\__ _| |_ __ _| |__   __ _ ___  ___ \n" +
                "\\ \\/  \\/ / _ \\| '__| |/ _` |  / _ \\| |_    / /\\ / _` | __/ _` | '_ \\ / _` / __|/ _ \\\n" +
                " \\  /\\  / (_) | |  | | (_| | | (_) |  _|  / /_// (_| | || (_| | |_) | (_| \\__ \\  __/\n" +
                "  \\/  \\/ \\___/|_|  |_|\\__,_|  \\___/|_|   /___,' \\__,_|\\__\\__,_|_.__/ \\__,_|___/\\___|");
        System.out.print(Menu.ANSI_RESET);
    }

    @Override
    public void print() {
        System.out.println("---------------------------------");
        System.out.println("| 0.  RECREATE database         |");
        System.out.println("---------------------------------");
        System.out.println("| 1.  LIST all users            |");
        System.out.println("| 2.  LIST all characters       |");
        System.out.println("---------------------------------");
        System.out.println("| 3.  SHOW user's services      |");
        System.out.println("| 4.  SHOW a user               |");
        System.out.println("| 5.  ADD a user                |");
        System.out.println("| 6.  EDIT a user               |");
        System.out.println("| 7.  DELETE a user             |");
        System.out.println("---------------------------------");
        System.out.println("| 8.  SHOW a character          |");
        System.out.println("| 9.  ADD a character           |");
        System.out.println("| 10. EDIT a character          |");
        System.out.println("| 11. DELETE a character        |");
        System.out.println("| 12. GIVE item to character    |");
        System.out.println("| 13. SHOW character's stats    |");
        System.out.println("---------------------------------");
        System.out.println("| 14. LIST all classes          |");
        System.out.println("| 15. SHOW a class              |");
        System.out.println("| 16. ADD a class               |");
        System.out.println("| 17. EDIT a class              |");
        System.out.println("| 18. DELETE a class            |");
        System.out.println("---------------------------------");
        System.out.println("| 19. SHOW class skills         |");
        System.out.println("| 20. ADD a skill to class      |");
        System.out.println("| 21. DELETE a skill from class |");
        System.out.println("---------------------------------");
        System.out.println("| 22. LIST all races            |");
        System.out.println("| 23. ADD a class to race       |");
        System.out.println("---------------------------------");
        System.out.println("| 24. transfer character        |");
        System.out.println("| 25. change appearance         |");
        System.out.println("| 26. do game action            |");
        System.out.println("| 27. simulate battle           |");
        System.out.println("| 28. reward bounty             |");
        System.out.println("| 29. show jack statistics      |");
        System.out.println("| 30. show occupancy statistics |");
        System.out.println("| 31. show best players         |");
        System.out.println("| 32. exit                      |");
        System.out.println("---------------------------------\n");
        System.out.print(ANSI_BLUE + "What would you like to do? " + ANSI_RESET);
    }

    @Override
    public void handle(String option) {
        try {
            switch (option) {
                case "0":   recreateDatabase(); break;
                case "1":   listAllUsers(); break;
                case "2":   listAllChars(); break;
                case "3":   showServices(); break;
                case "4":   showAUser(); break;
                case "5":   addAUser(); break;
                case "6":   editAUser(); break;
                case "7":   deleteAUser(); break;
                case "8":   showACharacter(); break;
                case "9":   addACharacter(); break;
                case "10":  editACharacter(); break;
                case "11":  deleteACharacter(); break;
                case "12":  giveItem(); break;
                case "13":  showStats(); break;
                case "14":  listAllClasses(); break;
                case "15":  showAClass(); break;
                case "16":  addAClass(); break;
                case "17":  editAClass(); break;
                case "18":  deleteAClass(); break;
                case "19":  showClassSkills(); break;
                case "20":  addSkillToClass(); break;
                case "21":  deleteSkillFromClass(); break;
                case "22":  listAllRaces(); break;
                case "23":  addClassToRace(); break;
                case "24":  transferChar(); break;
                case "25":  changeAppearance(); break;
                case "26":  doAction(); break;
                case "27":  simulateBattle(); break;
                case "28":  rewardBounty(); break;
                case "29":  showJackStatistics(); break;
                case "30":  showOccupancyStatistics(); break;
                case "31":  showBestPlayers(); break;
                case "32":  exit(); return;
                default:    System.out.println("Unknown option"); break;
            }
            System.out.print("Press ENTER to continue...");
            br.readLine();
        } catch (IOException|NumberFormatException ex) {
            printError("ERROR: Wrong input!");
        } catch (ServiceException|ActionException|MenuException|FormatException ex) {
            System.out.println(ex.getMessage());
        } catch (SQLException ex) {
            printError(ex.getMessage());
            if (ex.getMessage().contains("foreign key constraint"))
                printError("There are rows depending on this one!");
        }
    }

    private void recreateDatabase() {
        long cas = System.currentTimeMillis();
        Main.createScript();
        System.out.println(Menu.ANSI_YELLOW + "Execution time: " + (System.currentTimeMillis() - cas) + "ms" + Menu.ANSI_RESET);
    }

    private void listAllUsers() throws SQLException, IOException, MenuException {
        System.out.print("Enter page: ");
        int pageCount = Integer.valueOf(br.readLine());
        List<User> users = UserFinder.getInstance().findAllPage(PAGE_SIZE, pageCount);
        if (users.size() == 0)
            throw new MenuException("Invalid page number!");
        printBorder(6, 22);
        for (User user : users) {
            UserPrinter.print(user);
        }
        printBorder(6, 22);
    }

    private void listAllChars() throws SQLException, IOException, MenuException {
        System.out.print("Enter page: ");
        int pageCount = Integer.valueOf(br.readLine());
        long cas = System.currentTimeMillis();
        List<Charakter> charakters = CharakterFinder.getInstance().findAllPage(PAGE_SIZE, pageCount);
        if (charakters.size() == 0)
            throw new MenuException("Invalid page number!");
        printBorder(10, 15);
        for (Charakter charakter : charakters) {
            CharacterPrinter.print(charakter);
        }
        printBorder(10,15);
        System.out.println(Menu.ANSI_YELLOW + "Execution time: " + (System.currentTimeMillis() - cas) + "ms" + Menu.ANSI_RESET);
    }

    private void showServices() throws SQLException, IOException, MenuException {
        System.out.print("Enter user's id: ");
        int userId = Integer.valueOf(br.readLine());
        User user = UserFinder.getInstance().findById(userId);
        if (user == null)
            throw new MenuException("No such user exists!");
        printBorder(5, 15);
        ServicesPrinter.print(ServiceFinder.getInstance().findByOwnerId(userId));
        printBorder(5, 15);
    }

    private void showAUser() throws SQLException, IOException, MenuException {
        System.out.print("Enter users's id: ");
        int userId = Integer.valueOf(br.readLine());

        User user = UserFinder.getInstance().findById(userId);
        if (user == null) {
            throw new MenuException("No such user exists!");
        } else {
            printBorder(6, 22);
            UserPrinter.print(user);
            printBorder(6, 22);
        }
    }

    private void addAUser() throws IOException, SQLException, FormatException {
        User user = new User();
        System.out.print("Enter first name: ");
        user.setFirstName(br.readLine());
        System.out.print("Enter last name: ");
        user.setLastName(br.readLine());
        System.out.print("Enter username: ");
        user.setUsername(br.readLine());
        System.out.print("Enter password: ");
        user.setPassword(br.readLine());
        user.setCredit(0);
        user.setDateCreated(Date.valueOf(LocalDate.now()));

        user.insert();
        System.out.println("The user has been successfully added.");
    }

    private void editAUser() throws IOException, SQLException, MenuException, FormatException {
        System.out.print("Enter users's id: ");
        int userId = Integer.valueOf(br.readLine());

        User user = UserFinder.getInstance().findById(userId);
        if (user == null) {
            throw new MenuException("No such user exists!");
        } else {
            System.out.println("Original data: ");
            printBorder(6, 22);
            UserPrinter.print(user);
            printBorder(6, 22);

            System.out.print("Enter new first name: ");
            user.setFirstName(br.readLine());
            System.out.print("Enter new last name: ");
            user.setLastName(br.readLine());
            System.out.print("Enter new username: ");
            user.setUsername(br.readLine());
            System.out.print("Enter new password: ");
            user.setPassword(br.readLine());
            System.out.print("Enter new credit: ");
            user.setCredit(Integer.valueOf(br.readLine()));

            user.update();
            System.out.println("The user has been successfully updated.");
        }
    }

    private void deleteAUser() throws SQLException, IOException, MenuException {
        System.out.print("Enter users's id: ");
        int userId = Integer.valueOf(br.readLine());

        User user = UserFinder.getInstance().findById(userId);

        if (user == null) {
            throw new MenuException("No such user exists");
        } else {
            user.delete();
            System.out.println("The user has been successfully deleted.");
        }
    }

    private void showACharacter() throws SQLException, IOException, MenuException {
        System.out.print("Enter character's id: ");
        int characterId = Integer.valueOf(br.readLine());

        Charakter charakter = CharakterFinder.getInstance().findById(characterId);
        
        if (charakter == null) {
            throw new MenuException("No such character exists!");
        } else {
            printBorder(10, 15);
            CharacterPrinter.print(charakter);
            printBorder(10, 15);
        }
    }

    private void addACharacter() throws IOException, SQLException, MenuException, FormatException {
        Charakter charakter = new Charakter();

        System.out.print("Enter name: ");
        charakter.setName(br.readLine());
        System.out.print("Enter owner id: ");
        charakter.setUserId(Integer.valueOf(br.readLine()));
        System.out.print("Enter experience: ");
        charakter.setExperience(Integer.valueOf(br.readLine()));
        System.out.print("Enter model id: ");
        charakter.setModelId(Integer.valueOf(br.readLine()));
        System.out.print("Enter race id: ");
        charakter.setRaceId(Integer.valueOf(br.readLine()));
        System.out.print("Enter class id: ");

        Integer classId = Integer.valueOf(br.readLine());
        if (!OtherActions.classesForRace(charakter.getRaceId()).contains(classId))
            throw new MenuException("Class does not exist or is not available for chosen race!");
        charakter.setClassId(classId);

        charakter.setHealth(10);
        charakter.insert();
        System.out.println("The character has been successfully added.");
    }

    private void editACharacter() throws IOException, SQLException, MenuException, FormatException {
        System.out.print("Enter character's id: ");
        int characterId = Integer.valueOf(br.readLine());

        Charakter charakter = CharakterFinder.getInstance().findById(characterId);

        if (charakter == null) {
            throw new MenuException("No such character exists!");
        } else {
            System.out.println("Original data: ");
            printBorder(10, 15);
            CharacterPrinter.print(charakter);
            printBorder(10, 15);

            System.out.print("Enter name: ");
            charakter.setName(br.readLine());
            System.out.print("Enter experience: ");
            charakter.setExperience(Integer.valueOf(br.readLine()));
            System.out.print("Enter health: ");
            int newHealth = Integer.valueOf(br.readLine());
            Stats s = StatsFinder.getInstance().findById(characterId);
            if (newHealth > s.getMaxHealth())
                throw new MenuException("You cannot exceed maximum health of this character!");
            charakter.setHealth(newHealth);

            charakter.update();
            System.out.println("The character has been successfully updated.");
        }
    }

    private void deleteACharacter() throws SQLException, IOException, MenuException {
        System.out.print("Enter character's id: ");
        int characterId = Integer.valueOf(br.readLine());

        Charakter charakter = CharakterFinder.getInstance().findById(characterId);

        if (charakter == null) {
            throw new MenuException("No such character exists!");
        } else {
            charakter.delete();
            System.out.println("The character has been successfully deleted.");
        }
    }

    private void giveItem() throws SQLException, IOException, ActionException {
        System.out.print("Enter character's id: ");
        int characterId = Integer.valueOf(br.readLine());

        GameActions.getInstance().giveItemToChar(characterId);
        System.out.println("Character now have this item.");
    }

    private void showStats() throws SQLException, IOException, MenuException {
        System.out.print("Enter character's id: ");
        int characterId = Integer.valueOf(br.readLine());

        Stats stats = StatsFinder.getInstance().findById(characterId);

        if (stats == null) {
            throw new MenuException("Character does not exist!");
        } else {
            StatsPrinter.print(stats);
        }
    }

    private void listAllClasses() throws SQLException {
        printBorder(8, 17);
        for (Klassa klassa : KlassaFinder.getInstance().findAll()) {
            KlassaPrinter.print(klassa);
        }
        printBorder(8, 17);
    }

    private void showAClass() throws SQLException, IOException, MenuException {
        System.out.print("Enter class id: ");
        int classId = Integer.valueOf(br.readLine());

        Klassa klassa = KlassaFinder.getInstance().findById(classId);

        if (klassa == null) {
            throw new MenuException("No such class exists!");
        }
        else {
            printBorder(8, 17);
            KlassaPrinter.print(klassa);
            printBorder(8, 17);
        }
    }

    private void addAClass() throws IOException, SQLException {
        Klassa klassa = new Klassa();

        System.out.print("Enter name: ");
        klassa.setName(br.readLine());
        System.out.print("Enter default hp: ");
        klassa.setDefaultHp(Integer.valueOf(br.readLine()));
        System.out.print("Enter default def: ");
        klassa.setDefaultDef(Integer.valueOf(br.readLine()));
        System.out.print("Enter default str: ");
        klassa.setDefaultStr(Integer.valueOf(br.readLine()));
        System.out.print("Enter hp mod ");
        klassa.setHpMod(Integer.valueOf(br.readLine()));
        System.out.print("Enter str mod ");
        klassa.setStrMod(Integer.valueOf(br.readLine()));
        System.out.print("Enter def mod ");
        klassa.setDefMod(Integer.valueOf(br.readLine()));

        klassa.insert();
        System.out.println("The class has been successfully added.");
    }

    private void editAClass() throws IOException, SQLException, MenuException {
        System.out.print("Enter class id: ");
        int classId = Integer.valueOf(br.readLine());

        Klassa klassa = KlassaFinder.getInstance().findById(classId);

        if (klassa == null) {
            throw new MenuException("No such class exists!");
        } else {
            System.out.println("Original data: ");
            printBorder(8, 17);
            KlassaPrinter.print(klassa);
            printBorder(8, 17);

            System.out.print("Enter new name: ");
            klassa.setName(br.readLine());
            System.out.print("Enter new default hp: ");
            klassa.setDefaultHp(Integer.valueOf(br.readLine()));
            System.out.print("Enter new default str: ");
            klassa.setDefaultStr(Integer.valueOf(br.readLine()));
            System.out.print("Enter new default def: ");
            klassa.setDefaultDef(Integer.valueOf(br.readLine()));
            System.out.print("Enter new hp mod ");
            klassa.setHpMod(Integer.valueOf(br.readLine()));
            System.out.print("Enter new str mod ");
            klassa.setStrMod(Integer.valueOf(br.readLine()));
            System.out.print("Enter new def mod ");
            klassa.setDefMod(Integer.valueOf(br.readLine()));

            klassa.update();

            System.out.println("The class has been successfully updated.");
        }
    }

    private void deleteAClass() throws SQLException, IOException, MenuException {
        System.out.print("Enter class id: ");
        int classId = Integer.valueOf(br.readLine());

        Klassa klassa = KlassaFinder.getInstance().findById(classId);

        if (klassa == null) {
            throw new MenuException("No such class exists!");
        } else {
            klassa.delete();
            System.out.println("The class has been successfully deleted.");
        }
    }

    private void showClassSkills() throws IOException, SQLException, MenuException {
        System.out.print("Enter class id: ");
        int classId = Integer.valueOf(br.readLine());
        Klassa klassa = KlassaFinder.getInstance().findById(classId);
        if (klassa == null)
            throw new MenuException("Class with this ID does not exist!");

        printBorder(8, 18);
        for (Skill skill : SkillFinder.classSkills(classId)) {
            SkillPrinter.print(skill);
        }
        printBorder(8, 18);
    }

    private void addSkillToClass() throws IOException, SQLException, MenuException {
        System.out.print("Enter class id: ");
        Klassa klassa = KlassaFinder.getInstance().findById(Integer.valueOf(br.readLine()));
        if (klassa == null)
            throw new MenuException("Class with this ID does not exist!");

        System.out.println("Available skills:");
        printBorder(8, 18);
        for (Skill skill : SkillFinder.getInstance().findAll())
            SkillPrinter.print(skill);
        printBorder(8, 18);

        System.out.print("Enter skill id: ");
        Skill skill = SkillFinder.getInstance().findById(Integer.valueOf(br.readLine()));
        if (skill == null)
            throw new MenuException("Skill with this ID does not exist!");

        OtherActions.addSkillToClass(skill, klassa);
        System.out.println("Successfully added skill to class.");
    }

    private void deleteSkillFromClass() throws IOException, MenuException, SQLException {
        System.out.print("Enter class id: ");
        Klassa klassa = KlassaFinder.getInstance().findById(Integer.valueOf(br.readLine()));
        if (klassa == null)
            throw new MenuException("Class with this ID does not exist!");

        System.out.println("Skills of this Class:");
        printBorder(8, 18);
        for (Skill skill : SkillFinder.classSkills(klassa.getId()))
            SkillPrinter.print(skill);
        printBorder(8, 18);

        System.out.print("Enter skill id: ");
        Skill skill = SkillFinder.getInstance().findById(Integer.valueOf(br.readLine()));
        if (skill == null)
            throw new MenuException("Skill with this ID does not exist!");

        OtherActions.deleteSkillFromClass(skill, klassa);
        System.out.println("Successfully deleted skill from class.");
    }

    private void listAllRaces() throws SQLException {
        printBorder(11, 15);
        for (Race race : RaceFinder.getInstance().findAll()) {
            RacePrinter.print(race);
        }
        printBorder(11, 15);
    }

    private void addClassToRace() throws IOException, SQLException, MenuException {
        System.out.print("Enter race id: ");
        Race race = RaceFinder.getInstance().findById(Integer.valueOf(br.readLine()));
        if (race == null)
            throw new MenuException("Race with this ID does not exist!");

        System.out.print("Enter class id: ");
        Klassa klassa = KlassaFinder.getInstance().findById(Integer.valueOf(br.readLine()));
        if (klassa == null)
            throw new MenuException("Class with this ID does not exist!");

        OtherActions.addClassToRace(klassa, race);
        System.out.println("Successfully added class to a race.");
    }

    private void transferChar() throws IOException, SQLException, ServiceException, MenuException, FormatException {
        System.out.print("Enter source account username: ");
        User sourceUser = UserFinder.getInstance().findByUsername(br.readLine());
        if (sourceUser == null) throw new MenuException("No such user exists!");
        System.out.println("Available characters: ");
        printBorder(10, 15);
        for (Charakter charakter : CharakterFinder.getInstance().findByOwnerId(sourceUser.getId())) {
            CharacterPrinter.print(charakter);
        }
        printBorder(10, 15);

        System.out.print("Which character would you like to transfer? (id): ");
        Charakter charakter = CharakterFinder.getInstance().findById(Integer.valueOf(br.readLine()));
        System.out.print("Enter destination account username: ");
        User destUser = UserFinder.getInstance().findByUsername(br.readLine());

        PlayerServices.getInstance().transferChar(destUser, sourceUser, charakter);
        System.out.println("Transfer has been successful!");
    }

    private void changeAppearance() throws IOException, SQLException, ServiceException, MenuException {
        System.out.print("Enter user id: ");
        User user = UserFinder.getInstance().findById(Integer.valueOf(br.readLine()));
        if (user == null) throw new MenuException("User does not exist!");

        System.out.println("Available characters: ");
        printBorder(10, 15);
        for (Charakter charakter : CharakterFinder.getInstance().findByOwnerId(user.getId()))
            CharacterPrinter.print(charakter);
        printBorder(10, 15);

        System.out.print("Enter character id: ");
        int characterId = Integer.valueOf(br.readLine());
        System.out.println("\nAvailable models: ");
        ModelPrinter.print(ModelFinder.getInstance().findAll());
        System.out.print("Choose model (id) or type 'N' to create a new one: ");

        String input = br.readLine();
        if (input.equals("N")) addAModel(br);

        Model model = ModelFinder.getInstance().findById(Integer.valueOf(input));

        PlayerServices.getInstance().changeAppearance(characterId, model.getId(), user);
        System.out.println("Appearance has been changed!");
    }

    private void addAModel(BufferedReader br) throws IOException, SQLException {
        Model model = new Model();
        System.out.print("Enter gender: ");
        String input = br.readLine();
        while (!(input.equals("M") || input.equals("F"))) // kontrola vstupu Java
            printError("Gender can only be F or M, enter again!: ");
        model.setGender(input);
        System.out.print("Enter height: ");
        model.setHeight(Integer.valueOf(br.readLine()));
        System.out.print("Enter weight: ");
        model.setWeight(Integer.valueOf(br.readLine()));
        System.out.print("Enter hair color: ");
        model.setHairColor(br.readLine());
        System.out.print("Enter eye color: ");
        model.setEyeColor(br.readLine());
        model.insert();

    }

    private void doAction() throws IOException, SQLException, ActionException {
        System.out.print("Enter attacking character id: ");
        int attackerId = Integer.valueOf(br.readLine());
        System.out.print("Enter target character's id: ");
        int targetId = Integer.valueOf(br.readLine());

        GameActions.getInstance().doAction(attackerId, targetId);
    }

    private void simulateBattle() throws IOException, SQLException, ActionException {
        System.out.print("Enter character1 id: ");
        int ch1Id = Integer.valueOf(br.readLine());
        System.out.print("Enter character2 id: ");
        int ch2Id = Integer.valueOf(br.readLine());

        GameActions.getInstance().simulateBattle(ch1Id, ch2Id);
    }

    private void rewardBounty() throws IOException, SQLException, ActionException {
        System.out.print("Enter month: ");
        GameActions.getInstance().rewardBounty(Integer.valueOf(br.readLine()));
    }

    private void showJackStatistics() throws SQLException {
        printBorder(7, 15);
        JackStatisticsPrinter.print(JackStatisticFinder.getInstance().findAll());
        printBorder(7, 15);
    }

    private void showOccupancyStatistics() throws SQLException, IOException {
        System.out.println("Would you like to print full (type FULL) statistics or lite version only (type LITE)?");
        System.out.println("(WARNING: FULL can take some time!)");
        String choice = br.readLine();
        while (!choice.equals("FULL") && !choice.equals("LITE")) {
            System.out.println("Wrong input! Type again!");
            choice = br.readLine();
        }
        long cas = System.currentTimeMillis();
        printBorder(6, 15);
        if (choice.equals("FULL"))
            OccupancyStatisticsPrinter.print(OccupancyStatisticFinder.getInstance().findAll());
        else
            OccupancyStatisticsPrinter.print(OccupancyStatisticFinder.getInstance().findAllMin());
        printBorder(6, 15);
        System.out.println(Menu.ANSI_YELLOW + "Execution time: " + (System.currentTimeMillis() - cas) + "ms" + Menu.ANSI_RESET);
    }

    private void showBestPlayers() throws SQLException {
        ShowBestPlayers.print();
    }

    ///HELPER FUNCTIONS///

    private void printError(String msg) {
        System.out.println(ANSI_RED + msg + ANSI_RESET);
    }

    public static String fillSpace(String str, int num) {
        StringBuffer novy = new StringBuffer(str);
        for (int i = 0; i < (num - str.length()); i++) {
            novy.append(" ");
        }
        return novy.toString();
    }

    public static void printBorder(int columnCount, int columnWidth){
        System.out.print(ANSI_BLUE);
        for (int i = 1; i <= columnCount; i++) {
            for (int j = 0; j < columnWidth; j++) {
                System.out.print("-");
            }
        }
        System.out.println("-" + ANSI_RESET);
    }

}