package algorithmX

import algorithmX.matrix.Matrix
import algorithmX.name.Name
import algorithmX.name.StringName
import algorithmX.node.ItemNode
import algorithmX.node.Node
import algorithmX.node.OptionNode
import algorithmX.printer.Printer
import algorithmX.printer.NamePrinter

class AlgorithmX(matrix : Matrix,
                 names : List<Name>? = null,
                 private val printer : Printer = NamePrinter(),
                 private val allSolutions : Boolean = true,
                 private val printSolutions : Boolean = true) {

    //attributes ----------------------------------------------------------------
    private val visited = mutableListOf<Node>()
    private lateinit var header: ItemNode
    private var solvingAllowed = true

    //constants ----------------------------------------------------------------
    private val MAX_NUM = Int.MAX_VALUE
    private val HEADER_NAME = "header"

    //constructors -------------------------------------------------------------
    init {
        createDataStructure(matrix, names)
    }

    //methods ------------------------------------------------------------------
    private fun createDataStructure(matrix : Matrix, names : List<Name>?) {
        createHeader()
        if (matrix.matrix.isNotEmpty()) {
            createItems(matrix, names)
            createOptions(matrix)
        }
    }

    private fun createHeader() {
        header = ItemNode(StringName(HEADER_NAME), MAX_NUM)
        header.rLink = header
        header.lLink = header
        header.uLink = header
        header.dLink = header
    }

    private fun createItems(matrix : Matrix, names : List<Name>?) {
        for (i in matrix.matrix[0].indices) {
            val node =
                    if (names == null || names.size <= i)
                        ItemNode(StringName((i + 1).toString()))
                    else
                        ItemNode(names[i])
            node.uLink = node
            node.dLink = node
            node.rLink = header
            node.lLink = header.lLink
            header.lLink!!.rLink = node
            header.lLink = node
        }
    }

    private fun createOptions(matrix : Matrix) {
        val rowList = mutableListOf<OptionNode>()
        for (row in matrix.matrix) {
            createOptionsRow(row, rowList)
        }
    }

    private fun createOptionsRow(row : List<Int>, rowList : MutableList<OptionNode>) {
        rowList.clear()
        fillOptionRow(row, rowList)
        if (rowList.isNotEmpty()) {
            connectRowInGrid(rowList)
        }
    }

    private fun fillOptionRow(row : List<Int>, rowList : MutableList<OptionNode>) {
        var itemNode : ItemNode = (header.rLink as ItemNode)
        for (num in row) {
            if (num == 1) {
                addNodeToGrid(rowList, itemNode)
            }
            itemNode = (itemNode.rLink as ItemNode)
        }
    }

    private fun addNodeToGrid(rowList : MutableList<OptionNode>, itemNode : ItemNode) {
        val optionNode = OptionNode()
        optionNode.top = itemNode
        optionNode.lLink = optionNode
        optionNode.rLink = optionNode
        optionNode.uLink = itemNode.uLink
        optionNode.dLink = itemNode
        itemNode.uLink!!.dLink = optionNode
        itemNode.uLink = optionNode
        itemNode.length++
        rowList.add(optionNode)
    }

    private fun connectRowInGrid(rowList : MutableList<OptionNode>) {
        val optionHeader : OptionNode = rowList[0]
        for (i in 1 until rowList.size) {
            val optionNode = rowList[i]
            optionNode.lLink = optionHeader.lLink
            optionNode.rLink = optionHeader
            optionHeader.lLink!!.rLink = optionNode
            optionHeader.lLink = optionNode
        }
    }

    fun solve() : Int {
        if (header.rLink == header) {
            if (visited.isEmpty()){
                return 0
            }
            if (printSolutions) {
                printer.add(createResult())
            }
            solvingAllowed = allSolutions
            return 1
        } else {
            return backTrack()
        }
    }

    private fun backTrack() : Int {
        var count = 0
        val item = selectItem()
        var x = item.dLink!!
        cover(item)
        while (x != item) {
            visited.add(x)
            rowCover(x)
            if (solvingAllowed) {
                count += solve()
            }
            rowUnCover(x)
            visited.remove(x)
            x = x.dLink!!
        }
        unCover(item)
        return count
    }

    private fun createResult() : List<List<Name>> {
        val names = mutableListOf<List<Name>>()
        for (optionNode in visited) {
            names.add(createResultRow(optionNode))
        }
        return names
    }

    private fun createResultRow(optionNode : Node) : List<Name> {
        val line = mutableListOf<Name>()
        line.add((((optionNode as OptionNode).top as ItemNode?)!!.name))
        var p : OptionNode = (optionNode.rLink as OptionNode?)!!
        while (p != optionNode) {
            line.add(((p.top as ItemNode?)!!.name))
            p = (p.rLink as OptionNode?)!!
        }
        return line
    }

    private fun selectItem() : ItemNode {
        var minItem: ItemNode = header
        var tempItem: ItemNode = (header.rLink as ItemNode?)!!
        while (tempItem != header) {
            if (tempItem.length < minItem.length) {
                minItem = tempItem
            }
            tempItem = (tempItem.rLink as ItemNode?)!!
        }
        return minItem
    }

    private fun rowCover(x : Node) {
        var p: Node = x.rLink!!
        while (p != x) {
            cover((p as OptionNode).top!!)
            p = p.rLink!!
        }
    }

    private fun rowUnCover(x : Node) {
        var p = x.lLink!!
        while (p != x) {
            unCover((p as OptionNode).top!!)
            p = p.lLink!!
        }
    }

    private fun cover(itemNode : Node) {
        var p: Node = itemNode.dLink!!
        while (p != itemNode) {
            hide(p)
            p = p.dLink!!
        }
        removeNode(itemNode)
    }

    private fun removeNode(itemNode : Node) {
        itemNode.lLink!!.rLink = itemNode.rLink
        itemNode.rLink!!.lLink = itemNode.lLink
    }

    private fun hide(optionNode : Node) {
        var q: Node = optionNode.rLink!!
        while (q != optionNode) {
            q.uLink!!.dLink = q.dLink
            q.dLink!!.uLink = q.uLink
            q = q.rLink!!
        }
    }

    private fun unCover(itemNode : Node) {
        unRemoveNode(itemNode)
        var p: Node = itemNode.uLink!!
        while (p != itemNode) {
            unHide(p)
            p = p.uLink!!
        }
    }

    private fun unRemoveNode(itemNode : Node) {
        itemNode.lLink!!.rLink = itemNode
        itemNode.rLink!!.lLink = itemNode
    }

    private fun unHide(optionNode : Node) {
        var q: Node = optionNode.lLink!!
        while (q != optionNode) {
            q.uLink!!.dLink = q
            q.dLink!!.uLink = q
            q = q.lLink!!
        }
    }

}