package cubes

import algorithmX.Benchmark
import algorithmX.matrix.Matrix
import algorithmX.name.Name
import cubes.point.Point
import java.io.File
import java.nio.file.Files
import java.nio.file.StandardOpenOption

class Formalisation {

    //attributes ---------------------------------------------------------------------
    private lateinit var matrix : MutableList<MutableList<Int>>
    private lateinit var names : MutableList<Name>
    private lateinit var shapes : MutableSet<Shape>
    private lateinit var symmetricShapes : MutableSet<Shape>
    private lateinit var rotationMatrices : List<Matrix>
    private var shapeWithMostRotations = Shape.emptyShape()
    private lateinit var shape : Shape
    private lateinit var task : Task
    private var index : Int = 0
    private var isRepeatable = false
    private var addSymmetries = false
    private var withoutSymmetries = false
    companion object {

        var writeToFile = false

    }

    //constants ----------------------------------------------------------------------
    private val NOT_FOUND = -1
    private val EMPTY = ""
    private val DELIMETER = " "
    private val NEW_LINE = "\n"
    private val TIME = 15L

    //methods ------------------------------------------------------------------------
    fun createMatrixAndNamesFromTask(task : Task, withoutSymmetries : Boolean = false) : Pair<Matrix, List<Name>> {
        this.task = task
        this.withoutSymmetries = withoutSymmetries
        initialiseAttributes()
        createNames()
        fillMatrix()
        writeToFile()
        return Pair(Matrix(matrix), names)
    }

    private fun initialiseAttributes() {
        matrix = mutableListOf()
        shapes = mutableSetOf()
        symmetricShapes = mutableSetOf()
        names = mutableListOf()
        rotationMatrices = Point.instance.rotations()
    }

    private fun createNames() {
        createItemNames()
        createShapeNames()
    }

    private fun createItemNames() {
        for (point in task.puzzle.points) {
            names.add(point as Name)
        }
    }

    private fun createShapeNames() {
        names.addAll(
                task.shapes
                        .map { it as Name }
                        .plus(task.fixedShapes
                                .map { it as Name }
                        )
        )
    }

    private fun fillMatrix() {
        findShapeWithMostRotations()
        isRepeatable = false
        fillMatrixWithShapes(task.shapes)
        fillMatrixWithFixedShapes(task.fixedShapes, task.shapes.size)
        isRepeatable = true
        fillMatrixWithShapes(task.repeatableShapes)
    }

    private fun findShapeWithMostRotations() {
        if (withoutSymmetries) {
            shapeWithMostRotations = Shape.emptyShape()
            for (shape in task.shapes) {
                if (shape.numberOfRotations() > shapeWithMostRotations.numberOfRotations()) {
                    shapeWithMostRotations = shape
                }
            }
        }
    }

    private fun fillMatrixWithFixedShapes(shapesSet : Set<Shape>, indexOffset : Int = 0) {
        for ((ind, value) in shapesSet.withIndex()) {
            index = ind + indexOffset
            checkAndAddToRow(value)
        }
    }

    private fun fillMatrixWithShapes(shapesSet : Set<Shape>, indexOffset : Int = 0) {
        for ((ind, value) in shapesSet.withIndex()) {
            shapes.clear()
            symmetricShapes.clear()
            shape = value
            index = ind + indexOffset
            addSymmetries = shape == shapeWithMostRotations
            rotateShape()
        }
    }

    private fun rotateShape() {
        for (rotationMatrix in rotationMatrices) {
            shape.rotate(rotationMatrix).normalise()
            checkAndMoveShapeRotation()
        }
    }

    private fun checkAndMoveShapeRotation() {
        if (isGoodShapeRotation()) {
            moveShapeRotation()
        }
    }

    private fun isGoodShapeRotation() : Boolean {
        return shape.size.isLessOrEqual(task.puzzle.size) &&
                shapeNotAdded()
    }

    private fun shapeNotAdded() : Boolean {
        return shapes.add(shape.copy())
    }

    private fun moveShapeRotation() {
        for (point in task.puzzle.points) {
            val shapeCopy = shape.copy().translate(point)
            checkAndAddToRow(shapeCopy)
            addSymmetries(shapeCopy)
        }
    }

    private fun addSymmetries(shape: Shape) {
        if (addSymmetries) {
            val puzzleCopy = task.puzzle.copy()
            for (rotationMatrix in rotationMatrices) {
                puzzleCopy.rotate(rotationMatrix)
                shape.rotate(rotationMatrix)
                val offset = Point.instance.getMinimalCoordinates(puzzleCopy.points).invert()
                puzzleCopy.translate(offset)
                shape.translate(offset)
                if (task.puzzle == puzzleCopy) {
                    symmetricShapes.add(shape.copy())
                }
            }
        }
    }

    private fun checkAndAddToRow(shape : Shape) {
        if (isShapeGoodToBeAdded(shape)) {
            addToRow(shape)
        }
    }

    private fun isShapeGoodToBeAdded(shape: Shape) : Boolean {
        return isShapeWithin(shape) && (!withoutSymmetries || shapeNotSymmetric(shape))
    }

    private fun isShapeWithin(shape: Shape) : Boolean {
        for (point in shape.points) {
            if (task.puzzle.points.indexOf(point) == NOT_FOUND) {
                return false
            }
        }
        return true
    }

    private fun shapeNotSymmetric(shape : Shape) : Boolean {
        return symmetricShapes.add(shape.copy())
    }

    private fun addToRow(shape : Shape) {
        val matrixRow = MutableList(names.size) { 0 }
        for (point in shape.points) {
            matrixRow[task.puzzle.points.indexOf(point)] = 1
        }
        if (!isRepeatable) {
            matrixRow[task.puzzle.points.size + index] = 1
        }
        matrix.add(matrixRow)
    }

    private fun writeToFile() {
        if (!writeToFile) {
            return
        }
        var matrixTextRepresentation = EMPTY
        for (line in matrix) {
            matrixTextRepresentation += line.joinToString(DELIMETER) + NEW_LINE
        }
        matrixTextRepresentation += NEW_LINE
        val file = File(Benchmark.FILE)
        file.createNewFile()
        Thread.sleep(TIME)
        Files.write(file.toPath(), matrixTextRepresentation.toByteArray(), StandardOpenOption.APPEND)
    }

}