package cubes.point

import algorithmX.name.Name
import algorithmX.matrix.Matrix

class Point3D(var x : Int = 0,
              var y : Int = 0,
              var z : Int = 0) : Point {

    //attributes ----------------------------------------------------------------------
    private val rotations = mutableListOf<Matrix>()

    //constants -----------------------------------------------------------------------
    private val QUARTER = 90
    private val HALF = 180
    private val THREE_QUARTER = 270

    //constructor ---------------------------------------------------------------------
    init {
        Point.instance = this
    }

    //methods -------------------------------------------------------------------------
    override fun compareTo(other: Name) : Int {
        return toString().compareTo(other.toString())
    }

    override fun toString() : String {
        return "($x, $y, $z)"
    }

    override fun equals(other : Any?) : Boolean {
        return other is Point3D &&
                x == other.x &&
                y == other.y &&
                z == other.z
    }

    override fun hashCode() : Int {
        return (100 * x).hashCode() + (10 * y).hashCode() + z.hashCode()
    }

    override fun copy() : Point {
        return Point3D(x, y, z)
    }

    override fun one() : Point {
        return Point3D(1, 1, 1)
    }

    override fun left(): Point {
        return Point3D(x - 1, y, z)
    }

    override fun right(): Point {
        return Point3D(x + 1, y, z)
    }

    override fun top(): Point {
        return Point3D(x, y - 1, z)
    }

    override fun bottom(): Point {
        return Point3D(x, y + 1, z)
    }

    override fun front(): Point {
        return Point3D(x, y, z + 1)
    }

    override fun set(other: Point) : Point {
        if (other is Point3D) {
            x = other.x
            y = other.y
            z = other.z
        }
        return this
    }

    override fun add(other : Point) : Point {
        if (other is Point3D) {
            x += other.x
            y += other.y
            z += other.z
        }
        return this
    }

    override fun subtract(other : Point) : Point {
        if (other is Point3D) {
            x -= other.x
            y -= other.y
            z -= other.z
        }
        return this
    }

    override fun multiply(other: Point): Point {
        if (other is Point3D) {
            x *= other.x
            y *= other.y
            z *= other.z
        }
        return this
    }

    override fun divide(other: Point): Point {
        if (other is Point3D) {
            x /= other.x
            y /= other.y
            z /= other.z
        }
        return this
    }

    override fun invert() : Point {
        x = -x
        y = -y
        z = -z
        return this
    }

    override fun absolute(): Point {
        x = kotlin.math.abs(x)
        y = kotlin.math.abs(y)
        z = kotlin.math.abs(z)
        return this
    }

    override fun rotate(matrix: Matrix) : Point {
        val pointMatrix = matrix.multiply(toMatrix(this))
        val newPoint = fromMatrix(pointMatrix)
        set(newPoint)
        return this
    }

    private fun toMatrix(point : Point3D) : Matrix {
        return Matrix(listOf(
            listOf(point.x),
            listOf(point.y),
            listOf(point.z)
        ))
    }

    private fun fromMatrix(matrix : Matrix) : Point {
        return Point3D(
            matrix.matrix[0][0],
            matrix.matrix[1][0],
            matrix.matrix[2][0]
        )
    }

    override fun isLessOrEqual(other: Point) : Boolean {
        other as Point3D
        return x <= other.x &&
                y <= other.y &&
                z <= other.z
    }

    override fun getMinimalCoordinates(points : Set<Point>) : Point {
        val minCoordinates = Point3D(Int.MAX_VALUE, Int.MAX_VALUE, Int.MAX_VALUE)
        for (point in points) {
            if (point is Point3D) {
                if (point.x < minCoordinates.x) {
                    minCoordinates.x = point.x
                }
                if (point.y < minCoordinates.y) {
                    minCoordinates.y = point.y
                }
                if (point.z < minCoordinates.z) {
                    minCoordinates.z = point.z
                }
            }
        }
        return minCoordinates
    }

    override fun getMaximalCoordinates(points : Set<Point>) : Point {
        val maxCoordinates = Point3D(Int.MIN_VALUE, Int.MIN_VALUE, Int.MIN_VALUE)
        for (point in points) {
            if (point is Point3D) {
                if (point.x > maxCoordinates.x) {
                    maxCoordinates.x = point.x
                }
                if (point.y > maxCoordinates.y) {
                    maxCoordinates.y = point.y
                }
                if (point.z > maxCoordinates.z) {
                    maxCoordinates.z = point.z
                }
            }
        }
        return maxCoordinates
    }

    override fun rotations() : List<Matrix> {
        rotations.clear()
        yRotations()
        rotations.add(rotationMatrixAxisX(QUARTER))
        xRotations()
        rotations.add(rotationMatrixAxisX(THREE_QUARTER))
        return rotations
    }

    private fun xRotations() {
        for (i in 0 until 2) {
            rotations.add(rotationMatrixAxisX(HALF))
            zRotations()
        }
    }

    private fun yRotations() {
        for (i in 0 until 4) {
            rotations.add(rotationMatrixAxisY(QUARTER))
            zRotations()
        }
    }

    private fun zRotations() {
        for (i in 0 until 4) {
            rotations.add(rotationMatrixAxisZ(QUARTER))
        }
    }

    private fun rotationMatrixAxisX(angle : Int) : Matrix {
        return Matrix(
            listOf(
                listOf(1, 0, 0),
                listOf(0, cos(angle), -sin(angle)),
                listOf(0, sin(angle), cos(angle))
            )
        )
    }

    private fun rotationMatrixAxisY(angle : Int) : Matrix {
        return Matrix(
            listOf(
                listOf(cos(angle), 0, sin(angle)),
                listOf(0, 1, 0),
                listOf(-sin(angle), 0, cos(angle))
            )
        )
    }

    private fun rotationMatrixAxisZ(angle : Int) : Matrix {
        return Matrix(
            listOf(
                listOf(cos(angle), -sin(angle), 0),
                listOf(sin(angle), cos(angle), 0),
                listOf(0, 0, 1)
            )
        )
    }

}