package games.geniusSquare.game

import algorithmX.printer.Printer
import java.lang.IllegalArgumentException

open class GeniusSquareAllGames : GeniusSquareBase() {

    //attributes -----------------------------------------------------------------
    private val frequencyMap = HashMap<Int, Int>()
    private var min = Int.MAX_VALUE
    private var max = Int.MIN_VALUE
    private var minString = ""
    private var maxString = ""
    private var sum = 0
    private var count = 0

    //constants ------------------------------------------------------------------
    private val PRINT_DIFFICULTY = false
    private val GRANULITY = 1000
    private val NUMBER_OF_DICES = 7
    private val NUMER_OF_DICE_WALLS = 6
    private val MINIMAL_NUMBER = 1
    private val MAXIMAL_NUMBER = 6
    private val EMPTY_STRING = ""
    private val MINIMAL_CHARACTER = 'A'
    private val MAXIMAL_CHARACTER = 'F'
    private val MINIMAL_SOLUTION = "has minimal number of solutions:"
    private val MAXIMAL_SOLUTION = "has maximal number of solutions:"
    private val AVERAGE_SOLUTION = "average number of solutions is:"
    private val FREQUENCY_OF_SOLUTIONS = "frequency of number of solutions:"
    private val WRONG_DICES_COUNT = "Number of dices is not $NUMBER_OF_DICES"
    private val WRONG_DICE_SIDES_COUNT = "Dice has less then 1 or more" +
            " then $NUMER_OF_DICE_WALLS walls"
    private val WRONG_DICE_CHARACTER = "Dice contains character which is" +
            " not from range $MINIMAL_CHARACTER - $MAXIMAL_CHARACTER"
    private val WRONG_DICE_NUMBER = "Dice contains number which is" +
            " not from range $MINIMAL_NUMBER - $MAXIMAL_NUMBER"

    //methods ---------------------------------------------------------------------
    fun solveAllGamesWithSpecifiedDices(dices: List<List<Pair<Char, Int>>>,
                                        printer: Printer,
                                        allSolutions : Boolean,
                                        printSolution : Boolean,
                                        printResults : Boolean) {
        checkDices(dices)
        doSolveAllGames(dices, printer, allSolutions, printSolution, printResults)
    }

    private fun checkDices(dices: List<List<Pair<Char, Int>>>) {
        if (dices.size != NUMBER_OF_DICES) {
            throw IllegalArgumentException(WRONG_DICES_COUNT)
        }
        for (dice in dices) {
            if (dice.isEmpty() || dice.size > NUMER_OF_DICE_WALLS) {
                throw IllegalArgumentException(WRONG_DICE_SIDES_COUNT)
            }
            for (position in dice) {
                if (!position.first.isLetter()) {
                    throw IllegalArgumentException(WRONG_DICE_CHARACTER)
                }
                val char = position.first.toUpperCase()
                if (char < MINIMAL_CHARACTER || char > MAXIMAL_CHARACTER) {
                    throw IllegalArgumentException(WRONG_DICE_CHARACTER)
                }
                val number = position.second
                if (number < MINIMAL_NUMBER || number > MAXIMAL_NUMBER) {
                    throw IllegalArgumentException(WRONG_DICE_NUMBER)
                }
            }
        }
    }

    fun solveAllGames(printer: Printer,
                      allSolutions : Boolean,
                      printSolution : Boolean,
                      printResults : Boolean) {
        doSolveAllGames(DICES, printer, allSolutions, printSolution, printResults)
    }

    protected fun doSolveAllGames(dices : List<List<Pair<Char, Int>>>,
                                  printer: Printer,
                                  allSolutions : Boolean,
                                  printSolution : Boolean,
                                  printResults : Boolean) : Boolean {
        if (printResults) {
            resetStats()
        }
        for (i1 in dices[0].indices) {
            for (i2 in dices[1].indices) {
                for (i3 in dices[2].indices) {
                    for (i4 in dices[3].indices) {
                        for (i5 in dices[4].indices) {
                            for (i6 in dices[5].indices) {
                                for (i7 in dices[6].indices) {
                                    val numberOfSolutions = generateGameAndSolveIt(
                                        dices,
                                        listOf(i1, i2, i3, i4, i5, i6, i7),
                                        printer,
                                        allSolutions,
                                        printSolution,
                                        printResults,
                                        PRINT_DIFFICULTY)
                                    if (numberOfSolutions == 0) {
                                        return false
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        if (printResults) {
            printStats()
        }
        return true
    }

    private fun resetStats() {
        min = Int.MAX_VALUE
        minString = EMPTY_STRING
        max = Int.MIN_VALUE
        maxString = EMPTY_STRING
        count = 0
        sum = 0
        frequencyMap.clear()
    }

    override fun generateGameAndSolveIt(dices : List<List<Pair<Char, Int>>>,
                                        indices : List<Int>,
                                        printer : Printer,
                                        allSolutions : Boolean,
                                        printSolution : Boolean,
                                        printResults : Boolean,
                                        printDifficulty: Boolean) : Int {
        val numberOfSolutions = super.generateGameAndSolveIt(
            dices,
            indices,
            printer,
            allSolutions,
            printSolution,
            printResults,
            printDifficulty
        )
        if (printResults) {
            addToStats(dices, indices, numberOfSolutions)
        }
        return numberOfSolutions
    }

    private fun addToStats(dices : List<List<Pair<Char, Int>>>,
                           indices : List<Int>, numberOfSolutions : Int) {
        count++
        sum += numberOfSolutions
        if (numberOfSolutions < min){
            min = numberOfSolutions
            minString = printIndices(dices, indices)
        }
        if (numberOfSolutions > max){
            max = numberOfSolutions
            maxString = printIndices(dices, indices)
        }
        frequencyMap[numberOfSolutions / GRANULITY] =
            frequencyMap.getOrPut(numberOfSolutions / GRANULITY, { 0 }) + 1
    }

    private fun printStats() {
        println()
        println("$minString $MINIMAL_SOLUTION $min")
        println("$maxString $MAXIMAL_SOLUTION $max")
        println("$AVERAGE_SOLUTION ${(sum.toDouble() / count.toDouble())}")
        println()
        println(FREQUENCY_OF_SOLUTIONS)
        for ((key, value) in frequencyMap.entries.sortedBy { it.key }) {
            println("$value games has from ${(key * GRANULITY)} " +
                    "to ${(key * GRANULITY + GRANULITY - 1)} solutions")
        }
    }

}