package games.ubongo.game

import algorithmX.AlgorithmX
import algorithmX.name.Name
import algorithmX.printer.NamePrinter
import algorithmX.printer.Printer
import cubes.Formalisation
import cubes.Puzzle
import cubes.Shape
import cubes.Task
import cubes.point.Point
import cubes.point.Point3D
import cubes.Difficulty
import games.ubongo.Ubongo
import java.lang.IllegalArgumentException

class UbongoGenerator {

    //attributes -----------------------------------------------------------------------------
    private val points = mutableSetOf<Point>()
    private val allPositions = mutableSetOf<Point>()
    private var shapes = mutableSetOf<Shape>()
    private var difficulty = 0

    //constants ------------------------------------------------------------------------------
    private val WRONG_LEVEL = "wrong difficulty level parameter"
    private val NAME = ""
    private val MAXIMAL_WIDTH = 5
    private val MAXIMAL_HEIGHT = 5
    private val MINIMAL_LEVEL = 1
    private val MAXIMAL_LEVEL = 4
    private val MINIMAL_POINTS_NUMBER = 15
    private val LEVEL0 = 1.0
    private val LEVEL1 = 0.015
    private val LEVEL2 = 0.001
    private val LEVEL3 = 0.00001
    private val LEVEL4 = 0.0000001
    private val ALL_SOLLUTIONS = true
    private val ROTATION_MATRICES = Point3D().rotations()
    private val PRINTER = NamePrinter()
    private val SHAPES = mutableSetOf(
        Ubongo.shape2I,
        Ubongo.shape3I,
        Ubongo.shape3V,
        Ubongo.shape4I,
        Ubongo.shape4L,
        Ubongo.shape4O,
        Ubongo.shape4S,
        Ubongo.shape4T,
        Ubongo.shape4V1,
        Ubongo.shape4V2,
        Ubongo.shape4V3,
        Ubongo.shape5L,
        Ubongo.shape5V,
        Ubongo.shape5Y,
        Ubongo.shape5LV1,
        Ubongo.shape5LV2,
        Ubongo.shape5W1
    )

    //methods --------------------------------------------------------------------------------
    fun generateAndSolveGame(printer: Printer,
                             difficultyLevel: Int,
                             allSolutions : Boolean,
                             printSolution : Boolean,
                             printResult : Boolean,
                             printDifficulty: Boolean) {
        checkDifficulty(difficultyLevel)
        generateGame(difficultyLevel)
        solveGame(printer,
            allSolutions,
            printSolution,
            printResult,
            printDifficulty
        )
    }

    private fun checkDifficulty(difficultyLevel: Int) {
        if (difficultyLevel < MINIMAL_LEVEL || difficultyLevel > MAXIMAL_LEVEL) {
            throw IllegalArgumentException(WRONG_LEVEL)
        }
    }

    private fun generateGame(difficultyLevel: Int) {
        generateAllPoints()
        while (true) {
            checkConditions()
            generatePoint()
            solveGameRepeatable()
            findNonRepeatingSolution()
            checkDifficultyOfSolution()
            if (difficulty == difficultyLevel) {
                break
            }
        }
    }

    private fun generateAllPoints() {
        allPositions.clear()
        for (x in 0 until MAXIMAL_WIDTH) {
            for (y in 0 until MAXIMAL_HEIGHT) {
                allPositions.add(Point3D(x, y))
            }
        }
    }

    private fun checkConditions() {
        if (allPositions.size <= MINIMAL_POINTS_NUMBER) {
            generateAllPoints()
            points.clear()
        }
    }

    private fun generatePoint() {
        while (true) {
            val point = allPositions.random()
            if (isAdjacentPoint(point)) {
                allPositions.remove(point)
                points.add(point)
                points.add(point.front())
                break
            }
        }
    }

    private fun isAdjacentPoint(point: Point) : Boolean {
        return points.isEmpty() ||
                points.contains(point.left()) ||
                points.contains(point.right()) ||
                points.contains(point.top()) ||
                points.contains(point.bottom())
    }

    private fun solveGameRepeatable(){
        val task = Task(
            Puzzle(points.map { it.copy() }.toSet()).normalise(),
            repeatableShapes = SHAPES
        )
        val pair = Formalisation().createMatrixAndNamesFromTask(task, false)
        PRINTER.reset()
        AlgorithmX(
            pair.first,
            pair.second,
            PRINTER,
            ALL_SOLLUTIONS
        ).solve()
    }

    private fun findNonRepeatingSolution() {
        shapes.clear()
        val solutions = PRINTER.solutions
        for (solution in solutions) {
            var repetition = false
            val solutionShapes = solutionToShapes(solution)
            for (shape in solutionShapes) {
                repetition = rotateShape(shape)
                if (repetition) {
                    break
                }
            }
            if (!repetition) {
                break
            }
            shapes.clear()
        }
    }

    private fun solutionToShapes(solution : List<List<Name>>) : List<Shape> {
        val shapes = mutableListOf<Shape>()
        for (names in solution) {
            shapes.add(Shape(
                NAME,
                names.map{ (it as Point).copy() }.toSet()).normalise()
            )
        }
        return shapes
    }

    private fun rotateShape(shape: Shape) : Boolean {
        val shapeCopy = shape.copy()
        for (rotationMatrix in ROTATION_MATRICES) {
            shapeCopy.rotate(rotationMatrix).normalise()
            if (shapes.contains(shapeCopy.copy())) {
                return true
            }
        }
        shapes.add(shapeCopy)
        return false
    }

    private fun checkDifficultyOfSolution() {
        val shapes = mutableSetOf<Shape>()
        for (shape in this.shapes) {
            shapes.add(Shape(shape.toString(), shape.points))
        }
        val task = Task(Puzzle(points), shapes = shapes)
        val difficultyTemp = Difficulty().calculateDifficulty(task)
        difficulty = difficultyToLevel(difficultyTemp)
    }

    private fun difficultyToLevel(difficulty: Double) : Int {
        when (difficulty) {
            0.0 -> return 0
            in LEVEL1..LEVEL0 -> return 0
            in LEVEL2..LEVEL1 -> return 1
            in LEVEL3..LEVEL2 -> return 2
            in LEVEL4..LEVEL3 -> return 3
            else -> return 4
        }
    }

    private fun solveGame(printer: Printer,
                          allSolutions : Boolean,
                          printSolution : Boolean,
                          printResult : Boolean,
                          printDifficulty: Boolean) {
        val shapes = mutableSetOf<Shape>()
        for (shape in this.shapes) {
            shapes.add(Shape(shape.toString(), shape.points))
        }
        val task = Task(Puzzle(points), shapes = shapes)
        UbongoBase().solveSpecifiedGame(
            task,
            printer,
            allSolutions,
            printSolution,
            printResult,
            printDifficulty
        )
    }

}