package games.ubongo.printer

import algorithmX.name.Name
import algorithmX.printer.Printer
import cubes.Shape
import cubes.point.Point
import cubes.point.Point3D
import games.Colour
import org.lwjgl.glfw.GLFW.*
import org.lwjgl.glfw.GLFWVidMode
import org.lwjgl.opengl.GL.*
import org.lwjgl.opengl.GL41.*
import org.lwjgl.system.MemoryStack
import org.lwjgl.system.MemoryUtil.NULL
import java.util.*
import kotlin.math.tan

/**
 * GUI instruction
 *
 * X axis rotation      A
 * Y axis rotation      S
 * Z axis rotation      D
 * zoom in              +
 * zoom out             -
 * next solution        RIGHT
 * previous solution    LEFT
 * recolour shapes      SPACE
 * animation of build   ENTER
 * remove shape         LEFT CLICK
 * undo last remove     RIGHT CLICK
 */

open class GLUbongoPrinter : Printer {

    //attributes ----------------------------------------------------------------------
    private val rnd = Random()
    private var window  = 0L
    private var angleX = 0.0
    private var angleY = 0.0
    private var angleZ = 0.0
    private var distance = 20.0
    private var xPosition = 0
    private var yPosition = 0
    private var solutionsIndex = 0
    private var colourIndex = 0
    private var animationIndex = 0
    private var centralisationCalculated = false
    private var center = Point3D()
    private var offset = Point3D()
    private var colour = Colour()
    private var colours = mutableListOf<Colour>()
    private val usedColours = mutableMapOf<String, Colour>()
    private val deletedColours = mutableListOf<Colour>()
    override val solutions = mutableListOf<List<List<Name>>>()
    
    //constants -----------------------------------------------------------------------
    private val ZERO = 0.0
    private val ONE = 1.0
    private val TWO = 2.0
    private val ANGLE = 5.0
    private val DISTANCE = -20.0
    private val MIN_DISTANCE = -12.0
    private val MAX_DISTANCE = -30.0
    private val MAX_COLOUR_VALUE = 0.9
    private val MIN_COLOUR_VALUE = 0.1
    private val LINE_WIDTH = 3f
    private val LINE_OFFSET = 1f
    private val BACKGROUND = 1f
    private val COLOUR_DEFUALT = 0f
    private val SLEEP_TIME = 500L
    private val FIRST_INDEX = 0
    private val WRONG_INDEX = -1
    protected val NEXT = 1
    protected val PREVIOUS = -1
    private val WIDTH = 800
    private val HEIGHT = 800
    private val PIXEL_SIZE = 1
    private val FRUSTRUM_N = 1000.0
    private val FULL_ANGLE = 360
    private val PART_ANGLE = 60
    private val COLOUR_SIZE = 3
    private val X = 0
    private val Y = 1
    private val Z = 2
    private val A = 0
    private val D = 3
    private val R = 0
    private val G = 1
    private val B = 2

    private val SPECIFIED_CLOURS = true

    private val FAILED_INIT = "Unable to initialize GLFW"
    private val FAILED_WINDOW = "Failed to create the GLFW window"
    private val WINDOW_DESCRIPTION = "    -    number of solutions : "
    protected open val WINDOW_NAME = "Ubingo"

    private val LINING_COLOUR = Colour(ZERO, ZERO, ZERO)

    private val POINTS = listOf(
        listOf(-1.0, 1.0, 1.0),
        listOf(1.0, 1.0, 1.0),
        listOf(1.0, -1.0, 1.0),
        listOf(-1.0, -1.0, 1.0),
        listOf(-1.0, 1.0, -1.0),
        listOf(1.0, 1.0, -1.0),
        listOf(1.0, -1.0, -1.0),
        listOf(-1.0, -1.0, -1.0)
    )
    private val WALLS = listOf(
        listOf(0, 1, 2, 3),
        listOf(4, 5, 6, 7),
        listOf(0, 1, 5, 4),
        listOf(3, 2, 6, 7),
        listOf(0, 3, 7, 4),
        listOf(1, 2, 6, 5)
    )

    //methods -------------------------------------------------------------------------
    override fun reset() {
        super.reset()
        usedColours.clear()
        deletedColours.clear()
        colours.clear()
        colours.addAll(Colour.COLOURS)
        angleX = ZERO
        angleY = ZERO
        angleZ = ZERO
        solutionsIndex = FIRST_INDEX
        colourIndex = FIRST_INDEX
        distance = DISTANCE
        centralisationCalculated = false
    }

    override fun add(solution: List<List<Name>>) {
        super.add(solution)
        if (!centralisationCalculated) {
            calculateCentre()
            setAnimationEnd()
        }
    }

    private fun calculateCentre() {
        centralisationCalculated = true
        val points = pointsOfSolution(solutions[FIRST_INDEX])
        val minCoordinates = Point.instance.getMinimalCoordinates(points)
        val maxCoordinates = Point.instance.getMaximalCoordinates(points)
        center = maxCoordinates.subtract(minCoordinates) as Point3D
    }

    private fun pointsOfSolution(solution: List<List<Name>>) : Set<Point> {
        val points = mutableSetOf<Point>()
        for (row in solution) {
            for (name in row) {
                if (name is Point) {
                    points.add(name)
                }
            }
        }
        return points
    }

    override fun print() {
        if (solutions.isNotEmpty()) {
            init()
            render()
            while (!glfwWindowShouldClose(window)) {
                glfwWaitEvents()
            }
        }
    }

    private fun init() {
        initGL()
        windowGL()
        mouseCallbackGL()
        keyCallbackGL()
        threadGL()
        displayGL()
        setupGL()
    }

    private fun initGL() {
        if (!glfwInit()) {
            throw IllegalStateException(FAILED_INIT)
        }
        glfwDefaultWindowHints()
        glfwWindowHint(GLFW_VISIBLE, GLFW_FALSE)
        glfwWindowHint(GLFW_RESIZABLE, GLFW_TRUE)
    }

    private fun windowGL() {
        window = glfwCreateWindow(
            WIDTH, HEIGHT,
            WINDOW_NAME + WINDOW_DESCRIPTION + solutions.size,
            NULL, NULL
        )
        if (window == NULL){
            throw RuntimeException(FAILED_WINDOW)
        }
    }

    private fun mouseCallbackGL() {
        glfwSetCursorPosCallback(window) { _ : Long, x : Double,
                                           y : Double ->
            xPosition = x.toInt()
            yPosition = (HEIGHT - y).toInt()
        }
        glfwSetMouseButtonCallback(window) { _ : Long, button: Int,
                                             action: Int, _: Int ->
            if (action == GLFW_PRESS) {
                mouseReact(button)
                render()
            }
        }
    }

    private fun mouseReact(button: Int) {
        when (button) {
            GLFW_MOUSE_BUTTON_LEFT -> addDeletedColour()
            GLFW_MOUSE_BUTTON_RIGHT -> removeDeletedColor()
        }
    }

    private fun addDeletedColour() {
        val colourArray = FloatArray(COLOUR_SIZE){ COLOUR_DEFUALT }
        glfwSwapBuffers(window)
        glReadPixels(xPosition, yPosition, PIXEL_SIZE, PIXEL_SIZE, GL_RGB, GL_FLOAT, colourArray)
        glfwSwapBuffers(window)
        addColor(Colour(colourArray[R], colourArray[G], colourArray[B]))
    }

    protected open fun addColor(colour: Colour) {
        if (usedColours.values.contains(colour)) {
            deletedColours.add(colour)
        }
    }

    private fun removeDeletedColor() {
        if (deletedColours.isNotEmpty()) {
            deletedColours.removeAt(deletedColours.size - 1)
        }
    }

    private fun keyCallbackGL() {
        glfwSetKeyCallback(window) { window: Long, key: Int,
                                     _: Int, action: Int, _: Int ->
            if (action != GLFW_RELEASE) {
                keyReact(window, key)
                render()
            }
        }
    }

    protected open fun keyReact(window : Long, key : Int) {
        when (key) {
            GLFW_KEY_ESCAPE -> glfwSetWindowShouldClose(window, true)
            GLFW_KEY_LEFT -> changeSolutionIndex(PREVIOUS)
            GLFW_KEY_RIGHT -> changeSolutionIndex(NEXT)
            GLFW_KEY_KP_SUBTRACT -> changeDistance(PREVIOUS)
            GLFW_KEY_KP_ADD -> changeDistance(NEXT)
            GLFW_KEY_ENTER -> animate()
            GLFW_KEY_SPACE -> recolour()
            GLFW_KEY_A -> angleX += ANGLE
            GLFW_KEY_S -> angleY += ANGLE
            GLFW_KEY_D -> angleZ += ANGLE
        }
    }

    private fun setAnimationBegin(){
        animationIndex = FIRST_INDEX
    }

    private fun setAnimationEnd() {
        animationIndex = solutions[FIRST_INDEX].size
    }

    protected fun changeSolutionIndex(difference : Int) {
        solutionsIndex = (solutionsIndex + difference) % solutions.size
        if (solutionsIndex == WRONG_INDEX) {
            solutionsIndex = solutions.size - 1
        }
        setAnimationEnd()
    }

    private fun changeAnimationIndex(difference: Int) {
        animationIndex = (animationIndex + difference)
            .coerceAtMost(solutions[FIRST_INDEX].size)
            .coerceAtLeast(FIRST_INDEX)
    }

    protected fun changeDistance(difference: Int) {
        distance = (distance + difference)
            .coerceAtMost(MIN_DISTANCE)
            .coerceAtLeast(MAX_DISTANCE)
    }

    protected fun recolour() {
        colours.shuffle()
        usedColours.clear()
        deletedColours.clear()
        render()
    }

    private fun threadGL() {
        MemoryStack.stackPush().use { stack ->
            val pWidth = stack.mallocInt(ONE.toInt())
            val pHeight = stack.mallocInt(ONE.toInt())
            glfwGetWindowSize(window, pWidth, pHeight)
            val vidmode: GLFWVidMode = glfwGetVideoMode(glfwGetPrimaryMonitor())!!
            glfwSetWindowPos(
                window,
                (vidmode.width() - pWidth[ZERO.toInt()]) / TWO.toInt(),
                (vidmode.height() - pHeight[ZERO.toInt()]) / TWO.toInt()
            )
        }
    }

    private fun displayGL() {
        glfwMakeContextCurrent(window)
        glfwShowWindow(window)
        createCapabilities()
    }

    private fun setupGL() {
        glClearColor(BACKGROUND, BACKGROUND, BACKGROUND, BACKGROUND)
        glEnable(GL_DEPTH_TEST)
        val d: Double = tan(PART_ANGLE * Math.PI / FULL_ANGLE)
        val r: Double = ONE * WIDTH / HEIGHT
        glViewport(ZERO.toInt(), ZERO.toInt(), WIDTH, HEIGHT)
        glMatrixMode(GL_PROJECTION)
        glFrustum(-d * r, d * r, -d, +d, ONE, FRUSTRUM_N)
        glMatrixMode(GL_MODELVIEW)
        glClearDepth(ONE)
    }

    protected fun animate() {
        setAnimationBegin()
        while (animationIndex < solutions[FIRST_INDEX].size) {
            render()
            changeAnimationIndex(NEXT)
            Thread.sleep(SLEEP_TIME)
        }
        render()
    }

    private fun render() {
        glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT)
        transform()
        renderSolution()
        glfwSwapBuffers(window)
    }

    private fun transform() {
        glLoadIdentity()
        glTranslated(ZERO, ZERO, distance)
        glRotated(angleX, ONE, ZERO, ZERO)
        glRotated(angleY, ZERO, ONE, ZERO)
        glRotated(angleZ, ZERO, ZERO, ONE)
    }

    private fun renderSolution() {
        val solution = solutions[solutionsIndex]
        for ((index, row) in solution.withIndex()) {
            if (index == animationIndex) {
                return
            }
            selectColour(row)
            if (deletedColours.contains(colour)) {
                continue
            }
            drawShape(row)
        }
    }

    private fun selectColour(row: List<Name>) {
        val keyName = selectName(row)
        if (!usedColours.contains(keyName)) {
            usedColours[keyName] = generateColour(keyName)
        }
        colour = usedColours[keyName]!!
    }

    private fun selectName(row : List<Name>) : String {
        var keyName = row.toString()
        for (name in row) {
            if (name is Shape) {
                keyName = name.name
            }
        }
        return keyName
    }


    protected open fun generateColour(name : String) : Colour {
        if (SPECIFIED_CLOURS && colourIndex < colours.size) {
            colour = colours[colourIndex]
            colourIndex++
            return colour
        }
        return randomColor()
    }

    private fun randomColor() : Colour {
        return Colour(
            rnd.nextDouble() * (MAX_COLOUR_VALUE - MIN_COLOUR_VALUE) + MIN_COLOUR_VALUE,
            rnd.nextDouble() * (MAX_COLOUR_VALUE - MIN_COLOUR_VALUE) + MIN_COLOUR_VALUE,
            rnd.nextDouble() * (MAX_COLOUR_VALUE - MIN_COLOUR_VALUE) + MIN_COLOUR_VALUE
        )
    }

    private fun drawShape(row: List<Name>) {
        for (name in row) {
            if (name is Point3D) {
                offset = name
                drawCube()
            }
        }
    }

    private fun drawCube() {
        for (wallIndex in WALLS.indices) {
            drawWall(wallIndex)
        }
    }

    private fun drawWall(wallIndex : Int) {
        glPolygonMode(GL_FRONT_AND_BACK, GL_LINE)
        glLineWidth(LINE_WIDTH)
        drawLining(wallIndex)
        glPushAttrib(GL_ALL_ATTRIB_BITS)
        glEnable(GL_POLYGON_OFFSET_FILL)
        glPolygonOffset(LINE_OFFSET, LINE_OFFSET)
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL)
        drawSquare(wallIndex)
        glPopAttrib()
    }

    private fun drawSquare(wallIndex : Int) {
        glColor3d(colour.r, colour.g, colour.b)
        glBegin(GL_QUADS)
        for (pointIndex in A..D) {
            drawPoint(wallIndex, pointIndex)
        }
        glEnd()
    }

    private fun drawLining(wallIndex : Int) {
        glColor3d(LINING_COLOUR.r, LINING_COLOUR.g, LINING_COLOUR.b)
        glBegin(GL_LINE_LOOP)
        for (pointIndex in A..D) {
            drawPoint(wallIndex, pointIndex)
        }
        glEnd()
    }

    private fun drawPoint(wallIndex : Int, pointIndex : Int) {
        glVertex3d(
            POINTS[WALLS[wallIndex][pointIndex]][X] + offset.x * TWO - center.x,
            POINTS[WALLS[wallIndex][pointIndex]][Y] - offset.y * TWO + center.y,
            POINTS[WALLS[wallIndex][pointIndex]][Z] + offset.z * TWO - center.z
        )
    }

}