/*
	=============================================================================
	Name: correctFileSpectrum.c
	Author: F. Kundracik
	Description: Program pre korekciu spektra ziarenia svetelnho dmu
	Date of latest update: 26.5.2020
	26.5.2020
        - uz nie priblizny vypocet rozvojom do eliptickych integralov
        - vypocet integralu adaptivnou lichobeznikovou metodou
	24.6.2019
        - !!!!! nahradenie vypoctu eliptickych integralov Simpsonom, rozvoj bol zly
        - uprava pre spracovanie suboru spektra
	=============================================================================
	Run:		./correctFileSpectrum <filename>

	Compile: 	gcc correctFileSpectrum.c -o correctFileSpectrum -lm
	=============================================================================
*/

// system header files
#include 	<math.h>
#include 	<stdio.h>
#include 	<stdlib.h>
#include 	<string.h>
#include 	<unistd.h>

// recognise OS
#if _WIN32
  	// OS = Windows
	const int os = 1;
#else
#ifdef __linux__
	// OS = Linux
   	const int os = 2;
//#else
//#ifdef (__APPLE__) && defined(__MACH__)
//	// OS = Apple
//  const int os = 3;
#else
	// unknown OS
   	const int os = 0;
//#endif
#endif
#endif

typedef enum{
   LIN,
   LOG
} tScale ;

double LAMBDA_0=0.5; //micrometres
double TAU_A0=0.4;
double ALPHA=1.;
double ASY=0.9;

double HA=2000;  //m - characteristic height of the aerosol layer
double HR=8000; //m - characteristic height of the molecular atmosphere

double D=8000;// in m - distance to the lighht source
//double MIN_LAMBDA=0.300;
//double MAX_LAMBDA=0.800;
double Q_0=1e5;
double z_deg = 60.;
double z_rad;
double z_max=M_PI;  //limit of integration (blocking the light horizontally escaping the city)

double lambda_nm, lambda_um;

// used functions
double f(double x);
double g();
double TauA();
double TauR();
double T(double z0);

double calcResult();
double AdaptiveIntegral(double xmin,double xmax,double (*Function)(double),tScale scale, double step0,double precision);


int main(int argc, char *argv[]) {

  FILE  *fw,*fr,*fd; // pointers to files with input and output data
	double  value, res;


	char fileinname[1024];
	char fileoutname[1024];
	char configfile[1024]="correctconfig.txt";
	char *position;

	char line[1024];

  if(argc!=2)
  {
      printf("Usage: correctFileSpectrum <spectrum filename>\nParameters are stored in 'correctconfig.txt'-file\n");
      system("pause");
      return 1;
  }

  strcpy(fileinname,argv[1]);
  strcpy(fileoutname,argv[1]);
  position=strstr(fileoutname,".txt");
  *position=0;
  strcat(fileoutname,"_cor.txt");
  //printf("%s\n",fileoutname);
  //system("pause");



  if ((fw = fopen(fileoutname, "w")) == NULL) {
		printf("Unable to open the file: %s\n",fileoutname);
		return 1;
	}

  if ((fr = fopen(fileinname, "r")) == NULL) {
		printf("Unable to open the file: %s\n",fileinname);
		fclose(fw);
		return 1;
	}
  if ((fd = fopen(configfile, "r")) == NULL) {
		printf("Unable to open the file: %s\n",configfile);
		fclose(fw);
		fclose(fr);
		return 1;
  }
  printf("Reading the parameters...\n");

  fgets(line, 1000, fd);
  sscanf(line,"%lf",&z_deg);
  z_rad=z_deg/180.0*M_PI;
  printf("Zenith angle of the spectrometer: %f [deg]\n",z_deg);

  fgets(line, 1000, fd);
  sscanf(line,"%lf",&D);
  D*=1000; // to m
  printf("Distance to the light source: %f [km]\n",D/1000);

  fgets(line, 1000, fd);
  sscanf(line,"%lf",&z_max);
  z_max*=M_PI/180.0; // to rad
  printf("Shielding zenith angle: %f [deg]\n",z_max/M_PI*180);

  fgets(line, 1000, fd);
  sscanf(line,"%lf",&TAU_A0);
  printf("Aerosol optical depth at reference wavelength: %f [-]\n",TAU_A0);

  fgets(line, 1000, fd);
  sscanf(line,"%lf",&LAMBDA_0);
  printf("Reference wavelength: %f [mkm]\n",LAMBDA_0);

  fgets(line, 1000, fd);
  sscanf(line,"%lf",&ALPHA);
  printf("Angstrom exponent: %f [-]\n",ALPHA);

  fgets(line, 1000, fd);
  sscanf(line,"%lf",&ASY);
  printf("Coefficient of asymmetry: %f [-]\n",ASY);


  //fgets(line, 1000, fd);
  //sscanf(line,"%lf",&A);
  //printf("Particle size distribution function (A-parameter): %f [-]\n",A);

  //fgets(line, 1000, fd);
  //sscanf(line,"%lf",&B);
  //printf("Particle size distribution function (B-parameter): %f [1/mkm]\n",B);

  fgets(line, 1000, fd);
  sscanf(line,"%lf",&HA);
  HA*=1000;
  printf("Characteristic height of the aerosol layer: %f [km]\n",HA/1000);

  fgets(line, 1000, fd);
  sscanf(line,"%lf",&HR);
  HR*=1000;
  printf("Characteristic height of the molecular atmosphere: %f [km]\n",HR/1000);

  fgets(line, 1000, fd);
  sscanf(line,"%lf",&Q_0);
  printf("Scaling factor of the corrected spectrum: %g\n",Q_0);


  // START main loop that calculates correction

  fprintf(fw,"# --------------------------------------------------------------------\n");
  fprintf(fw,"# Spectrum corrected by 'correctFileSpectrum'-program\n# Parameters:\n");
  fprintf(fw,"# Zenith angle of the spectrometer: %f [deg]\n",z_deg);
  fprintf(fw,"# Distance to the light source: %f [km]\n",D/1000);
  fprintf(fw,"# Shielding zenith angle: %f [deg]\n",z_max/M_PI*180);
  fprintf(fw,"# Aerosol optical depth at reference wavelength: %f [-]\n",TAU_A0);
  fprintf(fw,"# Reference wavelength: %f [mkm]\n",LAMBDA_0);
  fprintf(fw,"# Angstrom exponent: %f [-]\n",ALPHA);
  fprintf(fw,"# Coefficient of asymmetry: %f [-]\n",ASY);
  fprintf(fw,"# Characteristic height of the aerosol layer: %f [km]\n",HA/1000);
  fprintf(fw,"# Characteristic height of the molecular atmosphere: %f [km]\n",HR/1000);
  fprintf(fw,"# Scaling factor of the corrected spectrum: %g [-]\n",Q_0);
  fprintf(fw,"# --------------------------------------------------------------------\n");

      while (fgets(line, 1000, fr) !=NULL) {
          if(line[0]=='#') //comment line
          {
              fprintf(fw,"%s",line);
              continue;
          }
          if(sscanf(line,"%lf%lf",&lambda_nm,&value)==2)
          {
                lambda_um=lambda_nm/1000.0;
                res = calcResult();
                fprintf(fw,"%.0f\t%.3e\n",lambda_nm,value/res);
          }

    }


  if (fclose(fw) == EOF) {
		printf("Unable to close the file: %s\n",fileoutname);
		return 1;
	}
  if (fclose(fr) == EOF) {
		printf("Unable to close the file: %s\n",fileinname);
		return 1;
	}
  if (fclose(fd) == EOF) {
		printf("Unable to close the file: %s\n",configfile);
		return 1;
	}
  printf("'%s'-file successfully written, finishing.\n",fileoutname);
  lambda_um=0.6;
  double c600=calcResult();
  lambda_um=0.45;
  double c450=calcResult();
  printf("Suppression 450nm/600nm: %f\n",c450/c600);

  system("pause");
  return 0;
}


double T(double z0)
{
   //return 1; //pre testovacie vypocty s T=1
   double tmp;
   double ksi=D/(tan(z0)+tan(z_rad));
   tmp= exp(-(TauR()+TauA()-TauR()*exp(-ksi/HR)-TauA()*exp(-ksi/HA))*(1/cos(z0)+1/cos(z_rad)));
   //printf("%f \t%f\t%f\t%g\n",lambda_um,z0,ksi,tmp);
   return tmp;
}

double ksca(double theta,double h)
{
    return TauR()*exp(-h/HR)/(4*M_PI*HR)*3.0/4.0*(1+cos(theta)*cos(theta)) + TauA()*exp(-h/HA)/(4*M_PI*HA)*(1-g()*g())/pow((1+g()*g()-2*g()*cos(theta)),3.0/2.0);
}


double f(double z0)
{
        double theta=M_PI-z_rad-z0;
        double t=T(z0);
        double ksi=D/(tan(z_rad)+tan(z0));
        double k=ksca(theta,ksi);
        //printf("%f     %f      %f\n",z0,t,k);
        return t*k;
}


double g() {
  return ASY;
}


double TauA() {
  return TAU_A0*pow(lambda_um/LAMBDA_0, -ALPHA);
}

double TauR() {
    //return 0;
    return 0.00879*pow(lambda_um, -4.09);
}

double calcResult() {
    double res;
    //double tmp=AdaptiveIntegral(-z_rad, M_PI/2, f, LIN, 0.1, 0.0001); //bez shieldingu
    double zz=z_rad;
    if(z_max<z_rad) zz=z_max;
    double tmp=AdaptiveIntegral(-zz, z_max, f, LIN, 0.1, 0.0001); //so shieldingom
    double tmp2=tmp/cos(z_rad)/D;
    res = 1.0/Q_0*tmp2;
    //printf("%f\t%g\t%g\n",lambda_um,tmp,tmp2);
    return res;
}

/**********************************************************************************/
/*  Function AdaptiveIntegral: returns the value of the integral of the Function*/
/* between xmin and xmax calculated  trapezian method  with free step length*/
/* Linear or Logarithmic steps can be used. Initial step and desired relative precision can be set*/
/***********************************************************************************/
double AdaptiveIntegral(double xmin,double xmax,double (*Function)(double),tScale scale, double step0,double precision)
{
    double oldresult=0, newresult=1e99;
    double x,xold;
    double step;
    double oldvalue,newvalue;

    step=step0;
    while(fabs(oldresult-newresult)>precision*fabs(newresult))  //adaptive cycle
    {
        oldresult=newresult;
        newresult=0;
        switch(scale)
        {
            case LIN: step=step/2.0;
                break;
            case LOG: step=sqrt(step);
                break;
        }
        x=xmin;
        xold=xmin;
        oldvalue=Function(xold);
        while(x<xmax) //integration cycle
        {
           switch(scale)
            {
                case LIN: x+=step;
                break;

                case LOG: x*=step;
                break;
            }
            if(x>xmax) x=xmax;
            newvalue=Function(x);
            newresult+=(oldvalue+newvalue)*(x-xold)/2.0; //trapesian method of integration with free step
            xold=x;
            oldvalue=newvalue;
        }
        //printf("                           %.6f, %.6e, %.6e\n",step, oldresult,newresult);
    }
    //printf("-------   %.6e\n",newresult);
    return newresult;
  }
