package application;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Random;

/**
 * @author Michal Pazmany
 * @version 1.1
 *
 * Trieda Sudoku ma v sebe celu logiku hry.
 */
public class Sudoku {

	/**
	 * Rozmer sudoku.
	 */
	static int N = Main.rows;
	/**
	 * Rozmer boxu zvyrazneny hrubymi ciarami (bezne sudoku BxB = 3x3).
	 */
	static int B = Main.B;

	/**
	 * Pomocka na zobrazenie.
	 */
	static String pomocka;

	/**
	 * Inicializacia generatora nahodnych hodnot.
	 */
	static Random rand = new Random();

	/**
	 * Inicializuje dvojrozmerne pole integerov na jednoduchu interpretaciu Hry.
	 * 
	 * @param mriezka
	 *            Dvojrozmerne pole.
	 */
	public static void inicializujSudokuBunky(int mriezka[][]) {

		for (int i = 0; i < N; i++)
			for (int j = 0; j < N; j++)
				mriezka[i][j] = 0;

	}

	/**
	 * Generuje nahodne cislo z rozsahu min a max vratane.
	 * 
	 * @param min
	 *            Minimalna hodnota.
	 * @param max
	 *            Maximalna hodnota.
	 * @return Vrati nahodne cislo medzi min a max vratane.
	 */
	private static int getRandom(int min, int max) {
		return rand.nextInt((max - min) + 1) + min;
	}

	/**
	 * Do vstupneho 2d pola generuje pocetCisiel nahodnych Cisiel tak, ze hlada
	 * policka s hodnotou 0 a snazi sa vzdy doplnit nahodnu cifru aby bolo sudoku
	 * platne podla pravidiel sudoku. Ak sa mu nepodari do 12 pokusov doplnit cislo
	 * na nahodne vybrane policko, presunie sa na ine policko. Zacina s nami
	 * dosadenym polom plnym 0;
	 * 
	 * @param mriezka
	 *            Vstupne 2d pole integerov.
	 * @param pocetCisiel
	 *            Pocet cisiel na pridanie.
	 */
	public static void pridajCisla(int mriezka[][], int pocetCisiel) {
		int cisloBunky, riadok, stlpec, pocetPokusovBunka;
		boolean platneSudoku;

		while (pocetCisiel != 0) {
			pocetPokusovBunka = 12;
			do {
				cisloBunky = getRandom(0, N * N - 1);
				riadok = cisloBunky / N;
				stlpec = cisloBunky - riadok * N;

			} while (mriezka[riadok][stlpec] != 0);

			do {
				mriezka[riadok][stlpec] = getRandom(1, N);

				platneSudoku = jeSudokuPlatne(mriezka);
				pocetPokusovBunka--;
			} while (!platneSudoku && (pocetPokusovBunka != 0));

			if (platneSudoku) {
				pocetCisiel--;
			} else {
				mriezka[riadok][stlpec] = 0;
			}
		}
	}

	/**
	 * Odoberie zo vstupneho 2d pola integerov pocetCisiel tak, ze generuje nahodne
	 * pozicie a k je cislo rozne od 0, tak ho vymaze a zoberie cislo na dalsej
	 * nahodnej pozicii. Zadavame ako argument iba plne pole cisiel roznych od 0.
	 * 
	 * @param mriezka 	Vstupne 2d pole
	 * @param pocetCisiel	Pocet cisiel na odobranie.
	 */
	public static void odoberCisla(int mriezka[][], int pocetCisiel) {

		int cisloBunky, riadok, stlpec;

		while (pocetCisiel != 0) {
			// vygeneruj nahodny riadok a stlpec
			cisloBunky = getRandom(0, N * N - 1);
			riadok = cisloBunky / N;
			stlpec = cisloBunky - riadok * N;

			if (mriezka[riadok][stlpec] != 0) {
				pocetCisiel--;
				mriezka[riadok][stlpec] = 0;
			}
		}
	}

	/**
	 * Generuje platne zadanie hry sudoku. Zacina s nami zadanym polom, ktore naplni
	 * cislami 0. Najskor prida pocetCisielPridat cisiel nahodne pomocou metody
	 * pridajCisla, aby sme backtrackingom nedostali vzdy prve vyhovujuce rovnake
	 * riesenie. Potom sa sudoku vyriesi metodou ries. Vznikne nam hotove sudoku z
	 * ktoreho nasledne odoberieme pocetCisielOdobrat cisiel pomocou metody
	 * odoberCisla a vznikne nam platne zadanie sudoku. Pocet odobranych cisiel
	 * urcuje naocnost sudoku.
	 * 
	 * @param mriezka
	 *            Vstupne 2d pole integerov naplnene cislami 0.
	 * @param pocetCisielPridat
	 *            Pocet cisiel na pridanie na zaciatku.
	 * @param pocetCisielOdobrat
	 *            Pocet cisiel na odobranie na konci.
	 * @return
	 */
	static boolean generujSudoku(int mriezka[][], int pocetCisielPridat, int pocetCisielOdobrat) {

		do {
			inicializujSudokuBunky(mriezka);

			pridajCisla(mriezka, pocetCisielPridat);
		} while (!ries(0, 0, mriezka));
		odoberCisla(mriezka, pocetCisielOdobrat);
		return true;
	}

	/**
	 * Skontroluje ci je 2d pole integerov mriezka platne podla pravidiel sudoku.
	 * Teda ci su cisla v riadkoch a stlpcoch rozdielne(1 az 9) a aj v boxoch
	 * velkosti parametra B (bezne sudoku ma 3x3).
	 * 
	 * @param mriezka
	 *            Vstupne 2d pole integerov.
	 * @return Vrati true ak mriezka splna pravidla sudoku. Vrati false ak mriezka
	 *         nesplna pravidla sudoku.
	 */
	static boolean jeSudokuPlatne(int mriezka[][]) {
		int riadok, stlpec, hodnota;
		HashSet<Integer> hset = new HashSet<Integer>();

		// skontroluj riadky
		for (riadok = 0; riadok < N; riadok++) {
			hset.clear();
			for (stlpec = 0; stlpec < N; stlpec++) {
				hodnota = mriezka[riadok][stlpec];
				if (hodnota < 0 || hodnota > N)
					return false;
				if (hodnota != 0)
					if (!hset.add(hodnota))
						return false;
			}
		}
		// skontroluj stlpce
		for (stlpec = 0; stlpec < N; stlpec++) {
			hset.clear();
			for (riadok = 0; riadok < N; riadok++) {
				hodnota = mriezka[riadok][stlpec];
				if (hodnota < 0 || hodnota > N)
					return false;
				if (hodnota != 0)
					if (!hset.add(hodnota))
						return false;
			}
		}
		// skontroluj mriezku 3 x 3
		for (riadok = 0; riadok < 7; riadok = riadok + 3) {
			hset.clear();
			for (stlpec = 0; stlpec < 7; stlpec = stlpec + 3) {
				hset.clear();
				for (int x = riadok; x <= riadok + 2; x++)
					for (int y = stlpec; y <= stlpec + 2; y++) {
						hodnota = mriezka[x][y];
						if (hodnota != 0)
							if (!hset.add(hodnota))
								return false;
					}
			}
		}
		return true;
	}

	/**
	 * Samotny backtracking algoritmus na riesenie sudoku. Volame s parametrami i=
	 * 0, j =0 - suradnice pociatocnej bunky. Vyskusa postupne vsetky mozne cisla v
	 * kazdej bunke pokym nenajde prve riesenie.
	 * 
	 * @param i
	 *            Vstupny parameter, zaciatok backtrackingu, zadame i= 0.
	 * @param j
	 *            Vstupny parameter, zaciatok backtrackingu, zadame j= 0.
	 * @param mriezka
	 *            2d pole ako stav sudoku ktory sa vyriesi a ziskame naplnenu
	 *            mriezku.
	 * @return Ak sa nam podari najst riesenie, vrati true, inak false.
	 */
	static boolean ries(int i, int j, int[][] mriezka) {
		if (i == N) {
			i = 0;
			if (++j == N)
				return true;
		}
		if (mriezka[i][j] != 0) // preskoc vyplnene bunky
			return ries(i + 1, j, mriezka);

		for (int hod = 1; hod <= N; hod++) {
			if (vyhovujePravidlam(i, j, hod, mriezka)) {
				mriezka[i][j] = hod;
				if (ries(i + 1, j, mriezka))
					return true;
			}
		}
		mriezka[i][j] = 0; // daj do povodneho stavu pri backtrackingu
		return false;
	}

	/**
	 * Skontroluje ci nami zadana hodnota hod na pozicii (i,j) v mriezke splna
	 * pravidla sudoku. Nekontroluje cele sudoku, ale iba jedno zadane cislo.
	 * 
	 * @param i
	 *            X-ova pozicia v mriezke.
	 * @param j
	 *            Y-ova pozicia v mriezke.
	 * @param hod
	 *            Hodnota na kontrolovanie.
	 * @param mriezka
	 *            Pole do ktoreho chceme doplnit hodnotu.
	 * @return Vrati true ak nami zadane cislo vyhovuje pravidlam. Vrati false ak
	 *         nami zadane cislo nevyhovuje pravidlam.
	 */
	static boolean vyhovujePravidlam(int i, int j, int hod, int[][] mriezka) {
		for (int k = 0; k < N; k++) // riadok
			if (hod == mriezka[k][j])
				return false;

		for (int k = 0; k < N; k++) // stlpec
			if (hod == mriezka[i][k])
				return false;

		int boxRiadokOffset = (i / B) * B;
		int boxStlpecOffset = (j / B) * B;
		for (int k = 0; k < B; k++) // box
			for (int m = 0; m < B; m++)
				if (hod == mriezka[boxRiadokOffset + k][boxStlpecOffset + m])
					return false;

		return true; // vyhovuje pravidlam
	}

	/**
	 * Vypisanie 2d pola mriezka do konzoly.
	 * 
	 * @param mriezka
	 *            Vstupne 2d pole.
	 */
	static void vytlacMriezku(int mriezka[][]) {
		for (int riadok = 0; riadok < N; riadok++) {
			for (int stlpec = 0; stlpec < N; stlpec++)
				System.out.print(mriezka[riadok][stlpec]);
			System.out.println();
		}
	}

	/**
	 * Zisti rozdiel dvoch 2d poli a nahodne vyberie jedno cislo z tych ktore sa
	 * nezhoduju. Pouziva sa pri Hinte. Nahodny vyber preto, aby napoveda nebola
	 * vzdy prve vyhovujuce cislo zlava zhora.
	 * 
	 * @param riesenie
	 *            Zadame vyriesene sudoku mriezky.
	 * @param mriezka
	 *            Zadame mriezku.
	 * @return Vrati trojicu cisiel v poli. Prve dve oznacuju poziciu v mriezke a
	 *         tretie oznacuje hodnotu na doplnenie.
	 */
	static int[] zistiRozdiel(int[][] riesenie, int mriezka[][]) {
		List<Integer> nuly = new ArrayList<>();
		for (int i = 0; i < N; ++i) {
			for (int j = 0; j < N; ++j) {
				if (mriezka[i][j] == 0) {
					nuly.add(i * N + j);
				}
			}
		}
		if (nuly.isEmpty()) {
			return null;
		}

		int indexNahodny = getRandom(0, nuly.size() - 1);
		int index = nuly.get(indexNahodny);
		int i = index / N;
		int j = index % N;
		return new int[] { i, j, riesenie[i][j] };
	}

	/**
	 * Tato metoda je volana z event handlera pre button pomoc - hint. Skopiruje
	 * aktualny stav mriezky do pomocnej a zavola metodu ries na novovytvorenu
	 * pomocnu mriezku v novom threade. Zavola metodu zistiRozdiel a ziskame
	 * napovedu, ktoru vratime ako String vhodny na vypis.
	 * 
	 * @param mriezka
	 *            Vstupne 2d pole.
	 * @return Vrati string pomocka ktory zobrazuje ako pomocku pre hraca.
	 */
	static String onHint(int mriezka[][]) {

		// skopiruje si aktualny stav do druheho pomocneho pola (lebo solver doplni
		// bunky)
		int[][] riesenie = new int[N][N];// kopia mriezky do pomocnej
		for (int i = 0; i < N; ++i) {
			for (int j = 0; j < N; ++j) {
				riesenie[i][j] = mriezka[i][j];
			}
		}

		try {
			Thread t = new Thread(() -> {
				boolean hasHint = ries(0, 0, riesenie);
				if (hasHint) {
					// najdi prvy rozdiel - hx,hy, hv - porovnanim mriezok
					int[] hint = zistiRozdiel(riesenie, mriezka);
					if (hint != null) {
						pomocka = "Hint: column: " + (hint[0] + 1) + ", row: " + (hint[1] + 1) + ", value=" + hint[2];
					} else {
						pomocka = "Hint neexistuje - Sudoku je uz vyriesene";
					}
				} else {
					pomocka = "Hint neexistuje - Sudoku nema riesenie";
				}
			});
			t.start();
			t.join();
		} catch (InterruptedException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} finally {
		}

		return pomocka;
	}

}