from tkinter import ttk, font, Text
from typing import Generator

from Automata import FA, DFA, NFA
from NotebookTab import NotebookTab
from ToolTip import create_tool_tip

EPSILON = 'ε'

class Computer:
    def __init__(self, tab: NotebookTab):
        self.tab = tab
        self.fa: FA = tab.fa
        tab.compute_exists = True
        self.frame = ttk.Frame(tab.frame)
        self.frame.grid(row=2, columnspan=2, column=0, sticky='we')
        self.exit_button = ttk.Button(self.frame, text="❌", command=self.exit, width=4)
        self.exit_button.pack(side='right', fill='both')
        create_tool_tip(self.exit_button, 'Ukončiť')
        self.start_button = ttk.Button(self.frame, text='▶', command=self.start, width=4)
        self.start_button.pack(side='left', fill='both')
        create_tool_tip(self.start_button, 'Začať/Zastaviť výpočet')
        self.step_button = ttk.Button(self.frame, text='⊢', command=self.step, width=4)
        self.step_button.pack(side='left', fill='both')
        create_tool_tip(self.step_button, 'Ďalší krok výpočtu')
        self.step_button['state'] = 'disabled'
        self.text = Text(self.frame, width=100, height=1)
        self.text.pack(fill='both')
        self.computing = False
        self.finished = False
        f = font.nametofont('TkFixedFont')
        self.text.tag_configure('current', foreground='blue', background='yellow',
                                font=font.Font(family=f['family'], weight='bold', size=f['size']))
        self.computer: Generator[str, str, int]
        self.prev_state, self.state = None, None



    def exit(self):
        if self.prev_state:
            self.set_normal(self.prev_state)
            self.set_normal_arrow(self.prev_state, self.state)
        self.set_normal(self.state)
        self.tab.compute_exists = False
        self.frame.destroy()

    def start(self):
        if self.computing:
            self.computing = False
            self.text['state'] = 'normal'
            self.start_button.configure(text='▶')
            if self.prev_state:
                self.set_normal(self.prev_state)
                self.set_normal_arrow(self.prev_state, self.state)
            self.set_normal(self.state)
            self.step_button['state'] = 'disabled'

        else:
            self.computing = True
            self.finished = False
            self.text['state'] = 'disabled'
            self.start_button.configure(text='■')
            self.state = self.fa.initial
            self.prev_state = None
            self.pos = 0
            self.set_current(self.state)
            if isinstance(self.fa, DFA):
                self.computer = self.dfa_computer(self.text.get('1.0', '1.end'), self.fa.initial)
            else:
                self.computer = self.nfa_computer(self.text.get('1.0', '1.end'), 0, self.fa.initial)
            self.step_button['state'] = 'normal'

    def step(self):
        if self.finished:
            self.text['state'] = 'normal'
            if self.pos >= 0:
                self.text.tag_remove('current', '1.' + str(self.pos))
            if self.prev_state:
                self.set_normal(self.prev_state)
                self.set_normal_arrow(self.prev_state, self.state)

            if self.pos == len(self.text.get('1.0', '1.end')) - 1 and self.state in self.fa.finals:
                self.set_accept(self.state)
                self.text.insert('1.end', '✓')
            else:
                self.set_not_accept(self.state)
                self.text.insert('1.end', '❌')
            self.step_button['state'] = 'disabled'
        else:
            try:
                prev_state, next_state, pos = next(self.computer)
            except StopIteration:
                self.finished = True
                self.step()
                return
            if self.pos >= 0:
                self.text.tag_remove('current', '1.' + str(self.pos))
            if not next_state:
                if prev_state in self.fa.finals:
                    self.finished = True
                self.step()
                return
            self.text.tag_add('current', '1.' + str(pos))
            if self.prev_state:
                self.set_normal(self.prev_state)
                self.set_normal_arrow(self.prev_state, self.state)
            self.set_normal(self.state)
            self.set_previous(prev_state)
            self.set_current(next_state)
            self.set_current_arrow(prev_state, next_state)
            self.prev_state = prev_state
            self.state = next_state
            self.pos = pos

    def step2(self):
        letter = self.text.get('1.'+str(self.pos))
        if letter == '\n':
            self.text['state'] = 'normal'
            if self.pos > 0:
                self.text.tag_remove('current', '1.' + str(self.pos - 1))
            if self.state in self.fa.finals:
                self.set_accept(self.state)
                self.text.insert('1.end', '✓')
            else:
                self.set_not_accept(self.state)
                self.text.insert('1.end', '❌')
            return
        elif letter == '❌' or letter == '✓':
            return
        next_state = self.fa.delta[self.state, letter]
        if self.pos > 0:
            self.text.tag_remove('current', '1.' + str(self.pos-1))
        self.text.tag_add('current', '1.'+str(self.pos))
        if self.prev_state:
            self.set_normal(self.prev_state)
            self.set_normal_arrow(self.prev_state, self.state)
        self.set_previous(self.state)
        self.set_current(next_state)
        self.set_current_arrow(self.state, next_state)
        self.prev_state = self.state
        self.state = next_state
        self.pos += 1


    def set_current(self, state):
        self.color_state(state, 'blue', 5)

    def set_normal(self, state):
        self.color_state(state, 'black', 1)

    def set_previous(self, state):
        self.color_state(state, 'yellow', 3)

    def set_not_accept(self, state):
        self.color_state(state, 'red', 5)

    def set_accept(self, state):
        self.color_state(state, 'green', 5)

    def color_state(self, state, color, width):
        d = self.tab.diagram
        d.canvas.itemconfig(d.state_dict[state].circle_id, width=width, outline=color)

    def set_current_arrow(self, st1, st2):
        self.color_arrow(st1, st2, 'blue', 5)

    def set_normal_arrow(self, st1, st2):
        self.color_arrow(st1, st2, 'black', 1)

    def color_arrow(self, st1, st2, color, width):
        d = self.tab.diagram
        d.canvas.itemconfig(d.arrow_dict[st1, st2].arrow_id, width=width, fill=color)
        if d.arrow_dict[st1, st2].line_id >= 0 :
            d.canvas.itemconfig(d.arrow_dict[st1, st2].line_id, width=width, outline=color)

    def dfa_computer(self, w, state: str):
        for i, l in enumerate(w):
            if l not in self.fa.alphabet: return
            s = self.fa.delta[state, l]
            yield state, s, i
            state = s

    def nfa_computer(self, w: str, pos: int, state: str):
        if (self.state, EPSILON) in self.fa.delta:
            for s in self.fa.delta[state, EPSILON]:
                yield state, s, pos
                yield from self.nfa_computer(w, pos, s)
        if not len(w):
            yield state, '', pos
            return
        l, v = w[0], w[1:]
        if (state, l) in self.fa.delta:
            for s in self.fa.delta[state, l]:
                yield state, s, pos
                yield from self.nfa_computer(v, pos+1, s)
