from typing import Dict, Optional, Set, List, Tuple

from Automata import FA, DFA
from NotebookTab import NotebookTab


class SaveData:
    def __init__(self):
        self.special_characters: Dict[str, str] = {'ε': r'\varepsilon', 'Σ': r'\Sigma', '𝛿': r'\delta'}

    def to_tex(self) -> str:
        return 'empty'

    def show_name(self) -> str:
        return 'empty'


class SaveStructure(SaveData):
    def __init__(self, notebook_tab: NotebookTab):
        super().__init__()
        self.notebook_tab = notebook_tab


class SaveFA(SaveStructure):
    def __init__(self, notebook_tab: NotebookTab = None, fa: FA = None):
        super().__init__(notebook_tab)
        self.det: bool = isinstance(self.notebook_tab.fa if self.notebook_tab else fa, DFA)
        self.fa = fa

    def show_name(self) -> str:
        return ('D' if self.det else 'N') + 'KA ' + self.notebook_tab.name.get()

    def to_tex(self) -> str:
        fa: FA = self.notebook_tab.fa if self.notebook_tab else self.fa
        tex = f'{"D" if self.det else "N"}KA~ {fa.name} = ({fa.sets_name[0]}, {fa.sets_name[1]}, {fa.sets_name[2]}, ' \
              f'{fa.initial}, {fa.sets_name[3]})'.replace('{', r'\{').replace('}', r'\}') \
              + r'\\' + f'\n{fa.sets_name[0]} = ' + r'\{'
        for i, state in enumerate(sorted(fa.states)):
            tex += state.replace('{', r'\{').replace('}', r'\}')
            if i != len(fa.states) - 1:
                tex += ', '
        tex += r'\}\\' + f'\n{fa.sets_name[3]} = ' + r'\{'
        for i, final in enumerate(sorted(fa.finals)):
            tex += final.replace('{', r'\{').replace('}', r'\}')
            if i != len(fa.finals) - 1:
                tex += ', '
        tex += r'\}\\' + f'\n{fa.sets_name[1]} = ' + r'\{'
        for i, letter in enumerate(sorted(fa.alphabet)):
            tex += letter
            if i != len(fa.alphabet) - 1:
                tex += ', '
        tex += r'\}\\' + '\n'
        for k, v in fa.delta.items():
            tex += f'{fa.sets_name[2]}({k[0]}, {k[1]}) = '.replace('{', r'\{').replace('}', r'\}')
            if self.det:
                tex += v.replace('{', r'\{').replace('}', r'\}') + r'\\'
            else:
                tex += r'\{'
                for i, n in enumerate(sorted(v)):
                    if i != 0:
                        tex += ", "
                    tex += n
                tex += r'\}\\' + '\n'
        for k, v in self.special_characters.items():
            tex = tex.replace(k, v)
        return tex


class SaveAction(SaveData):
    def __init__(self, save_data1):
        super().__init__()
        self.save_data1 = save_data1
        self.save_data2: SaveData = SaveData()
        self.action_name: str = ''

    def add_save_data2(self, save_data2: SaveData):
        self.save_data2 = save_data2

    def show_name(self) -> str:
        return self.save_data1.show_name() + self.action_name + self.save_data2.show_name()


class SaveEpsilonFreeNFA(SaveAction):
    def __init__(self, save_data1: SaveData):
        super().__init__(save_data1)
        self.closure: Dict[str, Set[str]] = {}
        self.action_name = " -> odepsilonuj -> "

    def add_closure(self, closure):
        self.closure = closure

    def to_tex(self) -> str:
        tex = r" $Zadaný je $"
        tex += self.save_data1.to_tex()
        closure_tex = r"$ Epsilónový uzáver: $\\"
        for state in sorted(self.closure.keys()):
            closure_tex += '[0pt][' + state + r']^{\varepsilon} = \{'
            for i, st in enumerate(sorted(self.closure[state])):
                closure_tex += st
                if i != len(self.closure[state]) - 1:
                    closure_tex += ', '
            closure_tex += r'\}\\' + '\n'
        for k, v in self.special_characters.items():
            closure_tex = closure_tex.replace(k, v)
        tex += r"\\" + '\n' + closure_tex
        tex += "$ Výsledný odepsilonovaný automat je:$"
        tex += r'\\' + '\n' + self.save_data2.to_tex()
        return tex


class SaveNFAToDFA(SaveAction):
    def __init__(self, save_data1: SaveData):
        super().__init__(save_data1)
        self.saveEpsilonFree: Optional[SaveEpsilonFreeNFA] = None
        self.action_name = " -> "

    def getSaveEpsilonFree(self) -> SaveEpsilonFreeNFA:
        self.saveEpsilonFree = SaveEpsilonFreeNFA(self.save_data1)
        return self.saveEpsilonFree

    def to_tex(self) -> str:
        tex = " $Chceme prerobiť nedeterministický automat $" + self.save_data1.show_name()[
                                                                4:] + "$ na deterministický. $ "
        if self.saveEpsilonFree:
            tex += self.save_data1.show_name()[4:] + r" $ má prechody na epsilon. Najprv ho odepsilonujeme.$\\ "
            tex += self.saveEpsilonFree.to_tex()
        else:
            tex += "$Zadaný je $"
            tex += self.save_data1.to_tex()
        tex += r'\\' + '\n' + r"$Výsledný deterministický automat vyrobíme z podmnožín zo stavov a výberom dosiahnuteľných z nich: $\\"
        tex += self.save_data2.to_tex()
        # for k, v in self.special_characters.items():
        #    tex = tex.replace(k, v)
        return tex


class SaveMinimalisation(SaveAction):
    def __init__(self, save_data1: SaveData):
        super().__init__(save_data1)
        self.action_name = " -> minimalizuj -> "
        self.first_partition: List[Set[str]] = []
        self.info: List[Tuple[str, str, str, str, str] | bool | List[Set[str]]] = []
        self.final_partition: List[Set[str]] = []
        self.delta_name: str = '𝛿'

    def change_delta(self, s):
        self.delta_name = s

    def add_first_partition(self, p):
        self.first_partition = p

    def unchanged(self):
        self.info.append(False)

    def add_partition(self, p):
        self.info.append(p)

    def add_p_data(self, s1, s2, letter, s3, s4):
        self.info.append((s1, s2, letter, s3, s4))

    def add_new_partition(self, p):
        self.info.append(False)
        self.info.append(p)

    def add_final_partition(self, p):
        self.final_partition = p

    def show_partition(self, p) -> str:
        text = ''
        for par in p:
            text += r'{' + r','.join(par) + r'}' + r'\\' +'\n'
        return text.replace('{', r'\{').replace('}', r'\}')

    def to_tex(self) -> str:
        tex = r" $Zadaný je $"
        tex += self.save_data1.to_tex() + r'\\'
        tex += r' $Počiatočné rozdelenie: $\\'+ '\n'
        tex += self.show_partition(self.first_partition) + r'\\'
        while self.info:
            tex += ' $Skupina stavov: $' + ','.join(self.info.pop(0)).replace('{', r'\{').replace('}', r'\}')\
                   + r'\\' +'\n'
            p = self.info[0]
            if not p:
                tex += r' $ sa už nebude deliť.$\\'+'\n'
                self.info.pop(0)
            else:
                while p := self.info.pop(0):
                    s = f'{self.delta_name}({p[0]}, {p[2]}) =  {p[3]} ' + r'\ne' + \
                        f'{self.delta_name}({p[1]}, {p[2]}) =  {p[4]}' + r'\\' + '\n' + \
                        f' $Stavy: ${p[0]}$ a ${p[1]}$ nebudú v rovnakej skupine. $' + r'\\' + '\n'
                    tex += s.replace('{', r'\{').replace('}', r'\}')
                tex += ' $Nové rozdelenie: $' + self.show_partition(self.info.pop(0)) + r'\\' + '\n'
        tex += ' $ Konečné rozdelenie: $' + self.show_partition(self.final_partition) + r'\\' + '\n'
        for k, v in self.special_characters.items():
            tex = tex.replace(k, v)
        tex += "$ Výsledný minimálny automat je:$"
        tex += r'\\' + '\n' + self.save_data2.to_tex()
        return tex
