import re
from tkinter import *
from tkinter import ttk
from typing import List, Tuple, Set, Dict

from Automata import DFA, NFA, FA, TEST_NFA

CAP_SIGMA = 'Σ'
DELTA = '𝛿'
EPSILON = 'ε'


def get_set(s) -> Set[str]:
    pattern = re.compile(r'\s*=\s*{([^}]*)}')
    m = re.search(pattern, s)
    if m:
        strings = set(re.split(r'[ ,;]', m[1]))
        strings.discard('')
        return strings
    return set()


class TextEditor:
    def __init__(self, frame, row, column):
        self.text = Text(frame, width=50, height=30, borderwidth=0)
        self.text.grid(row=row, column=column, sticky='nwes')
        # self.text['bg']='SystemButtonFace'
        self.text.insert('1.0', 'zdf')
        self.text.delete('1.0', 'end')
    
    def show_fa(self, fa: FA):
        if isinstance(fa, DFA):
            self.show_dfa(fa)
        elif isinstance(fa, NFA):
            self.show_nfa(fa)

    def show_dfa(self, dfa: DFA):
        self.text.delete('1.0', 'end')
        self.text.insert('1.0', f'DKA {dfa.name} = (K, {CAP_SIGMA}, {DELTA}, {dfa.initial}, F)\n')
        states = 'K = {'
        for i, state in enumerate(dfa.states):
            states += state
            if i != len(dfa.states) - 1:
                states += ', '
        states += '}\n'
        self.text.insert('2.0', states)

        finals = 'F = {'
        for i, final in enumerate(dfa.finals):
            finals += final
            if i != len(dfa.finals) - 1:
                finals += ', '
        finals += '}\n'
        self.text.insert('3.0', finals)

        alphabet = CAP_SIGMA + ' = {'
        for i, letter in enumerate(dfa.alphabet):
            alphabet += letter
            if i != len(dfa.alphabet) - 1:
                alphabet += ', '
        alphabet += '}\n'
        self.text.insert('4.0', alphabet)

        delta = ''
        for k, v in dfa.delta.items():
            delta += DELTA + f'({k[0]}, {k[1]}) = {v}\n'
        self.text.insert('5.0', delta)

    def show_nfa(self, nfa: NFA):
        self.text.delete('1.0', 'end')
        self.text.insert('1.0', f'NKA {nfa.name} = (K, {CAP_SIGMA}, {DELTA}, {nfa.initial}, F)\n')
        states = 'K = {'
        for i, state in enumerate(nfa.states):
            states += state
            if i != len(nfa.states) - 1:
                states += ', '
        states += '}\n'
        self.text.insert('2.0', states)

        finals = 'F = {'
        for i, final in enumerate(nfa.finals):
            finals += final
            if i != len(nfa.finals) - 1:
                finals += ', '
        finals += '}\n'
        self.text.insert('3.0', finals)

        alphabet = CAP_SIGMA + ' = {'
        for i, letter in enumerate(nfa.alphabet):
            alphabet += letter
            if i != len(nfa.alphabet) - 1:
                alphabet += ', '
        alphabet += '}\n'
        self.text.insert('4.0', alphabet)

        delta = ''
        for k, v in nfa.delta.items():
            delta += DELTA + f'({k[0]}, {k[1]}) = ' + '{'
            for i, n in enumerate(v):
                if i != 0:
                    delta += ", "
                delta += n
            delta += "}\n"
        self.text.insert('5.0', delta)

    def parse(self) -> FA | Tuple[str, List[str]]:
        match = None
        header = re.compile(r'([NnDd])[KkFf][Aa]'
                            r'\s*(\S*)\s*[^(]*\s*\(\s*([^ ,;]*)[ ,;]*([^ ,;]*)[ ,;]*'
                            r'([^ ,;]*)[ ,;]*([^ ,;]*)[ ,;]*([^ ,;]*)\s*\)')
        i = 1
        s = ''
        while not match:
            s = self.text.get(f'{i}.0', f'{i}.end')
            if not s:
                return "No header definition", []
            match = header.search(s)
            i += 1
        a_type = match[1]
        name = match[2]
        states_name = match[3]
        alphabet_name = match[4]
        function_name = match[5]
        initial_state = match[6]
        finals_name = match[7]
        states: Set[str] = set()
        finals: Set[str] = set()
        alphabet: Set[str] = set()
        delta: Dict[Tuple[str, str], Set[str]] = {}
        sets_name = (states_name, alphabet_name, function_name, finals_name)
        while s:
            s = self.text.get(f'{i}.0', f'{i}.end')
            line_start = re.compile(f'^({states_name}|{alphabet_name}|{function_name}|{finals_name})')
            match = re.search(line_start, s)
            if match:
                if match[1] == states_name:
                    states = get_set(s)
                elif match[1] == alphabet_name:
                    alphabet = get_set(s)
                elif match[1] == finals_name:
                    finals = get_set(s)
                elif match[1] == function_name:
                    parse_delta_line(s, delta)
            i += 1
        if initial_state not in states:
            return 'initial state not in states', []
        if not finals:
            return 'no final states', []
        if not finals <= states:
            l: List[str] = list(finals - states)
            return 'finals is not subset of states', l
        if not delta:
            return 'no function', []
        key_state_set = set(key[0] for key in delta)
        if not key_state_set <= states:
            l: List[str] = list(key_state_set - states)
            return 'function defined for states not in states', l
        key_letter_set = set(key[1] for key in delta) - set(EPSILON)
        if not key_letter_set <= alphabet:
            l: List[str] = list(key_letter_set - alphabet)
            return 'function defined for letters not in alphabet', l
        result_state_set = set()
        eps = None
        for key in delta:
            if key[1] == EPSILON:
                eps = key
            result_state_set |= delta[key]
        if not result_state_set <= states:
            l: List[str] = list(result_state_set - states)
            return 'function result for states not in states', l
        if a_type.lower() == 'd':
            if eps:
                return 'epsilon in dfa', [eps[0]]
            dfa_delta: Dict[Tuple[str, str], str] = {}
            for state in states:
                for letter in alphabet:
                    t = (state, letter)
                    if t not in delta or len(delta[t]) == 0:
                        return 'no transit for tuple ', [*t]
                    if len(delta[t]) > 1:
                        return 'nondeterministic transit in dfa', [*t]
                    else:
                        dfa_delta[t] = delta[t].pop()
            return DFA(states, alphabet, dfa_delta, initial_state, finals, name, sets_name)
        return NFA(states, alphabet, delta, initial_state, finals, name, sets_name)


def parse_delta_line(s, delta):
    delta_pattern = re.compile(r'[^(]*\(\s*([^ ,;]*)[ ,;]*([^ )]*)\)\s*=\s*{?([^}]*)}?')
    match = delta_pattern.search(s)
    if match:
        strings = set(re.split(r'[ ,;]', match[3]))
        strings.discard('')
        delta[(match[1], match[2])] = strings


if __name__ == '__main__':
    root = Tk()
    frame = ttk.Frame(root)
    frame.pack()
    t = TextEditor(frame, 0, 0)
    t.show_nfa(TEST_NFA)
    t.parse()
    root.mainloop()
