import os
import sys
from tkinter import *
from tkinter import ttk, messagebox
from typing import List, Dict, Callable

from Automata import DFA, NFA, FA, TEST_NFA, ABBA_NFA
from AutomataActions import diagram_to_dfa, diagram_to_nfa, \
    epsilon_free_nfa, remove_unreachable_states, nfa_to_dfa, minimalisation, regular_expression
from NotebookTab import NotebookTab
from SaveData import SaveData, SaveFA, SaveNFAToDFA, SaveEpsilonFreeNFA, SaveAction, SaveMinimalisation
from SaveWindow import SaveWindow
from Computation import Computer

EPSILON = 'ε'
CAP_SIGMA = 'Σ'
DELTA = '𝛿'


def new_notebook_tab():
    order = len(notebook_tabs_by_order)
    notebook_tab = NotebookTab(notebook, order, icons_dict)
    notebook_tabs_by_order.append(notebook_tab)
    notebook_tabs_by_frame[str(notebook_tab.frame)] = notebook_tab
    notebook.select(notebook_tab.frame)
    return notebook_tab


def delete_notebook_tab():
    s = notebook.select()
    tab = notebook_tabs_by_frame[s]
    del notebook_tabs_by_frame[s]
    notebook_tabs_by_order.remove(tab)
    notebook.forget(s)


def move_left():
    s = notebook.select()
    i = notebook.index(s)
    if i > 0:
        notebook.insert(i - 1, s)


def move_right():
    s = notebook.select()
    i = notebook.index(s)
    if i < notebook.index('end') - 1:
        notebook.insert(i + 1, s)


def save():
    SaveWindow().create_save_window(save_data)


def diagram_to_dfa_call():
    frame = notebook.select()
    tab = notebook_tabs_by_frame[frame]
    dfa = diagram_to_dfa(tab.diagram)
    if isinstance(dfa, DFA):
        tab.fa = dfa
        tab.text.show_dfa(dfa)
        tab.name.set(dfa.name)
        notebook.tab(frame, text=dfa.name)
        save = SaveFA(tab)
        save_data.append(save)
        save_data_by_tab[tab] = save
        tab.regex_label.config(text="Regulárny výraz: "+ regular_expression(dfa))
    else:
        error_handler(dfa)


def diagram_to_nfa_call():
    frame = notebook.select()
    tab = notebook_tabs_by_frame[frame]
    nfa = diagram_to_nfa(tab.diagram)
    if isinstance(nfa, NFA):
        tab.fa = nfa
        tab.text.show_nfa(nfa)
        tab.name.set(nfa.name)
        notebook.tab(frame, text=nfa.name)
        save = SaveFA(tab)
        save_data.append(save)
        save_data_by_tab[tab] = save
        tab.regex_label.config(text="Regulárny výraz: "+ regular_expression(nfa))
    else:
        error_handler(nfa)



def parse_call():
    frame = notebook.select()
    tab = notebook_tabs_by_frame[frame]
    fa = tab.text.parse()
    if isinstance(fa, FA):
        tab.fa = fa
        tab.diagram.create_diagram(fa)
        tab.name.set(fa.name)
        notebook.tab(frame, text=fa.name)
        save = SaveFA(tab)
        save_data.append(save)
        save_data_by_tab[tab] = save
        tab.regex_label.config(text="Regulárny výraz: "+ regular_expression(fa))
    else:
        error_handler(fa)



def fa_action(tab: NotebookTab, save_data_class: Callable[[SaveData], SaveAction], action: Callable[[FA, SaveData],FA]):
    process_save = save_data_class(save_data_by_tab[tab])
    save_data.append(process_save)
    new_tab = new_notebook_tab()
    fa = action(tab.fa, process_save)
    notebook.tab(new_tab.frame, text=fa.name)
    new_tab.diagram.create_diagram(fa)
    new_tab.text.show_fa(fa)
    new_tab.regex_label.config(text="Regulárny výraz: " + regular_expression(fa))
    new_tab.fa = fa
    new_tab.name.set(fa.name)
    new_save_data = SaveFA(new_tab)
    save_data.append(new_save_data)
    save_data_by_tab[new_tab] = new_save_data
    process_save.add_save_data2(new_save_data)


def nfa_to_dfa_call_call():
    frame = notebook.select()
    if frame not in notebook_tabs_by_frame.keys():
        return
    tab = notebook_tabs_by_frame[frame]
    if not isinstance(tab.fa, NFA):
        return
    fa_action(tab, SaveNFAToDFA, lambda x, y: remove_unreachable_states(nfa_to_dfa(x, y)))


def epsilon_free_nfa_call():
    frame = notebook.select()
    if frame not in notebook_tabs_by_frame.keys():
        return
    tab = notebook_tabs_by_frame[frame]
    if not isinstance(tab.fa, NFA):
        return
    fa_action(tab, SaveEpsilonFreeNFA, lambda x, y: epsilon_free_nfa(x, y))


def minimalisation_call():
    frame = notebook.select()
    if frame not in notebook_tabs_by_frame.keys():
        return
    tab = notebook_tabs_by_frame[frame]
    if not isinstance(tab.fa, DFA):
        return
    fa_action(tab, SaveMinimalisation, lambda x, y: minimalisation(x, y))


def compute_call():
    frame = notebook.select()
    if frame not in notebook_tabs_by_frame.keys():
        return
    tab: NotebookTab = notebook_tabs_by_frame[frame]
    if tab.compute_exists or not isinstance(tab.fa, FA):
        return
    if isinstance(tab.fa, NFA):
        diagram_to_nfa_call()
    else:
        diagram_to_dfa_call()
    Computer(tab)


def insert_button(f: ttk.Widget | Tk, char):
    widget = f.focus_get()
    if isinstance(widget, ttk.Entry) or isinstance(widget, Entry) or isinstance(widget, Text):
        index = widget.index(INSERT)
        widget.insert(index, char)

def error_handler(e):
    message = ''
    match e:
        case 'no initial state', []:
            message='Nie je počiatočný stav.'
        case 'no final states', []:
            message='Nie je žiaden akceptačný stav.'
        case 'no state name', []:
            message='Stav nemá meno.'
        case 'duplicate state name', [n]:
            message='Viac stavov s rovnakým menom: "' + n + '".'
        case 'no transitions', []:
            message = "Nie sú prechody."
        case 'empty transit', [s1, s2]:
            message = 'Prechod zo stavu "' + s1 + '" do stavu "' + s2 + '" je prázdny.'
        case 'epsilon in dfa', [s1, s2]:
            message = 'Prechod na ' + EPSILON + ' zo stavu "' + s1 + '" do stavu "' + s2 + '" v DKA.'
        case 'nondeterministic transit in dfa', [s, l]:
            message='Nedeterministický prechod v DKA\n' + "pre stav: '" + s + "' a znak: '" + l + "'"
        case 'no transit for tuple ', [s, l]:
            message = 'Nie je prechod pre stav "' + s + '" a znak "' + l + '" v DKA.'
        case "No header definition", []:
            message = 'Nie je definícia automatu - 5-ica.'
        case 'initial state not in states', []:
            message = 'Počiatočný stav nie je v množine stavov'
        case 'finals is not subset of states', l:
            message = 'Akceptačné stavy nie sú podmnožina stavov.\nStavy navyše: ' + ', '.join(l)
        case 'no function', []:
            message = "Nie je prechodová funkcia"
        case 'function defined for states not in states', l:
            message = 'Prechodová funkcia definovaná pre stavy, ktoré nie sú v množine stavov.\n' \
                      'Stavy navyše: ' + ', '.join(l)
        case 'function defined for letters not in alphabet', l:
            message = 'Prechodová funkcia definovaná pre znaky, ktoré nie sú v množine znakov.\n' \
                      'Znaky navyše: ' + ', '.join(l)
        case 'function result for states not in states', l:
            message = 'Výsledok funkcie obsahuje stavy, ktoré nie sú v množine stavov.\n' \
                      'Stavy navyše: ' + ', '.join(l)
    messagebox.showerror(title='Neplatný automat', message=message)


def example():
    new_tab = new_notebook_tab()
    new_tab.diagram.create_diagram(TEST_NFA)
    new_tab.text.show_nfa(TEST_NFA)
    new_tab.fa = TEST_NFA
    new_tab.name.set(TEST_NFA.name)
    notebook.tab(new_tab.frame, text=TEST_NFA.name)
    save = SaveFA(new_tab)
    save_data.append(save)
    save_data_by_tab[new_tab] = save
    new_tab.regex_label.config(text="Regulárny výraz: " + regular_expression(TEST_NFA))

def example2():
    new_tab = new_notebook_tab()
    new_tab.diagram.create_diagram(ABBA_NFA)
    new_tab.text.show_nfa(ABBA_NFA)
    new_tab.fa = ABBA_NFA
    new_tab.name.set(ABBA_NFA.name)
    notebook.tab(new_tab.frame, text=ABBA_NFA.name)
    save = SaveFA(new_tab)
    save_data.append(save)
    save_data_by_tab[new_tab] = save
    new_tab.regex_label.config(text="Regulárny výraz: " + regular_expression(ABBA_NFA))


if __name__ == '__main__':
    root = Tk()
    root.title("Editor konečných automatov")
    top_frame = ttk.Frame(root)
    top_frame.pack(side='top', fill='both', expand=True)
    top_frame.rowconfigure(0, weight=1)
    top_frame.rowconfigure(1, weight=1)
    top_frame.columnconfigure(1, weight=1)

    right_frame = ttk.Frame(top_frame)
    right_frame.grid(row=0, column=1, rowspan=2, sticky='nwes')
    right_frame.columnconfigure(0, weight=1)
    right_frame.rowconfigure(1, weight=1)

    notebook = ttk.Notebook(right_frame)
    notebook.grid(row=1, column=0, sticky='nwes', columnspan=2)
    notebook.columnconfigure(0, weight=1)
    notebook.rowconfigure(0, weight=1)
    notebook_tabs_by_order: List[NotebookTab] = []
    notebook_tabs_by_frame: Dict[str, NotebookTab] = {}
    save_data: List[SaveData] = []
    save_data_by_tab: Dict[NotebookTab, SaveData] = {}

    notebook_buttons_frame = ttk.Frame(right_frame)
    notebook_buttons_frame.grid(row=0, column=0, sticky='w')

    add_tab_button_text = StringVar(value='Nový automat')
    add_tab_button = ttk.Button(notebook_buttons_frame, text=add_tab_button_text.get(), command=new_notebook_tab)
    add_tab_button.pack(side='left')

    delete_tab_button_text = StringVar(value='Odtrániť automat')
    delete_tab_button = ttk.Button(notebook_buttons_frame, text=delete_tab_button_text.get(),
                                   command=delete_notebook_tab)
    delete_tab_button.pack(side='left')

    move_left_button = ttk.Button(notebook_buttons_frame, text='◀', width=2,
                                  command=move_left)
    move_left_button.pack(side='left')
    move_right_button = ttk.Button(notebook_buttons_frame, text='▶', width=2,
                                   command=move_right)
    move_right_button.pack(side='left')

    save_button = ttk.Button(right_frame, text='Uložiť', command=save)
    save_button.grid(row=0, column=1, sticky='e')

    actions_buttons_frame = ttk.Frame(top_frame)
    actions_buttons_frame.grid(row=0, column=0, sticky='n')
    # actions_buttons_frame.pack(side='top', fill='both')

    d_to_d_button_text = StringVar(value="diagram → DKA")
    d_to_d_button = ttk.Button(actions_buttons_frame, text=d_to_d_button_text.get(),
                               command=diagram_to_dfa_call)
    d_to_d_button.pack(side='top', fill='both')

    d_to_n_button_text = StringVar(value="diagram → NKA")
    d_to_n_button = ttk.Button(actions_buttons_frame, text=d_to_n_button_text.get(),
                               command=diagram_to_nfa_call)
    d_to_n_button.pack(side='top', fill='both')

    t_to_a_button_text = StringVar(value="text → NKA | DKA")
    t_to_a_button = ttk.Button(actions_buttons_frame, text=t_to_a_button_text.get(),
                               command=parse_call)
    t_to_a_button.pack(side='top', fill='both')

    eps_free_button_text = StringVar(value='Odepsilonuj NKA')
    eps_free_button = ttk.Button(actions_buttons_frame, text=eps_free_button_text.get(),
                                 command=epsilon_free_nfa_call)
    eps_free_button.pack(side='top', fill='both')

    nfa_to_dfa_button_text = StringVar(value='NKA → DKA')
    nfa_to_dfa_button = ttk.Button(actions_buttons_frame, text=nfa_to_dfa_button_text.get(),
                                   command=nfa_to_dfa_call_call)
    nfa_to_dfa_button.pack(side='top', fill='both')

    minim_button = ttk.Button(actions_buttons_frame, text='Minimalizuj DKA', command=minimalisation_call)
    minim_button.pack(side='top', fill='both')

    computation_button = ttk.Button(actions_buttons_frame, text='▶ Výpočet', command=compute_call)
    computation_button.pack(side='top', fill='both')

#    example_button_text = StringVar(value="Príklad")
#    example_button = ttk.Button(actions_buttons_frame, text=example_button_text.get(),
#                                command=example)
#    example_button.pack(side='top', fill='both')

    example2_button_text = StringVar(value="Ukážka")
    example2_button = ttk.Button(actions_buttons_frame, text=example2_button_text.get(),
                                command=example2)
    example2_button.pack(side='top', fill='both')

    insert_buttons_frame = ttk.Frame(top_frame)
    insert_buttons_frame.grid(row=1, column=0, sticky=(S, E, W))
    # insert_buttons_frame.pack(fill='both')

    epsilon_button = ttk.Button(insert_buttons_frame, text=EPSILON, takefocus=False,
                                command=lambda: insert_button(top_frame, EPSILON))
    epsilon_button.pack(side='bottom', fill='both')
    sigma_button = ttk.Button(insert_buttons_frame, text=CAP_SIGMA, takefocus=False,
                              command=lambda: insert_button(top_frame, CAP_SIGMA))
    sigma_button.pack(side='bottom', fill='both')
    delta_button = ttk.Button(insert_buttons_frame, text=DELTA, takefocus=False,
                              command=lambda: insert_button(top_frame, DELTA))
    delta_button.pack(side='bottom', fill='both')

    icons_dict: Dict[str, PhotoImage] = {}
    if getattr(sys, 'frozen', False):  # Running as compiled
        running_dir = sys._MEIPASS  # Same path name than pyinstaller option
        path = running_dir
    else:
        running_dir = "./"  # Path name when run with Python interpreter
        path = running_dir + 'icons'
    for filename in os.listdir(path):
        base_name, extension = os.path.splitext(filename)
        if extension != '.png':
            if extension == '.ico':
                root.iconbitmap(os.path.join(path, filename))
            continue

        icons_dict[base_name] = PhotoImage(file=os.path.join(path, filename))

    new_notebook_tab()
    root.mainloop()
