#lang typed/racket ;#:no-optimize
;#lang typed/racket/no-check

(require "expr.rkt")
(require "proof.rkt")

(provide formula->sexp
         ;p->sexp
         vt->sexp
         ht->sexp
         vts->sexps
         term->sexp
         terms->sexps
         proof->sexp
         signedformula->sexp
         signedformulas->sexps
         symbolf->sexp
         symbolp->sexp
         )

;;; from first-order expressions to symbolic expressions

;; symbols
(define-type PInvTableVt (HashTable Term Sexp))
(define-type PInvTablef (HashTable Symbol Sexp))
(define-type PInvTableP (HashTable Symbol Sexp))


(: PIT2 PInvTableVt)
(define PIT2
  (make-hash (list
              (cons (Vt 0) 'a)
              (cons (Vt 1) 'x)
              (cons (Vt 2) 'b)
              (cons (Vt 3) 'y)
              (cons (Vt 4) 'c)
              (cons (Vt 5) 'z)
              (cons (Vt 7) 'u)
              (cons (Vt 9) 'v)
              (cons (Vt 11) '1)
              (cons (Vt 13) '2)
              )))


(: PIT3 PInvTablef)
(define PIT3 (make-hash (list
                        (cons (cons 0 0) 'f^0)
                        (cons (cons 1 0) 'f^1)
                        (cons (cons 2 0) 'f^2)
                        (cons (cons 3 0) 'f^3)
                        (cons (cons 4 0) 'f^4)
                        (cons (cons 0 1) 'g^0)
                        (cons (cons 1 1) 'g^1)
                        (cons (cons 2 1) 'g^2)
                        (cons (cons 3 1) 'g^3)
                        (cons (cons 0 2) 'h^0)
                        (cons (cons 1 2) 'h^1)
                        (cons (cons 2 2) 'h^2)
                        (cons (cons 3 2) 'h^3))))


(: PIT4 PInvTableP)
(define PIT4 (make-hash (list
                        (cons (cons 0 0) 'P)
                        (cons (cons 1 0) 'P^1)
                        (cons (cons 2 0) 'P^2)
                        (cons (cons 3 0) 'P^3)
                        (cons (cons 0 1) 'Q)
                        (cons (cons 1 1) 'Q^1)
                        (cons (cons 2 1) 'Q^2)
                        (cons (cons 3 1) 'Q^3)
                        (cons (cons 0 2) 'R)
                        (cons (cons 1 2) 'R^1)
                        (cons (cons 2 2) 'R^2)
                        (cons (cons 3 2) 'R^3)
                        (cons (cons 0 3) 'A)
                        (cons (cons 1 3) 'A^1)
                        (cons (cons 2 3) 'A^2)
                        (cons (cons 3 3) 'A^3)
                        (cons (cons 0 4) 'B)
                        (cons (cons 0 5) 'C)
                        (cons (cons 0 6) 'D))))

;; from first-order expressions to symbolic expressions

(: symbolf->sexp (-> Symbol Sexp))
(define (symbolf->sexp f)
  (if (hash-has-key? PIT3 f)
      (hash-ref PIT3 f)
      (error "symbolf->sexp/" f)))

(: symbolp->sexp (-> Symbol Sexp))
(define (symbolp->sexp p)
  (if (hash-has-key? PIT4 p)
      (hash-ref PIT4 p)
      (error "symbolP->sexp/" p)))



(: vt->sexp (-> Vt Sexp))
(define (vt->sexp vt)
  (if (hash-has-key? PIT2 vt)
      (hash-ref PIT2 vt)
      (error "Vt->sexp" vt)))

(: at->sexp (-> Symbol Terms Sexp))
(define (at->sexp s ts)
  `(, (symbolf->sexp s)
    ,@(terms->sexps ts)))


(: ht->sexp (-> Formula Sexp))
(define (ht->sexp f)
  (match f
   ; [(Ht f) (formula->sexp f)]
    [_ (formula->sexp f)]))

(: name->sexp (-> Name Sexp))
(define (name->sexp name)
  (string->symbol name))


(: term->sexp (-> Term Sexp))
(define (term->sexp t)
  (match t
    [(Vt n) (vt->sexp t)]
    [(At s ts) (at->sexp s ts) ]
    [(Ht f) (ht->sexp f)]
    [_ (error "term->sexp" t)]))

(: vts->sexps (-> (Listof Vt) (Listof Sexp)))
(define (vts->sexps vts)
  (match vts
    ['() '()]
    [(list* vt vts1) `(,(vt->sexp vt)
                       ,@(vts->sexps vts1))]))

(: terms->sexps (-> Terms (Listof Sexp)))
(define (terms->sexps terms)
  (match terms
    ['() terms]
    [(list* t ts)  `(,(term->sexp t),@(terms->sexps ts) )]
    [_ (error "terms->sexp/" terms)]))




(: formula->sexp (-> Formula Sexp))
(define (formula->sexp a)
  (match a
    [(Af P terms)  (cond [(empty? terms) (symbolp->sexp P)]
                         [else `(, (symbolp->sexp P) ,@(terms->sexps terms))])]
    [(⊤) `⊤]
    [(⊥) `⊥]
    [(¬ b) `(¬ ,(formula->sexp b))]
    [(∧ b c) `(∧ ,(formula->sexp b) ,(formula->sexp c))]
    [(∨ b c) `(∨ ,(formula->sexp b) ,(formula->sexp c))]
    [(→ b c) `(→ ,(formula->sexp b) ,(formula->sexp c))]
    [(∃ x b) `(∃  ,(vt->sexp x) ,(formula->sexp b))]
    [(∀ x b) `(∀  ,(vt->sexp x) ,(formula->sexp b))]
    [_ (error "formula->sexp" a)]))


(: signedformula->sexp (-> SignedFormula Sexp))
(define (signedformula->sexp sa)
  (match sa
    [(Asmp a)
     (formula->sexp a)]
    [(Goal a)
         `(* (,@(formula->sexp a)))]))

(: signedformulas->sexps (-> SignedFormulas (Listof Sexp)))
(define (signedformulas->sexps a)
  (if (empty? a)
      empty
      (list* (signedformula->sexp (car a)) (signedformulas->sexps (cdr a)))))

(: node0-proof->sexp (-> Rule Sexp))
(define (node0-proof->sexp rule)
  (match rule
    [(■) '■]
    [(Axiom name  premises)
     `(,(name->sexp name)
       ,(signedformulas->sexps premises))]
    [_ (error "node0-proof->sexp" rule)]))



(: node1-proof->sexp (-> Rule Proof Sexp))
(define (node1-proof->sexp rule p1)
  (match rule
    [(Prule1 name premise conclusion)
     `(,(name->sexp name)
       ,(signedformula->sexp premise)
       ,(signedformula->sexp conclusion)
       ,(proof->sexp p1))]
    [(Qrule name premise term conclusion)
     `(,(name->sexp name)
       ,(signedformula->sexp premise)
       ,(term->sexp term)
       ,(signedformula->sexp conclusion)
       ,(proof->sexp p1))]
    [_ (error "node1-proof->sexp" rule)]))


(: node2-proof->sexp (-> Rule Proof Proof Sexp))
(define (node2-proof->sexp rule p1 p2)
  (match rule
    [(Prule2 name premise conclusion1 conclusion2)
     `(,(name->sexp name)
       ,(signedformula->sexp premise)
       ,(signedformula->sexp conclusion1)
       ,(signedformula->sexp conclusion2)
       ,(proof->sexp p1)
       ,(proof->sexp p2))
     ]
    [(Cut name conclusion1 conclusion2)
     `(,(name->sexp name)
       ,(signedformula->sexp conclusion1)
       ,(signedformula->sexp conclusion2)
       ,(proof->sexp p1)
       ,(proof->sexp p2))]
    [_ (error "node2-proof->sexp" rule)]))


(: proof->sexp (-> Proof Sexp))
(define (proof->sexp p)
  (match p
    [(Node0 rule) (node0-proof->sexp rule)]
    [(Node1 rule p1) (node1-proof->sexp rule p1)]
    [(Node2 rule p1 p2) (node2-proof->sexp rule p1 p2)]
    [_ (error "proof->sexp" p)]))