#ifndef GRAPH_MAKER
#define GRAPH_MAKER

#include <iostream> 
#include <stdlib.h>     
#include <vector>
#include <algorithm>
#include <time.h>  
#include <string>
#include <fstream>
using namespace std;

//Vsetky graf vytvarajuce funkcie nizzsie vytvoria acyklicky orientovany graf

/// <summary>
/// Graf, v ktorom je po zavolani tvorby nahodneho grafu vysledok.
/// </summary>
std::vector<std::vector<pair<int, int>>> graph;

/// <summary>
/// Do globalneho grafu prida hranu, ak su to jeho vrcholy, a ak by namala vzniknut 
/// nasobna hrana. Ak sa to neda, nerobi nic. Vracia uspesnost pridania.
/// </summary>
/// <param name="start">Pociatocny vrchol hrany</param>
/// <param name="end">Koncovy vrchol hrany</param>
/// <param name="capacity">Kapacita hrany</param>
/// <returns>true ak bola hrana vytvorena, inak false</returns>
bool add_edge(int start, int end, int capacity) {
	if ((start >= end) || (start < 0) || (end >= graph.size())) {
		return false;
	}
	for (int i = 0; i < graph.at(start).size(); i++) {
		if (graph.at(start).at(i).first == end) {
			return false;
		}
	}
	graph.at(start).push_back({ end, capacity });
	return true;
}

/// <summary>
/// Toto cislo hovori, kolko krat bola zavolana hociktora fcia pouzivana random na tvorbu grafu, kvoli tomu, 
/// ze ak by bola dana fcia zavolana 2x za sebou, cas by mohol byt este stale ten isty, a vratilo by to ten isty graf,
/// co je neziaduce.
/// </summary>
uint32_t random_graf_counter = 0;

/// <summary>
/// Vytvori nahodny graf z danym poctom vrcholov a danym poctom hran. 
/// </summary>
/// <param name="vertices_num">Pocet vrcholov</param>
/// <param name="edges_num">Pocet hran</param>
/// <param name="max_capacity">Maximalna kapacita</param>
/// <returns>Dany graf</returns>
std::vector<std::vector<pair<int, int>>> random_graph(int vertices_num, int edges_num, int max_capacity) {
	srand(time(NULL) + random_graf_counter);
	random_graf_counter++;
	graph.clear();
	for (int i = 0; i < vertices_num; i++) {
		graph.push_back(std::vector<pair<int, int>>());
	}

	int new_num_edges = 0;
	int capacity = (rand() % max_capacity) + 1;
	while (new_num_edges < edges_num) {
		int a = rand() % vertices_num;
		int b = rand() % vertices_num;
		int start = min(a, b);
		int end = max(a, b);
		if (add_edge(start, end, capacity)) {
			capacity = (rand() % max_capacity) + 1;
			new_num_edges++;
		}
	}
	return graph;
}

/// <summary>
/// Vytvori graf, ktory ma v takmer v kazdom vrchole okrem zdroja a ustia prave sparsity hran.
/// Vynimku moze tvorit najviac sparsity obycajnych vrcholov.
/// </summary>
/// <param name="vertices_num">Pocet vrcholov grafu</param>
/// <param name="sparsity">Pocet hran v kazdom obycajnom vrchole</param>
/// <param name="max_capacity">Maximalna kapacita kazdej hrany</param>
/// <returns>Dany graf</returns>
std::vector<std::vector<std::pair<int, int>>> random_sparsity_graph(int vertices_num, int sparsity, int max_capacity) {
	srand(time(NULL) + random_graf_counter);
	random_graf_counter++;
	graph.clear();
	for (int i = 0; i < vertices_num; i++) {
		graph.push_back(std::vector<std::pair<int, int>>());
	}

	std::vector<int> edges_nums;
	edges_nums.resize(vertices_num, 0);
	//vytvorenie nahodneho kladneho poctu hran vychadzajucich zo zdroja do vrcholov
	int source_edges_num = min((rand() % (sparsity * 4)) + 1, vertices_num - 1);
	int current_source_edges_num = 0;
	while (current_source_edges_num < source_edges_num) {
		int end = (rand() % vertices_num - 1) + 1;
		int capacity = (rand() % max_capacity) + 1;
		while (!add_edge(0, end, capacity)) { //ak uz bola dana hrana, tak sa skusi dalsi vrchol
			end = (end + 1) % vertices_num;
		}
		edges_nums.at(end)++;
		current_source_edges_num++;
	}
	//vytvorenie nahodneho kladneho poctu hran vchadzajucich hran z vrcholov do ustia
	int sink_edges_num = min((rand() % (sparsity << 2)) + 1, vertices_num - 1);
	int current_sink_edges_num = 0;
	while (current_sink_edges_num < sink_edges_num) {
		int start = rand() % vertices_num;
		int capacity = (rand() % max_capacity) + 1;
		while (!add_edge(start, vertices_num - 1, capacity)) {
			start = (start + 1) % vertices_num;
		}
		edges_nums.at(start)++;
		current_sink_edges_num++;
	}
	//vytvorenie spravneho poctu hran do kazdeho vrchola
	for (int i = 1; i < vertices_num - 1; i++) {
		//chceme teraz viest z i-teho vrchola tolko novych hran, aby bolo pocet jeho hran rovny sparsity
		std::vector<int> possible_vertices;
		for (int j = i + 1; j < vertices_num; j++) {
			if (edges_nums.at(j) < sparsity) {
				possible_vertices.push_back(j);
			}			
		}
		int capacity = (rand() % max_capacity) + 1;
		while ((edges_nums.at(i) < sparsity) && !possible_vertices.empty()) {
			int random_place = rand() % possible_vertices.size();
			if (edges_nums.at(possible_vertices.at(random_place)) < sparsity) {
				if (add_edge(i, possible_vertices.at(random_place), capacity)) {
					edges_nums.at(i)++;
					edges_nums.at(possible_vertices.at(random_place));
					capacity = (rand() % max_capacity) + 1;
				}
			}
			//kazdopadne, do daneho prvku uz nepojde uz hrana, odstranenie z possible_vertices
			possible_vertices.erase(possible_vertices.begin() + random_place);
		}
	}
	return graph;
}

/// <summary>
/// Vytvori graf, ktory ma v kazdom vrchole okram zdroja a ustia najviac sparsity hran v kazdom vrchole.
/// Celkovy pocet hran je zadany. edges_num musi byt mensie ako (sparsity * vertices_num)/2, pricom uz pri
/// postupnom dosahovani tejto hodnoty sa bude dlzka vykonavanie tejto funkcie kvadraticky zvacsovat.
/// </summary>
/// <param name="vertices_num">Pocet vrcholov grafu</param>
/// <param name="edges_num">Pocet hran grafu</param>
/// <param name="sparsity">Maximalny pocet hran obycajneho vrchola</param>
/// <param name="max_capacity">Maximalna kapacita kazdej hrany</param>
/// <returns>Dany graf</returns>
std::vector<std::vector<std::pair<int, int>>> random_sparsity_edges_graph(int vertices_num, int edges_num, int sparsity, int max_capacity) {
	srand(time(NULL) + random_graf_counter);
	random_graf_counter++;
	graph.clear();
	for (int i = 0; i < vertices_num; i++) {
		graph.push_back(std::vector<std::pair<int, int>>());
	}

	std::vector<int> edges_nums;
	edges_nums.resize(vertices_num, 0);
	int new_edges_num = 0;
	int capacity = (rand() % max_capacity) + 1;
	while (new_edges_num < edges_num) {	
		int a = rand() % vertices_num;
		while ((edges_nums.at(a) >= sparsity) || (a == 0)) { //aby bola zabezpecene maximalny pocet hran vrchola, pricom zdroj moze mat viac
			a = (a + 1) % vertices_num;
		}		
		int b = rand() % vertices_num;
		while ((edges_nums.at(b) >= sparsity) || (b == vertices_num - 1)) { //aby bola zabezpecene maximalny pocet hran vrchola, pricom ustie moze mat viac
			b = (b + 1) % vertices_num;
		}
		int start = min(a, b);
		int end = max(a, b);
		if (add_edge(start, end, capacity)) {
			capacity = (rand() % max_capacity) + 1;
			new_edges_num++;
			edges_nums.at(start)++;
			edges_nums.at(end)++;
		}
	}
	return graph;
}

/// <summary>
/// Vytvori graf podla zadaneho typu podla zzadanych parametrov - nie vsetky sa musia pouzit
/// </summary>
/// <param name="graph_type">Typ grafu</param>
/// <param name="vertices_num">Pocet vrcholov grafu</param>
/// <param name="edges_num">Pocet hran grafu</param>
/// <param name="sparsity">Najvyssi pocet hran v obycajnom vrchoole</param>
/// <param name="max_capacity">Maximalna kapacita hrany</param>
/// <returns>Ci bol dany graf vytvoreny</returns>
bool create_graph(std::string graph_type, int vertices_num, int edges_num, int sparsity, int max_capacity) {
	if (graph_type.compare("random_graph") == 0) {
		random_graph(vertices_num, edges_num, max_capacity);
	}
	else if (graph_type.compare("random_sparsity_graph") == 0) {
		random_sparsity_graph(vertices_num, sparsity, max_capacity);
	}
	else if (graph_type.compare("random_sparsity_edges_graph") == 0) {
		random_sparsity_edges_graph(vertices_num, edges_num, sparsity, max_capacity);
	}
	else {
		return false;
	}
	return true;
}

/// <summary>
/// Vytvori a ulozi zadany pocet pravdepodobne roznych grafov s rovnakymi parametrami, pricom nie vsetky
/// sa msia pouzit.
/// </summary>
/// <param name="graph_type">Typ grafov</param>
/// <param name="file_name">Nazov suboru kde sa ma toto ulozit</param>
/// <param name="graphs_num">Pocet tvorenych grafov</param>
/// <param name="vertices_num">Pocet vrcholov jednotlivych grafov</param>
/// <param name="edges_num">Pocet hran jednotlivych grafov</param>
/// <param name="sparsity">Najvyssi pocet hran obycajneho vrhcolu</param>
/// <param name="max_capacity">Maximalna kapacita hrany</param>
void create_and_save_graphs(std::string graph_type, std::string file_name, int graphs_num, int vertices_num, int edges_num, int sparsity, int max_capacity) {
	ofstream graph_file(file_name);
	graph_file << graphs_num << endl;
	for (int i = 0; i < graphs_num; i++) {
		create_graph(graph_type, vertices_num, edges_num, sparsity, max_capacity);
		graph_file << graph.size() << endl;
		for (int j = 0; j < graph.size(); j++) {
			graph_file << graph.at(j).size() << endl;
			for (int k = 0; k < graph.at(j).size(); k++) {
				graph_file << graph.at(j).at(k).first << " " << graph.at(j).at(k).second << endl;
			}
		}
	}
	graph_file.close();
}

#endif // !GRAPH_MAKER
