/*
zdroj: https://cp-algorithms.com/graph/dinic.html
autori: https://github.com/e-maxx-eng/e-maxx-eng/commits/master/src/graph/dinic.md (do 20.1.2021)
        menovite:
        - Yury Semenov https://github.com/SYury
        - Anton Algmyr https://github.com/algmyr
        - organizácia e-maxx-eng https://github.com/e-maxx-eng 
            - Jakob Kogler https://github.com/jakobkogler
licencia zdroju: CC-BT-SA-4.0 https://creativecommons.org/licenses/by-sa/4.0/legalcode

Ja, Adam Struharnansky som dopisal malhotra_ramodhkumar_maheshwari_max_flow, malhotra_ramodhkumar_maheshwari_max_flow_time,
pricom tieto pripajam pod licenciu CC-BT-SA-4.0.
*/

#ifndef MALHOTRA_PRAMODHKUMAR_MAHESHWARI
#define MALHOTRA_PRAMODHKUMAR_MAHESHWARI

#include <vector>
#include <list>
#include <iostream>
#include <chrono>

using namespace std;
using namespace std::chrono;

    struct FlowEdge {
        int v, u;
        long long cap, flow;
        FlowEdge() {}
        FlowEdge(int _v, int _u, long long _cap, long long _flow)
            : v(_v), u(_u), cap(_cap), flow(_flow) {}
        FlowEdge(int _v, int _u, long long _cap)
            : v(_v), u(_u), cap(_cap), flow(0ll) {}
    };

    const long long flow_inf = 1e18;
    vector<FlowEdge> edges;
    vector<char> alive;
    vector<long long> pin, pout;
    vector<list<int> > in, out;
    vector<vector<int> > adj;
    vector<long long> ex;
    int n, m = 0;
    int s, t;
    vector<int> level;
    vector<int> q;
    int qh, qt;

    void resize(int _n) {
        n = _n;
        ex.resize(n);
        q.resize(n);
        pin.resize(n);
        pout.resize(n);
        adj.resize(n);
        level.resize(n);
        in.resize(n);
        out.resize(n);
    }

    void add_edge(int v, int u, long long cap) {
        edges.push_back(FlowEdge(v, u, cap));
        edges.push_back(FlowEdge(u, v, 0));
        adj[v].push_back(m);
        adj[u].push_back(m + 1);
        m += 2;
    }

    bool bfs() {
        while (qh < qt) {
            int v = q[qh++];
            for (int id : adj[v]) {
                if (edges[id].cap - edges[id].flow < 1)continue;
                if (level[edges[id].u] != -1)continue;
                level[edges[id].u] = level[v] + 1;
                q[qt++] = edges[id].u;
            }
        }
        return level[t] != -1;
    }
    long long pot(int v) {
        return min(pin[v], pout[v]);
    }
    void remove_node(int v) {
        for (int i : in[v]) {
            int u = edges[i].v;
            auto it = find(out[u].begin(), out[u].end(), i);
            out[u].erase(it);
            pout[u] -= edges[i].cap - edges[i].flow;
        }
        for (int i : out[v]) {
            int u = edges[i].u;
            auto it = find(in[u].begin(), in[u].end(), i);
            in[u].erase(it);
            pin[u] -= edges[i].cap - edges[i].flow;
        }
    }
    void push(int from, int to, long long f, bool forw) {
        qh = qt = 0;
        ex.assign(n, 0);
        ex[from] = f;
        q[qt++] = from;
        while (qh < qt) {
            int v = q[qh++];
            if (v == to)
                break;
            long long must = ex[v];
            auto it = forw ? out[v].begin() : in[v].begin();
            while (true) {
                int u = forw ? edges[*it].u : edges[*it].v;
                long long pushed = min(must, edges[*it].cap - edges[*it].flow);
                if (pushed == 0)break;
                if (forw) {
                    pout[v] -= pushed;
                    pin[u] -= pushed;
                }
                else {
                    pin[v] -= pushed;
                    pout[u] -= pushed;
                }
                if (ex[u] == 0)
                    q[qt++] = u;
                ex[u] += pushed;
                edges[*it].flow += pushed;
                edges[(*it) ^ 1].flow -= pushed;
                must -= pushed;
                if (edges[*it].cap - edges[*it].flow == 0) {
                    auto jt = it;
                    ++jt;
                    if (forw) {
                        in[u].erase(find(in[u].begin(), in[u].end(), *it));
                        out[v].erase(it);
                    }
                    else {
                        out[u].erase(find(out[u].begin(), out[u].end(), *it));
                        in[v].erase(it);
                    }
                    it = jt;
                }
                else break;
                if (!must)break;
            }
        }
    }
    long long flow() {
        long long ans = 0;
        while (true) {
            pin.assign(n, 0);
            pout.assign(n, 0);
            level.assign(n, -1);
            alive.assign(n, true);
            level[s] = 0;
            qh = 0; qt = 1;
            q[0] = s;
            if (!bfs())
                break;
            for (int i = 0; i < n; i++) {
                out[i].clear();
                in[i].clear();
            }
            for (int i = 0; i < m; i++) {
                if (edges[i].cap - edges[i].flow == 0)
                    continue;
                int v = edges[i].v, u = edges[i].u;
                if (level[v] + 1 == level[u] && (level[u] < level[t] || u == t)) {
                    in[u].push_back(i);
                    out[v].push_back(i);
                    pin[u] += edges[i].cap - edges[i].flow;
                    pout[v] += edges[i].cap - edges[i].flow;
                }
            }
            pin[s] = pout[t] = flow_inf;
            while (true) {
                int v = -1;
                for (int i = 0; i < n; i++) {
                    if (!alive[i])continue;
                    if (v == -1 || pot(i) < pot(v))
                        v = i;
                }
                if (v == -1)
                    break;
                if (pot(v) == 0) {
                    alive[v] = false;
                    remove_node(v);
                    continue;
                }
                long long f = pot(v);
                ans += f;
                push(v, s, f, false);
                push(v, t, f, true);
                alive[v] = false;
                remove_node(v);
            }
        }
        return ans;
    }


    /// <summary>
    /// Graf prepise do dobreho tvaru a na tento zavola MPM algoritmus
    /// </summary>
    /// <param name="g">Graf na ktorom ma byt spsuteny MPM</param>
    /// <returns>Hodnotu maximalneho toku</returns>
    int malhotra_pramodhkumar_maheshwari_max_flow(std::vector<std::vector<std::pair<int,int>>> g) {
        //vyprazdnenie vsetkych struktur
        edges.resize(0);
        alive.resize(0);
        pin.resize(0);
        pout.resize(0);
        in.resize(0);
        out.resize(0);
        adj.resize(0);
        ex.resize(0);
        n = 0; m = 0;
        level.resize(0);
        q.resize(0);

        resize(g.size());
        s = 0;
        t = g.size() - 1;
        for (int i = 0; i < g.size(); i++) {
            for (int j = 0; j < g.at(i).size(); j++) {
                add_edge(i, g.at(i).at(j).first,(long long) g.at(i).at(j).second);
            }
        }
        return (int) flow();
    }

    /// <summary>
    /// Graf prepise do dobreho tvaru a na tento zavola MPM algoritmus, a potom
    /// zisti kolko trval samotny algoritmus v mikrosekundach.
    /// </summary>
    /// <param name="g">Graf na kotorm ma byt spusteny MPM</param>
    /// <returns>Cas trvania MPM v mikrosekundach</returns>
    long long malhotra_ramodhkumar_maheshwari_max_flow_time(std::vector<std::vector<std::pair<int, int>>> g) {
        edges.resize(0);
        alive.resize(0);
        pin.resize(0);
        pout.resize(0);
        in.resize(0);
        out.resize(0);
        adj.resize(0);
        ex.resize(0);
        n = 0; m = 0;
        level.resize(0);
        q.resize(0);

        resize(g.size());
        s = 0;
        t = g.size() - 1;
        for (int i = 0; i < g.size(); i++) {
            for (int j = 0; j < g.at(i).size(); j++) {
                add_edge(i, g.at(i).at(j).first, (long long)g.at(i).at(j).second);
            }
        }
        auto time_start = high_resolution_clock::now();
        flow();
        auto time_stop = high_resolution_clock::now();
        auto duration = duration_cast<microseconds>(time_stop - time_start);
        return duration.count();
    }

#endif MALHOTRA_PRAMODHKUMAR_MAHESHWARI