#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <string.h>

#include "crypto-conf.h"
#include "utils.h"
#include "_version.h"
#include "yeswrap.h"

enum {
    rec_size = public_key_size + hash_size,
};

static void help() {
    fputs(
"ct-man program: challenge table manipulation\n"
"vers. " FDAB_VERSION " (compiled " __DATE__ ")\n"
"Copyright (c) Mikita Alchimionak, 2025\n"
"Usage: ct-man <command> <table_file> [...]\n"
"The following commands are recognized:\n"
"\n"
"   scan      scan the given table for its fill rate\n"
"   poscheck  check the given table whether records are positioned\n"
"             properly (i.e., the first 8 bytes MOD count of recs\n"
"             always matches the record's index)\n"
"   cannibal  cannibalize another table to fill more records in\n"
"             the given one; takes exactly two args, the first\n"
"             for the table to be filled, and the second for the\n"
"             table to be cannibalized; the cannibalized records\n"
"             get zeroed in the second table\n"
"   fill      fill the empty records with random challenges and\n"
"             their respective hashes; this is WAY TOO SLOW so it\n"
"             is only recommended for 99%-filled tables\n"
"   help      show this help and exit\n", stdout);
}

static void unpack(unsigned char *data, unsigned long long index) {
    for (int i = 0; i < 8; i++) { data[i] = index & 0xFF; index >>= 8; }
}

unsigned char *mmap_table(const char *fname, 
                          int need_write, int *ct_size) {
    int fd, prot;
    long long size;
    void *start;
    
    fd = open(fname, need_write ? O_RDWR : O_RDONLY);
    if (fd == -1) { perror(fname); return NULL; }
    
    size = lseek(fd, 0, SEEK_END);
    if (size == -1) { perror("lseek"); close(fd); return NULL; }
    if (size % rec_size != 0) {
        fprintf(stderr, 
            "WARNING: the file size is not a multiple\
             of the record size\n");
    } 
    if (size < rec_size) {
        fprintf(stderr, 
                "FATAL: the table is empty, nothing to scan\n");
        close(fd);
        return NULL;
    }
    
    prot = need_write ? PROT_READ | PROT_WRITE : PROT_READ;
    start = mmap(NULL, size, prot, MAP_SHARED, fd, 0);
    if (start == MAP_FAILED) { 
        perror("mmap"); 
        close(fd); 
        return NULL; 
    }
    
    close(fd);
    *ct_size = size / rec_size;
    return start;
} 
   

static int do_scan(const char *fname) { 
    unsigned char *start;
    int ct_size, cnt; 

    if (!fname || !*fname) { 
        fprintf(stderr, "Please tell me what to scan\n");
        return 1;
    }
    
    start = mmap_table(fname, 0, &ct_size);
    if (!start) return 1;

    cnt = 0;
    for (int idx = 0; idx < ct_size; idx++) {
        if (!all_zeroes(
                start + idx * rec_size, public_key_size
            )) cnt++;
    }

    printf("total: %d   filled: %d  empty: %d   rate: %d%%\n",
        ct_size, cnt, ct_size - cnt, cnt * 100 / ct_size);    
    return 0; 
}

static int do_poscheck(const char *fname) {
    unsigned char *start;
    int ct_size;
    
    if (!fname || !*fname) {
        fprintf(stderr, "Please tell me what to check\n");
        return 1; 
    }
    
    start = mmap_table(fname, 0, &ct_size);
    if (!start) return 1;
    
    for (int i = 0; i < ct_size; i++) {
        unsigned long long idx;
        const unsigned char *cell_ptr = start + i * rec_size;
        if (all_zeroes(cell_ptr, rec_size)) continue;
        idx = pack_index(cell_ptr);
        idx %= ct_size;
        if (idx != i) {
            fprintf(stderr, 
                "ERROR: recnum %d index value %llud\n", i, idx);
            return 1;
        }
    }
    return 0;
}

static int do_cannibal(const char *canny, const char *dinny) {
    unsigned char *canny_ptr, *dinny_ptr; 
    int canny_recs_num, dinny_recs_num, i;
    /* cannibal's and his dinner's names */
    if (!canny || !*canny) { 
        fprintf(stderr, "Please tell me what to fill\n");
        return 1;
    }
    if (!dinny || !*dinny) {
        fprintf(stderr, "Please tell me what to cannibalize\n");
        return 1;
    }

    canny_ptr = mmap_table(canny, 1, &canny_recs_num);
    if (!canny_ptr) return 1;
    dinny_ptr = mmap_table(dinny, 1, &dinny_recs_num);
    if (!dinny_ptr) return 1;

    if(canny_recs_num != dinny_recs_num) { 
        fprintf(stderr, "ERROR: records count don't match: %d != %d\n",
                canny_recs_num, dinny_recs_num);
        return 1;
    } 
    
    for (i = 0; i < canny_recs_num; i++) {
        unsigned long long idx;
        unsigned char *canny_start = canny_ptr + i * rec_size;
        unsigned char *dinny_start = dinny_ptr + i * rec_size;
        if (!all_zeroes(canny_start, rec_size)) continue; // already exists
        if (all_zeroes(dinny_start, rec_size)) continue; // nothing to eat
        idx = pack_index(dinny_start);
        idx %= canny_recs_num;
        if (idx != i) {
            fprintf(stderr, "ERROR: recnum %d index value %llu\n",
                    i, idx);
            return 1;
        }
        memcpy(canny_start, dinny_start, rec_size);
        memset(dinny_start, 0, rec_size);
    }    
    return 0;
}

static int do_fill(const char *fname) {
    unsigned char *ptr;
    int recs_num, i;
    
    if (!fname || !*fname) {
        fprintf(stderr, "Please tell me what to fill \n");
        return 1;
    }
    
    ptr = mmap_table(fname, 1, &recs_num);
    if (!ptr) return 1;

    for (i = 0; i < recs_num; i++) {
        unsigned long long idx;
        unsigned char *start = ptr + i * rec_size;
        unsigned char buf[public_key_size];
        unsigned char hash[hash_size];
        
        if (!all_zeroes(start, rec_size)) continue;
        if (!get_random(buf, sizeof(buf))) {
            fprintf(stderr, "ERROR: unable to get random\n");
            return 1;
        }
        /* Fix first bytes of random data to match record position. */
        idx = pack_index(buf);
        idx += i - idx % recs_num;
        unpack(buf, idx);

        if (!take_hash(hash, buf)) {
            fprintf(stderr, "ERROR: couldn't take hash\n");
            return 1;
        }

        memcpy(start + public_key_size, hash, hash_size);
        memcpy(start + 8, buf + 8, public_key_size - 8);
        memcpy(start, buf, 8);
    }
    return 0;  
}

int main(int argc, const char * const *argv) {
    if (argc < 2) {
        fputs("Try ``ct-man help'' for help\n", stdout);
        return 1;
    }
    
    if (strcmp("help", argv[1]) == 0) { help(); return 0; }
    if (strcmp("scan", argv[1]) == 0) { return do_scan(argv[2]); }
    if (strcmp("poscheck", argv[1]) == 0) { return do_poscheck(argv[2]); }
    if (strcmp("cannibal", argv[1]) == 0) {
        if (argc < 4) {
            fputs("The ``cannibal''command \
                   needs exactly two arguments\n",
                stderr);
            return 1;
        }
        return do_cannibal(argv[2], argv[3]);
    }
    if (strcmp("fill", argv[1]) == 0) { return do_fill(argv[2]); }
    
    fprintf(stderr,
         "command ``%s'' not recognized, try ``help''\n", argv[1]);
    return 1;
}
