#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <stdlib.h>

#ifndef NODE_GEN_SCHED
#define NODE_GEN_SCHED 1
#endif

#if NODE_GEN_SCHED == 1
#define __USE_GNU  // for SCHED_IDLE to be available
#include <sched.h> 
#endif  

#include "monocypher/monocypher.h"

#include "crypto-conf.h"
#include "put-utils.h"
#include "utils.h"
#include "yeswrap.h" 
#include "fdab.h"
#include "_version.h"

enum {
    dflt_min_pref_len = 12,   // to save
    dot_display_freq = 32,
    dflt_ct_recs_num = 1048576, // 1 << 20
    ct_rec_size = public_key_size + hash_size
};

static void save_the_found(int pref_len, unsigned char *secret_key,
    unsigned char *public_key, unsigned char *hash) {
    int res, idx, id_idx, i; 
    char path[64]; 
    FILE *f;
    
    sprintf(path, "%d", pref_len);
    res = mkdir(path, 0700); 
    if(res == -1 && errno != EEXIST) { perror(path); return; }
    idx = strlen(path); 
    path[idx] = '/';
    idx++;
    path[idx] = 0;
    id_idx = idx; 
    for(i = public_key_size - node_id_size; i < public_key_size; i++) {
        sprintf(path+idx, "%02X", public_key[i]); idx += 2;
    }
    f = fopen(path, "w"); 
    if(!f) { perror(path); return; }
    put_string_field(f, "id", path + id_idx);
    put_int_field(f, "pref_len", pref_len);
    put_hex_field(f, "secret", secret_key, secret_key_size);
    put_hex_field(f, "public", public_key, public_key_size);
    put_hex_field(f, "yeshash", hash, hash_size);
    fclose(f);
}

#if NODE_GEN_SCHED == 1
static void set_nice() {
    int res;
    struct sched_param sp; 
    sp.sched_priority = 0;
    res = sched_setscheduler(0, SCHED_IDLE, &sp); 
    if (res == -1) perror("warning: sched_setscheduler");
}
#endif

static int to_base36(int num) { 
    return num < 10 ? '0' + num : 'A' - 10 + num; 
}

struct challenge_table {
    int recs_num;
    unsigned char *mapping;
}; 

static void save_challenge(struct challenge_table *tbl,
                           unsigned char *challenge,
                           unsigned char *hash) {
    unsigned long long index;
    unsigned char *start;
    
    index = pack_index(challenge);
    index %= tbl-> recs_num;
    start = tbl->mapping + index * ct_rec_size;
    
    if (!all_zeroes(start, ct_rec_size)) return;
    /* it's already filled */ 
    memcpy(start + public_key_size, hash, hash_size);
    memcpy(start + 8, challenge + 8, public_key_size - 8);
    memcpy(start, challenge, 8);
        /* we copy the index part last to minimize the chance that
           the program gets interrupted during the copying so the
           table will contain an  invalid record; even if the 
           interruption happens, we'll be able to detect the incomplete
           record with the ''ct-man poscheck'' command
        */
}

static int setup_ct(struct challenge_table *tbl,
                                 const char *fname, int recs_num) {
    int fd;
    long long size;

    tbl->recs_num = -1;
    tbl->mapping = NULL;
    
    fd = open(fname, O_RDWR | O_CREAT, 0600); 
    if (fd == -1) { perror(fname); return 0; }
    
    size = lseek(fd, 0, SEEK_END);
    if (size == -1) { perror("lseek"); close(fd); return 0; }
    
    if (size < ct_rec_size) {
        tbl->recs_num = recs_num > 0 ? recs_num : dflt_ct_recs_num;
        size = (long long)tbl->recs_num * ct_rec_size;  
        int res = ftruncate(fd, size); 
        if (res == -1) { perror("ftruncate"); close(fd); return 0; }
        else { tbl->recs_num = size /  ct_rec_size; }
    }  

    tbl->mapping = mmap(NULL, size, PROT_READ | PROT_WRITE,
                        MAP_SHARED, fd, 0); 
    if (tbl->mapping == MAP_FAILED) { 
        perror("mmap"); close(fd); return 0; 
    }

    return 1;  
}

static int perform_generation(int min_pref_len, struct challenge_table *tbl) {
    int res, pref_len;
    unsigned char buf[seed_size];
    umask(077);
#if NODE_GEN_SCHED == 1
    set_nice();
#endif    
    get_random(buf, sizeof(buf));  
    for (long long c = 0;; c++) {
        unsigned char seed[seed_size];
        unsigned char secret_key[secret_key_size];
        unsigned char public_key[public_key_size];
        unsigned char hash[hash_size];
        memcpy(seed, buf, sizeof(buf));
        crypto_eddsa_key_pair(secret_key, public_key, seed);
        res = take_hash(hash, public_key);

        if (!res) {
            fprintf(stderr, "couldn't take yespower hash, aborting\n");
            return 1;
        }
        pref_len = fdab_pref_len(hash, sizeof(hash)); 

        if (pref_len >= min_pref_len) {
            save_the_found(pref_len, secret_key, public_key, hash);
            crypto_wipe(secret_key, sizeof(secret_key));
            crypto_wipe(buf, sizeof(buf));
            get_random(buf, sizeof(buf)); 
            putchar(to_base36(pref_len));
            fflush(stdout); 
        } else {
            if (tbl) save_challenge(tbl, public_key, hash);
            if (!(c % dot_display_freq)) { putchar('.'); fflush(stdout); }
        }
        increment_buf(buf, sizeof(buf));
    }
    return 0; /* never reached, actually */
}

static void help() {
    fputs(
"node-gen program: node master key generation\n"
"vers. " FDAB_VERSION "(compiled " __DATE__ ")\n"
"Copyright (c) Mikita Alchimionak, 2025\n"
"\n"
"Usage: node-gen [<options>]\n"
"Options are:\n"
"   -d <directory>      change to this dir before start\n"
"   -m <minimal_length> save keys of this rank and higher\n"
"                       (default is 12)\n"
"   -t <table_file>     save challenges to this challenge table\n"
"   -r <records_number> let the challenge table contain this many\n"
"                       records; each record is 64 bytes long, the\n"
"                       default (for new files) is 1048576 records.\n"
"                       for existing files this option is ignored\n"
"   -h                  display this help and exit\n",  stdout);
}

struct cmdline_opts {
    int min_pref_len;
    const char *dir;
    const char *ct_fname;
    int ct_recs_num;
};

static void init_cmdline_opts(struct cmdline_opts *p) {
    memset(p, 0, sizeof(*p));
    p->min_pref_len = dflt_min_pref_len;
}

static int need_arg(int opt) {
    return opt == 'd' || opt == 'm' || opt == 't' || opt == 'r';
}

static int parse_cmdline(int argc, const char *const *argv,
                         struct cmdline_opts *opts) {
    int idx = 1;
    while(idx < argc) {
        if (argv[idx][0] == '-') {
            char opt = argv[idx][1];
            if (need_arg(opt) && 
                (idx + 1 > argc || argv[idx + 1][0] == '-')) {
                fprintf(stderr, 
                        "-%c needs argument; try -h for help\n", opt);
                return 0;
            }
            switch (opt) {
            case 'd':
                opts->dir = argv[idx + 1];
                idx += 2;
                break;
            case 'm': {
                char *endptr;
                int n = strtol(argv[idx + 1], &endptr, 10);
                if (*argv[idx + 1] != '\0' && *endptr == '\0' &&
                    n > 1 && n < 25) { opts->min_pref_len = n; }
                else {
                    fprintf(stderr, "Invalid minimum rank value\n");
                    return 0;
                }
                idx += 2;
                break;
            }
            case 't':
                opts->ct_fname = argv[idx + 1];
                idx += 2;
                break;
            case 'r': {
                char *endptr;
                int n = strtol(argv[idx + 1], &endptr, 10);
                if (*argv[idx + 1] != '\0' && *endptr == '\0' &&
                    n > 1 && n < 1<<30) { opts->ct_recs_num = n; }
                else {
                    fprintf(stderr, 
                            "Invalid challenge table record number\n");
                    return 0;
                }                 
                idx += 2;
                break;
            }
            case 'h':
                help();
                exit(0);
                break;
            default:
                fprintf(stderr,
                         "unknown option ``-%c''\n", argv[idx][1]);
                return 0;
            }
        } else { /* no ``-''; we don't know how to handle this */
            fprintf(stderr, "stray parameter ``%s''\n", argv[idx]);
            return 0;
        }
    }
    return 1;
}

int main(int argc, const char * const *argv) {
    struct cmdline_opts opts;
    struct challenge_table ct;
    int res;

    init_cmdline_opts(&opts);
    res = parse_cmdline(argc, argv, &opts);
    if (!res) return 1;
    if (opts.dir) {
        res = chdir(opts.dir);
        if (res == -1) { perror(opts.dir); return 1; }
    }
    /* simpler case */
    if (!opts.ct_fname) {
        return perform_generation(opts.min_pref_len, NULL);
    }

    res = setup_ct(&ct, opts.ct_fname, opts.ct_recs_num);
    if (!res) return 1;

    return perform_generation(opts.min_pref_len, &ct);
}
