import PyPDF2
import re
import json
from typing import List, Dict

def extract_hold_positions(pdf_path: str) -> List[Dict]:
    """
    Extract hold positions and rotations from the Moonboard PDF.
    Format: hold type followed by holds in format "hold number - position - orientation"
    """
    holds = []
    current_hold_set = None
    processed_positions = set()
    all_text_2d = []  # 2D array to store all text
    
    with open(pdf_path, 'rb') as file:
        pdf_reader = PyPDF2.PdfReader(file)
        
        # Process each page, skipping the first page
        for page_num, page in enumerate(pdf_reader.pages[1:], start=2):
            text = page.extract_text()
            
            # Split into lines and process
            lines = text.split('\n')
            page_lines = []  # Store lines for this page
            
            for line in lines:
                line = line.strip()
                if not line:
                    continue
                
                # Add line to page_lines array
                page_lines.append(line)
                
                # Process hold line (format: name - position - orientation)
                parts = line.split(' - ')
                
                # Check if this is a hold type line
                if len(parts) == 1:
                    current_hold_set = line
                    continue
                
                if len(parts) == 5:
                    position1 = parts[1].strip()
                    rotation1 = ''.join(char for char in parts[2].strip() if char.isalpha())
                    
                    position2 = parts[3].strip()
                    rotation2 = parts[4].strip()
                    
                    hold_data1 = {
                        "hold_set": current_hold_set,
                        "rotation": rotation1,
                        "position": position1
                    }
                    holds.append(hold_data1)
                    
                    hold_data2 = {
                        "hold_set": current_hold_set,
                        "rotation": rotation2,
                        "position": position2
                    }
                    holds.append(hold_data2)
                    
                if len(parts) == 3:
                    position = parts[1].strip()
                    processed_positions.add(position)
                    rotation = parts[2].strip().upper()
                    
                    hold_data = {
                        "hold_set": current_hold_set,
                        "rotation": rotation,
                        "position": position
                    }
                    holds.append(hold_data)
    return holds

def save_holds(holds: List[Dict], output_file: str):
    """Save the extracted hold positions to a JSON file"""
    data = {
        "wall": {
            "width": 11,  # Moonboard width
            "height": 18,  # Moonboard height
            "source": "moonboard_pdf",
            "holds": holds
        }
    }
    
    with open(output_file, 'w') as f:
        json.dump(data, f, indent=4)

def main():
    pdf_path = "holds/moonboard_holdsetups_2016.pdf"
    output_file = "holds/extracted_holds.json"
    
    holds = extract_hold_positions(pdf_path)
    save_holds(holds, output_file)

main()