import tkinter as tk
import json
import os
from datetime import datetime
import math
from tkinter import ttk
from tkinter import messagebox
import joblib
import numpy as np

class Hold:
    # Define valid rotations
    ROTATIONS = ['N', 'NE', 'E', 'SE', 'S', 'SW', 'W', 'NW']
    
    # Define colors for different hold types
    HOLD_COLORS = {
        "Original School Holds": "goldenrod",
        "Hold Set A": "beige",
        "Hold Set B": "dimgray",
    }
    
    def __init__(self, hold_set="", rotation='N', position=""):
        self.hold_set = hold_set
        self.rotation = rotation
        self.position = position
    
    def get_position_notation(self):
        return self.position
    
    def get_rotation_angle(self):
        """Convert rotation direction to angle in degrees"""
        rotation_angles = {
            'N': 0,
            'NE': 45,
            'E': 90,
            'SE': 135,
            'S': 180,
            'SW': 225,
            'W': 270,
            'NW': 315
        }
        return rotation_angles[self.rotation]
    
    def get_color(self):
        """Get the color for this hold based on its hold set"""
        return self.HOLD_COLORS.get(self.hold_set, "gray")  # Default to gray if hold set not found


class ClimbingWall:
    def __init__(self, width, height, holds_data, id = -1):
        self.width = width
        self.height = height
        self.holds = []
        self.current_problem = None
        self.id = id
        self.current_problem_index = 0  # Track current problem for sequential loading
        self.selected_holds = set()  # For custom problem creation
        for hold_data in holds_data:
            hold = Hold(
                hold_data.get("hold_set", ""),
                hold_data.get("rotation", "N"),
                hold_data.get("position", "")
            )
            self.holds.append(hold)

    def display(self):
        root = tk.Tk()
        root.title("Climbing Wall Editor")
        
        # Calculate window size based on wall dimensions
        TILE_SIZE = 30
        LABEL_SIZE = 20
        canvas_width = self.width * TILE_SIZE + LABEL_SIZE + 50
        canvas_height = self.height * TILE_SIZE + LABEL_SIZE
        window_width = canvas_width + 300
        window_height = max(canvas_height + 100, 400)
        
        # Center window on screen
        screen_width = root.winfo_screenwidth()
        screen_height = root.winfo_screenheight()
        x = (screen_width - window_width) // 2
        y = (screen_height - window_height) // 2
        root.geometry(f"{window_width}x{window_height}+{x}+{y}")
        
        # Create a frame for the canvas
        canvas_frame = tk.Frame(root)
        canvas_frame.pack(side=tk.LEFT, padx=10, pady=10)
        
        canvas = tk.Canvas(canvas_frame, width=canvas_width, height=canvas_height)
        canvas.pack()

        # Create a frame for buttons
        button_frame = tk.Frame(root)
        button_frame.pack(side=tk.LEFT, padx=10, pady=10)

        # Add state for selection mode
        selecting_holds = tk.BooleanVar(value=False)

        def load_random_problem():
            try:
                # Load from moonboard_problems.json
                filepath = "moonboard_problems_setup_2016.json"
                
                if not os.path.exists(filepath):
                    messagebox.showinfo("No Problems", "moonboard_problems.json not found.")
                    return
                
                with open(filepath, 'r') as f:
                    data = json.load(f)
                    
                    # Get a random problem
                    if data:
                        import random
                        problem_keys = list(data.keys())
                        random_key = random.choice(problem_keys)
                        problem = data[random_key]
                        self.current_problem = problem
                        
                        # Clear canvas but keep existing holds in self.holds
                        canvas.delete("all")
                        
                        # Draw the grid and labels
                        draw_grid()
                        
                        # Draw existing holds first
                        draw_holds()
                        
                        # Add problem holds as overlays
                        draw_problem_holds(problem)
                        
                        # Update problem info label
                        problem_name = problem.get('Name', 'Unnamed')
                        problem_grade = problem.get('Grade', 'Unknown')
                        problem_info.config(text=f"Random Problem: {problem_name} - Grade: {problem_grade}")
                    else:
                        messagebox.showinfo("No Problems", "No problems found in moonboard_problems.json.")
            except Exception as e:
                messagebox.showerror("Error", f"Error loading problem: {str(e)}")

        def load_next_problem():
            try:
                # Load from moonboard_problems.json
                filepath = "moonboard_problems_setup_2016.json"
                
                if not os.path.exists(filepath):
                    messagebox.showinfo("No Problems", "moonboard_problems.json not found.")
                    return
                
                with open(filepath, 'r') as f:
                    data = json.load(f)
                    
                    # Get problems in order
                    if data:
                        problem_keys = list(data.keys())
                        
                        # Get the next problem (or first if we've reached the end)
                        if self.current_problem_index >= len(problem_keys):
                            self.current_problem_index = 0  # Reset to beginning
                        
                        problem_key = problem_keys[self.current_problem_index]
                        problem = data[problem_key]
                        self.current_problem = problem
                        
                        # Clear canvas but keep existing holds in self.holds
                        canvas.delete("all")
                        
                        # Draw the grid and labels
                        draw_grid()
                        
                        # Draw existing holds first
                        draw_holds()
                        
                        # Add problem holds as overlays
                        draw_problem_holds(problem)
                        
                        # Update problem info label
                        problem_name = problem.get('Name', 'Unnamed')
                        problem_grade = problem.get('Grade', 'Unknown')
                        problem_info.config(text=f"Problem {self.current_problem_index + 1}/{len(problem_keys)}: {problem_name} - Grade: {problem_grade}")
                        
                        # Move to next problem
                        self.current_problem_index += 1
                    else:
                        messagebox.showinfo("No Problems", "No problems found in moonboard_problems.json.")
            except Exception as e:
                messagebox.showerror("Error", f"Error loading problem: {str(e)}")

        def draw_problem_holds(problem):
            """Draw problem holds as larger empty red circles on top of existing holds"""
            for move in problem.get('Moves', []):
                position = move.get('Description', '')  # e.g., "K18", "A5"
                if position:
                    # Convert position notation to coordinates
                    letter = position[0]  # e.g., "A"
                    number = int(position[1:])  # e.g., 1
                    
                    # Convert to x,y coordinates
                    x = ord(letter) - ord('A')  # A=0, B=1, etc.
                    y = 18 - number  # Convert from 1-18 to 0-17
                    
                    # Draw larger empty red circle for problem hold
                    problem_hold_size = TILE_SIZE * 0.7  # Bigger than regular holds
                    margin = (TILE_SIZE - problem_hold_size) / 2
                    
                    canvas.create_oval(
                        x*TILE_SIZE + margin + LABEL_SIZE,
                        y*TILE_SIZE + margin + LABEL_SIZE,
                        x*TILE_SIZE + margin + problem_hold_size + LABEL_SIZE,
                        y*TILE_SIZE + margin + problem_hold_size + LABEL_SIZE,
                        fill="", outline="red", width=3
                    )

        def draw_grid():
            # Draw column labels (A-K) at the very top
            for col in range(self.width):
                x = col * TILE_SIZE + LABEL_SIZE + TILE_SIZE/2
                y = LABEL_SIZE/2  # Move letters to the very top
                canvas.create_text(x, y, text=chr(65 + col), font=('Arial', 10, 'bold'))
            # Draw row labels (1-18) from top to bottom
            for row in range(self.height):
                x = LABEL_SIZE/2
                y = row * TILE_SIZE + LABEL_SIZE + TILE_SIZE/2
                row_number = self.height - row
                canvas.create_text(x, y, text=str(row_number), font=('Arial', 10, 'bold'))
            # Draw the grid
            for row in range(self.height):
                for col in range(self.width):
                    x1 = col * TILE_SIZE + LABEL_SIZE
                    y1 = row * TILE_SIZE + LABEL_SIZE
                    x2 = x1 + TILE_SIZE
                    y2 = y1 + TILE_SIZE
                    canvas.create_rectangle(x1, y1, x2, y2, fill="gray")

        def draw_holds():
            """Draw all existing holds on the canvas"""
            for hold in self.holds:
                position = hold.position
                letter = position[0]
                number = int(position[1:])
                x = ord(letter) - ord('A')
                y = 18 - number
                hold_size = TILE_SIZE * 0.5
                margin = (TILE_SIZE - hold_size) / 2
                canvas.create_oval(
                    x*TILE_SIZE + margin + LABEL_SIZE,
                    y*TILE_SIZE + margin + LABEL_SIZE,
                    x*TILE_SIZE + margin + hold_size + LABEL_SIZE,
                    y*TILE_SIZE + margin + hold_size + LABEL_SIZE,
                    fill=hold.get_color(), outline="black", width=1
                )
                center_x = x*TILE_SIZE + margin + hold_size/2 + LABEL_SIZE
                center_y = y*TILE_SIZE + margin + hold_size/2 + LABEL_SIZE
                angle = hold.get_rotation_angle()
                line_length = hold_size/2
                end_x = center_x + line_length * math.cos(math.radians(angle))
                end_y = center_y - line_length * math.sin(math.radians(angle))
                canvas.create_line(center_x, center_y, end_x, end_y, fill='black', width=2)

        def draw_selected_holds():
            # Draw selected holds for custom problem as blue overlays
            for pos in self.selected_holds:
                letter = pos[0]
                number = int(pos[1:])
                x = ord(letter) - ord('A')
                y = 18 - number
                hold_size = TILE_SIZE * 0.7
                margin = (TILE_SIZE - hold_size) / 2
                canvas.create_oval(
                    x*TILE_SIZE + margin + LABEL_SIZE,
                    y*TILE_SIZE + margin + LABEL_SIZE,
                    x*TILE_SIZE + margin + hold_size + LABEL_SIZE,
                    y*TILE_SIZE + margin + hold_size + LABEL_SIZE,
                    fill="", outline="blue", width=3
                )

        def select_custom_hold(event):
            # Convert click to wall coordinates
            x = (event.x - LABEL_SIZE) // TILE_SIZE
            y = (event.y - LABEL_SIZE) // TILE_SIZE
            if 0 <= x < self.width and 0 <= y < self.height:
                letter = chr(ord('A') + x)
                number = 18 - y
                pos = f"{letter}{number}"
                # Only allow selecting if there is a hold at this position
                if any(hold.position == pos for hold in self.holds):
                    if pos in self.selected_holds:
                        self.selected_holds.remove(pos)
                    else:
                        self.selected_holds.add(pos)
                    canvas.delete("all")
                    draw_grid()
                    draw_holds()
                    draw_selected_holds()

        def predict_grade():
            if not self.selected_holds:
                messagebox.showinfo("No Holds Selected", "Please select holds for your custom problem.")
                return
            # Prepare problem dict for extract_features
            moves = [{"Description": pos} for pos in self.selected_holds]
            problem = {"Moves": moves}
            # Load model and encoder
            try:
                results = joblib.load("ml_results.joblib")
                model = results['model']
                le = results['le']
                # Use extract_features from MLProblemEvaluation
                from MLProblemEvaluation import extract_features, calculate_hold_difficulty_scores
                # Use the same hold difficulty scores as in training
                with open("moonboard_problems_setup_2016.json", 'r') as f:
                    problems = json.load(f)
                hold_difficulty_scores = calculate_hold_difficulty_scores(problems)
                features = extract_features(problem, hold_difficulty_scores)
                import pandas as pd
                X_custom = pd.DataFrame([features])
                y_pred = model.predict(X_custom)
                grade = le.inverse_transform(y_pred)[0]
                messagebox.showinfo("Predicted Grade", f"Predicted grade for your custom problem: {grade}")
            except Exception as e:
                messagebox.showerror("Prediction Error", f"Could not predict grade: {str(e)}")

        def start_selecting():
            # Clear any shown problem and overlays
            self.current_problem = None
            self.selected_holds.clear()
            canvas.delete("all")
            draw_grid()
            draw_holds()
            draw_selected_holds()
            problem_info.config(text="Select holds for your custom problem.")
            canvas.bind("<Button-1>", select_custom_hold)
            select_button.config(text="Stop Selecting Holds", bg="red")
            selecting_holds.set(True)

        def stop_selecting():
            canvas.unbind("<Button-1>")
            select_button.config(text="Start Selecting Holds", bg="green")
            selecting_holds.set(False)

        def toggle_selecting():
            if not selecting_holds.get():
                start_selecting()
            else:
                stop_selecting()

        # Problem info label
        problem_info = tk.Label(button_frame, text="No problem loaded", wraplength=260, justify=tk.LEFT, anchor=tk.W, padx=10)
        problem_info.pack(pady=5, fill=tk.X, expand=True)

        # Random problem button
        random_problem_button = tk.Button(
            button_frame,
            text="Load Random Problem",
            command=load_random_problem,
            width=20
        )
        random_problem_button.pack(pady=5)

        # Next problem button
        next_problem_button = tk.Button(
            button_frame,
            text="Load Next Problem",
            command=load_next_problem,
            width=20
        )
        next_problem_button.pack(pady=5)

        # Insert the toggle button above the custom problem label
        select_button = tk.Button(
            button_frame,
            text="Start Selecting Holds",
            command=toggle_selecting,
            width=20,
            bg="green",
            fg="black"
        )
        select_button.pack(pady=5)

        # Custom problem and predict grade buttons
        custom_button = tk.Label(button_frame, text="Click holds to select/deselect for custom problem.", wraplength=260, fg="blue")
        custom_button.pack(pady=5)
        predict_button = tk.Button(
            button_frame,
            text="Predict Grade",
            command=predict_grade,
            width=20,
            bg="gold",
            fg="black"
        )
        predict_button.pack(pady=5)

        # Draw the wall grid with labels
        draw_grid()
        draw_holds()
        draw_selected_holds()
        root.mainloop()