import tkinter as tk
from tkinter import messagebox
from climbingWall import ClimbingWall
import json
import os
import subprocess
import sys

def create_main_window():
    root = tk.Tk()
    root.title("Climbing Wall Editor")
    
    # Calculate window size
    window_width = 400
    window_height = 420  # Reduced height for a shorter main window
    
    # Center window on screen
    screen_width = root.winfo_screenwidth()
    screen_height = root.winfo_screenheight()
    x = (screen_width - window_width) // 2
    y = (screen_height - window_height) // 2
    root.geometry(f"{window_width}x{window_height}+{x}+{y}")
    
    # Create a frame for buttons
    button_frame = tk.Frame(root)
    button_frame.pack(expand=True, fill='both', padx=20, pady=20)
    
    button_font = ("Arial", 12, "bold")
    button_width = 22
    
    # Load from holds directory button
    def load_from_holds():
        holds_dir = "holds"
        moonboard_file = os.path.join(holds_dir, "extracted_holds.json")
        
        if not os.path.exists(holds_dir):
            tk.messagebox.showerror("Error", "Holds directory not found.")
            return
            
        if not os.path.exists(moonboard_file):
            tk.messagebox.showerror("Error", f"Moonboard file '{moonboard_file}' not found.")
            return
            
        try:
            with open(moonboard_file, 'r') as f:
                data = json.load(f)
                if 'wall' in data:
                    wall_data = data['wall']
                    root.destroy()
                    wall = ClimbingWall(
                        wall_data["width"],
                        wall_data["height"],
                        wall_data["holds"]
                    )
                    wall.display()
                else:
                    tk.messagebox.showerror("Error", "Invalid wall data format in moonboard file.")
        except json.JSONDecodeError:
            tk.messagebox.showerror("Error", "Error reading moonboard JSON file.")
        except Exception as e:
            tk.messagebox.showerror("Error", f"Error loading moonboard wall: {str(e)}")
    
    load_holds_button = tk.Button(button_frame, text="Load Moonboard Wall", command=load_from_holds, width=button_width, font=button_font)
    load_holds_button.pack(pady=10)
    
    # Run ML Problem Evaluation button
    def run_ml_evaluation():
        try:
            # Check if the required data file exists
            if not os.path.exists("moonboard_problems_setup_2016.json"):
                tk.messagebox.showerror("Error", "Required data file 'moonboard_problems_setup_2016.json' not found.")
                return
            
            # First, try to import the required modules to check if they're available
            try:
                import sklearn
                import pandas
                import numpy
                import matplotlib
                import seaborn
            except ImportError as e:
                missing_module = str(e).split("'")[1] if "'" in str(e) else "unknown"
                error_msg = f"Missing required module: {missing_module}\n\n"
                error_msg += "Please install the required dependencies first.\n\n"
                
                # Create a dialog with options
                dialog = tk.Toplevel(root)
                dialog.title("Missing Dependencies")
                dialog.geometry("500x300")
                dialog.transient(root)
                dialog.grab_set()
                
                # Center the dialog
                dialog.update_idletasks()
                x = (dialog.winfo_screenwidth() // 2) - (500 // 2)
                y = (dialog.winfo_screenheight() // 2) - (300 // 2)
                dialog.geometry(f"500x300+{x}+{y}")
                
                # Add message
                msg_label = tk.Label(dialog, text=error_msg, font=('Arial', 10), justify='left')
                msg_label.pack(padx=20, pady=20)
                
                # Add close button
                close_button = tk.Button(dialog, text="Close", command=dialog.destroy)
                close_button.pack(pady=5)
                
                return
            
            # Show a message that the analysis is starting
            tk.messagebox.showinfo("ML Analysis", "Starting ML Problem Evaluation...\nThis may take a few moments.")
            
            # Run the ML evaluation in a separate process and capture output
            result = subprocess.run([sys.executable, "MLProblemEvaluation.py"], 
                                  capture_output=True, text=True, timeout=300)  # 5 minute timeout
            
            if result.returncode != 0:
                error_msg = result.stderr if result.stderr else "Unknown error occurred"
                # Check for common error patterns
                if "ModuleNotFoundError" in error_msg or "ImportError" in error_msg:
                    error_msg += "\n\nThis might be due to missing dependencies.\n"
                    error_msg += "Please run: pip install -r requirements.txt"
                tk.messagebox.showerror("Error", f"ML Problem Evaluation failed.\n\n{error_msg}")
                
        except subprocess.TimeoutExpired:
            tk.messagebox.showerror("Timeout", "ML Problem Evaluation took too long and was terminated.")
        except FileNotFoundError:
            tk.messagebox.showerror("Error", "MLProblemEvaluation.py file not found.")
        except Exception as e:
            tk.messagebox.showerror("Error", f"Error running ML evaluation: {str(e)}")
    
    ml_evaluation_button = tk.Button(button_frame, text="Run ML Problem Evaluation", command=run_ml_evaluation, width=button_width, font=button_font)
    ml_evaluation_button.pack(pady=10)
    
    # Show ML Results button
    def show_ml_results():
        import subprocess
        import sys
        try:
            result = subprocess.run([sys.executable, "show_ml_results.py"], capture_output=True, text=True, timeout=300)
            if result.returncode != 0:
                error_msg = result.stderr if result.stderr else "Unknown error occurred"
                tk.messagebox.showerror("Error", f"Failed to show ML results.\n\n{error_msg}")
        except Exception as e:
            tk.messagebox.showerror("Error", f"Error running show_ml_results.py: {str(e)}")
    
    show_results_button = tk.Button(button_frame, text="Show Last ML Results", command=show_ml_results, width=button_width, font=button_font)
    show_results_button.pack(pady=10)
    
    # Add a separator line
    separator = tk.Frame(button_frame, height=2, bg='gray')
    separator.pack(fill='x', pady=10)
    
    # Add info label
    info_label = tk.Label(button_frame, text="ML Analysis analyzes climbing problem difficulty\nusing machine learning techniques.", 
                         font=('Arial', 9), fg='gray', justify='center')
    info_label.pack(pady=5)
    
    root.mainloop()

if __name__ == "__main__":
    create_main_window()
