from kafka import KafkaConsumer, KafkaProducer
from constants import KAFKA_BROKER, SERVER_LOGS_TOPIC, VIEW_ITEM_COUNT_TOPIC
import json
from datetime import datetime
from collections import defaultdict

consumer = KafkaConsumer(
    SERVER_LOGS_TOPIC,
    bootstrap_servers=KAFKA_BROKER,
    auto_offset_reset='earliest',  
    enable_auto_commit=True,
    value_deserializer=lambda x: json.loads(x.decode('utf-8'))  # deserialize JSON message
)


producer = KafkaProducer(
    bootstrap_servers=KAFKA_BROKER,
    value_serializer=lambda x: json.dumps(x).encode('utf-8')  # serialize message to JSON format
)

# dictionary to track view_item events per day
view_item_count = defaultdict(int)

def get_day(timestamp):
   
    # convert timestamp to a datetime object
    timestamp = datetime.strptime(timestamp, "%Y-%m-%dT%H:%M:%S")
    
    # return the date part as the key (YYYY-MM-DD)
    return timestamp.date().strftime("%Y-%m-%d")

try:
    for message in consumer:
        log = message.value
        
        # check if the log is a 'view_item' event and contains a valid timestamp
        if log.get('event') == 'view_item' and 'timestamp' in log:
            timestamp = log['timestamp']
            
            # get the date from the timestamp
            day = get_day(timestamp)
            
            # increment the view_item count for this day
            view_item_count[day] += 1

            print(f"View Item Events for {day}: {view_item_count[day]}")
            
            # send the aggregated count to VIEW_ITEM_COUNT_TOPIC
            if view_item_count[day] > 0:
                producer.send(VIEW_ITEM_COUNT_TOPIC, value={'date': day, 'view_item_count': view_item_count[day]})
                producer.flush()

except KeyboardInterrupt:
    print("\nConsumer stopped by user.")
finally:
    consumer.close()  
    producer.close()
    print("Consumer and Producer connections closed.")